# Setup code manager on the given primary node.
#
# If the $control_repo_url is a file://, and $generate_empty_control_repo is
# not set to false, then the plan will install git and create an repo at that
# path on the primary with a single production branch commit with an empty
# manifests/site.pp. This is just a convenience for initial test setup for HA.
#
# @param primary The primary node of the PE installation.
# @param control_repo_url The url to the control repository (remote or local)
# @param control_repo_private_key_path If the repo requires a key to reach,
#   this is the absolute path to the location of the key on the primary.
# @param local_path_to_private_key If you want a local private key copied into
#   the $control_repo_private_key_path, set it here.
# @param generate_empty_control_repo Set this to false to skip
#   generating an empty control repository on the primary when
#   $control_repo_url is a file://
# @param rbac_account The account to deploy as. A temporary account
#   will be generated if not given.
# @param rbac_password The password for the $rbac_account.
plan enterprise_tasks::testing::setup_code_manager(
  Variant[Target,String] $primary,
  String $control_repo_url = constants()['default_control_repo_url'],
  Boolean $generate_empty_control_repo = true,
  Optional[Enterprise_tasks::Absolute_path] $control_repo_private_key_path = undef,
  Optional[Enterprise_tasks::Absolute_path] $local_path_to_private_key = undef,
  Optional[String] $rbac_account = undef,
  Optional[Sensitive[String]] $rbac_password = undef,
) {
  $primary_target = get_targets($primary)[0]

  if $local_path_to_private_key =~ NotUndef {
    $_private_key_dir = regsubst($control_repo_private_key_path, '/[^/]+$', '')
    $_ssh_key = file::read($local_path_to_private_key)
    apply($primary_target) {
      file { $_private_key_dir:
        ensure => directory
      }
      -> file { $control_repo_private_key_path:
        ensure  => present,
        content => Sensitive($_ssh_key),
        mode    => '0600',
        owner   => 'pe-puppet',
        group   => 'pe-puppet',
      }
    }
  }

  if $control_repo_url =~ /^file:\/\/(.*)/ and $generate_empty_control_repo {
    run_plan('enterprise_tasks::testing::generate_empty_code_repository',
      'primary'           => $primary_target,
      'control_repo_path' => $1,
    )
  }

  $update_group_result = run_task('enterprise_tasks::update_node_group', $primary_target,
    'group_name'       => 'PE Master',
    'class_parameters' => {
      'puppet_enterprise::profile::master' => {
        'code_manager_auto_configure' => true,
        'r10k_remote'                 => $control_repo_url,
        'r10k_private_key'            => $control_repo_private_key_path,
      }
    },
  ).first()
  $master_group = $update_group_result.value()['group']

  enterprise_tasks::message('setup_code_manager', "${master_group['name']} puppet_enterprise::profile::master: ${master_group['classes']['puppet_enterprise::profile::master']}")
  run_task('enterprise_tasks::run_puppet', $primary_target)
  enterprise_tasks::message('setup_code_manager', 'Running puppet a second time to modify configuration based on puppetdb_query')
  run_task('enterprise_tasks::run_puppet', $primary_target)

  enterprise_tasks::with_puppet_access($primary_target,
    'account'  => $rbac_account,
    'password' => $rbac_password,
  ) |$token_file, $_token| {
    run_command("${constants()['pe_bin']}/puppet-code deploy production --token-file=${token_file}", $primary_target)
  }
}
