# Orchestrate upgrade of PE's primary infrastructure.
#
# Upgrades any layout (monolithic, legacy split, primary/database, legacy
# split/database) based on the configuration read from pe.conf. It is
# intended for internal use in CI and manual testing, since PE tarballs are
# pulled from artifactory.delivery.puppetlabs.net.
#
# Puppet will have had to return a 0 exit code (no changes) from every target
# node before the plan will succeed.
#
# You must specify either $version or $tarball, but not both.
#
# @param $primary
#   The primary PE node. All other primary infrastructure nodes
#   are looked up from enterprise data configuration on the primary.
# @param $version
#   A version string indicating either the family line (2019.1, 2019.2, etc.)
#   or an exact version of PE to download and upgrade to. If just the family is
#   given, the latest development build from that line will be installed.
# @param $tarball
#   Alternately you may supply an absolute path to a PE tarball on the
#   localhost which you wish installed.
# @param $pe_conf
#   Absolute path to the pe.conf to read on the primary for configuration information.
# @param $use_tempdirs
#   If true, generate proper tempdirs as workdirs
#   for uploading PE and pe.conf files. Otherwise use /root. Defaults to true.
#   Set to false for manual testing where you want the simplicity of using /root
#   as the workdir.
# @param $upgrading_from
#   This version can be set if the version we are upgrading from impacts
#   upgrade behavior.
plan enterprise_tasks::testing::upgrade_pe(
  TargetSpec $primary,
  Boolean $use_tempdirs = true,
  Optional[Enterprise_tasks::Pe_version_or_family] $version = undef,
  Optional[Enterprise_tasks::Absolute_path] $tarball = undef,
  Optional[Enterprise_tasks::Absolute_path] $pe_conf = '/etc/puppetlabs/enterprise/conf.d/pe.conf',
  Optional[Enterprise_tasks::Pe_version_or_family] $upgrading_from = undef,
) {

  if ($version == undef and $tarball == undef) or ($version != undef and $tarball != undef) {
    fail_plan("You must specify either version or tarball, but not both. Received version: '${version}' and tarball: '${tarball}'")
  }

  $host_keys = ['puppet_enterprise::puppet_master_host','puppet_enterprise::console_host','puppet_enterprise::puppetdb_host','puppet_enterprise::database_host']
  $host_hash = run_task(enterprise_tasks::get_conf_values, $primary, path=>$pe_conf, keys=>$host_keys).first.value
  $primary_hostname = ($primary =~ Target) ? {
    true    => $primary.name,
    default => $primary
  }
  if $host_hash['puppet_enterprise::puppet_master_host'] != $primary_hostname {
    fail_plan("The primary parameter (${primary}) given to this plan does not match the puppet_enterprise::puppet_master_host (${host_hash['puppet_enterprise::puppet_master_host']}) parameter from ${pe_conf} on ${primary}.")
  }

  $primary_target = get_targets($primary)[0]
  debug("Primary target is: ${primary_target}")
  $database_target = get_targets(enterprise_tasks::first_defined(
    $host_hash['puppet_enterprise::database_host'],
    $host_hash['puppet_enterprise::puppetdb_host'],
    $primary
  ))[0]
  debug("Database target is: ${database_target}")
  $puppetdb_target = get_targets(enterprise_tasks::first_defined($host_hash['puppet_enterprise::puppetdb_host'], $primary))[0]
  debug("Puppetdb target is: ${puppetdb_target}")
  $console_target  = get_targets(enterprise_tasks::first_defined($host_hash['puppet_enterprise::console_host'], $primary))[0]
  debug("Console target is: ${console_target}")

  $db_split = ($primary_target != $database_target and $puppetdb_target != $database_target)
  $dbnode = $db_split ? {
    true  => $database_target,
    false => $puppetdb_target,
  }
  $targets = [$dbnode, $primary_target, $puppetdb_target, $console_target].unique - undef
  debug("Targets are: ${targets}")

  enterprise_tasks::message('upgrade_pe', 'Checking connectivity to infrastructure nodes.')
  enterprise_tasks::test_connection($targets)

  enterprise_tasks::message('upgrade_pe', 'Preparing work directories')
  if $use_tempdirs {
    run_plan('enterprise_tasks::create_tempdirs', nodes => $targets, 'purpose' => 'upgrade')
  } else {
    enterprise_tasks::set_workdirs($targets)
  }

  enterprise_tasks::message('upgrade_pe', "Distributing PE tarball based on ${enterprise_tasks::first_defined($tarball, $version)}.")
  run_plan('enterprise_tasks::testing::get_pe',
    'nodes'      => $targets,
    'version'    => $version,
    'tarball'    => $tarball,
  )

  enterprise_tasks::message('upgrade_pe', 'Syncing enterprise data.')
  run_plan(enterprise_tasks::sync_enterprise_data,
    'primary'        => $primary_target,
    'infrastructure' => $targets,
  )

  enterprise_tasks::message('upgrade_pe', 'Upgrading Infrastructure nodes.')
  $targets.each |$node| {
    run_plan(enterprise_tasks::testing::run_installer,
      'nodes'           => $node,
      'non_interactive' => true,
      'skip_pe_conf'    => true,
    )
  }

  enterprise_tasks::message('upgrade_pe', 'Running puppet on all nodes to modify configuration based on puppetdb_query and reset package repo configuration.')
  run_task(enterprise_tasks::run_puppet, $targets)

  if (($upgrading_from !~ Undef) and (versioncmp($upgrading_from, '2019.1') == -1)) {
    enterprise_tasks::message('upgrade_pe', 'Second puppet run on primary to introduce new service configs (bolt/ace) to services.conf when upgrading from <= 2019.1.')
    run_task(enterprise_tasks::run_puppet, $primary_target)
  }

  enterprise_tasks::message('upgrade_pe', 'Validating that all infrastructure nodes are steady state.')
  run_task(enterprise_tasks::run_puppet, $targets, 'exit_codes' => [0])
}
