#!/opt/puppetlabs/puppet/bin/ruby

require_relative '../files/enterprise_task_helper.rb'
require 'json'
require 'open3'
require 'etc'

class ProvisionReplica < EnterpriseTaskHelper

  def pe_version_greater_than?(threshold)
    version_file = '/opt/puppetlabs/server/pe_version'
    pe_version   = File.read(version_file).strip.to_s

    if pe_version.empty?
      raise EnterpriseTaskHelper::Error.new('No existing PE version found on host', 'puppetlabs.provision-replica/no-pe-version-file')
    end
    Gem::Version.new(pe_version) > Gem::Version.new(threshold)
  end

  def task(host:, replication_timeout:, streaming: true, token_file: nil, **_kwargs)
    puppet_infra = '/opt/puppetlabs/bin/puppet-infrastructure'

    command = [
      puppet_infra,
      'provision',
      'replica',
      host,
    ]
    command += ['--token-file', token_file] if !token_file.nil?
    if pe_version_greater_than?('2019.4')
      streaming_flag = streaming ? '--streaming' : '--no-streaming'
      command += [streaming_flag]
    end
    output, status = Open3.capture2e(*command)
    raise EnterpriseTaskHelper::Error.new("Failed to provision replica with certname #{host}", 'puppetlabs.provision-replica/provision-failed', 'output' => output) if !status.success?

    start_time = Time.now
    services_ok = false

    while (Time.now - start_time) < replication_timeout && !services_ok
      command = [
        puppet_infra,
        'status',
        '--host',
        host,
      ]
      output, _stderr, status = Open3.capture3(*command)
      break if !status.success?

      service_status = output.scan(%r{[0-9]+ of [0-9]+ services are fully operational})[0]
      services = service_status.nil? ?
        [] :
        service_status.scan(%r{[0-9]+})
      services_up = services[0]
      services_total = services[1]

      if services_up == services_total && services_up != 0
        services_ok = true
      else
        sleep(15)
      end
    end

    if !services_ok
      raise EnterpriseTaskHelper::Error.new(
        "Replica status check timed out or failed on host with certname #{host}",
        'puppetlabs.provision-replica/provision-timedout',
        {
          puppet_infra_status_output: output.split("\n"),
          exitcode: status.exitstatus,
          services_up: services_up,
          services_total: services_total,
        },
      )
    end

    result = {
      success: true,
      _output: output,
    }
    result.to_json
  end
end

# The HOME variable is required to run the provision command but may not exist
# when this task is executed using the orchestrator
ENV['HOME'] ||= Etc.getpwuid.dir

ProvisionReplica.run if __FILE__ == $PROGRAM_NAME
