require 'puppet'
require 'puppet/application'
require 'puppet/network/http_pool'
require 'puppet/util/puppetdb'
require 'pe_license'
require 'pe_license/status'

class Puppet::Application::License < Puppet::Application
  run_mode :master

  attr_accessor :license, :active_nodes
  attr_reader :log_level, :error

  def summary
    _("Display licensing information")
  end

  def help
      <<-HELP
puppet-license(8) -- #{summary}
SYNOPSIS
--------
Prints active node count, license count, and license status.
USAGE
-----
puppet license
COPYRIGHT
---------
Copyright (c) 2011 Puppet Inc., LLC Licensed under the Apache 2.0 License
      HELP
  end

  def setup
    Puppet::Util::Log.newdestination :console
    @buffer = []
    @log_level = :notice

    @url   = "https://buy.puppet.com/"
    @email = "sales@puppet.com"
    @phone = "+1 (877) 575-9775"
  end

  def license_key
    '/etc/puppetlabs/license.key'
  end

  def puppetdb_config
    @puppetdb_config ||= Puppet::Util::Puppetdb::Config.load
  end

  def load_license
    if File.exist?(license_key)
      begin
        @license = PELicense.from_file(license_key)
      rescue PELicense::InvalidLicenseError => detail
        complain_invalid_license detail
        exit 2
      end
    else
      # If the license doesn't exist, assume it's a complimentary trial install
      @license = PELicense.complimentary_license
    end
  end

  def retrieve_status
    uri=URI(puppetdb_config.server_urls.first)
    ssl_context = Puppet.lookup(:ssl_context)
    http = Puppet::Network::HttpPool.connection(uri.host, uri.port, ssl_context: ssl_context)
    result = PELicense::Status.retrieve(http)
    @active_nodes = result.nodes
  end

  def collect_status_messages
    messages = []

    messages << :node_count

    if license.start_date or license.end_date
      messages << :start_date if license.start_date
      messages << :end_date if license.end_date
      messages << :blank
    end

    messages << :licensee if license.to

    messages << :complimentary if license.complimentary?

    if active_nodes > license.nodes
      set_error
      messages << :exceeded_node_count
    end

    if license.expired?
      set_error
      messages << :expired
    elsif license.days_remaining <= 30
      set_warning
      messages << :expiring_soon
    end

    messages << :contact_details

    messages
  end

  def main
    load_license

    retrieve_status

    messages = collect_status_messages

    messages.each do |msg|
      send "append_#{msg}"
    end

    # (#12660) To avoid sending a massive string to the logging system, we send
    # the message line by line.  Sending a large string is a problem because it
    # gets truncated by syslog.
    @buffer.each do |line|
      Puppet.send @log_level, line
    end

    exit 0
  end

  def set_error
    @log_level = :alert
    @error = true
  end

  def set_warning
    @log_level = :warning unless @log_level == :alert
  end

  private

  def pluralize(noun, count)
    count == 1 ? noun : "#{noun}s"
  end

  def append(msg)
    @buffer << msg
  end

  def complain_invalid_license(reason)
    Puppet.crit <<-INVALID_LICENSE.gsub('  '*3, '')
      Your License is incorrectly formatted or corrupted!
      #{reason}

      Please contact Puppet Labs to correct this problem: email #{@email}
      or visit our website at:
      #{@url}
      INVALID_LICENSE
  end

  def append_node_count
    append "You have #{active_nodes == 0 ? 'no' : active_nodes} active #{pluralize 'node', active_nodes}."
    append "You are currently licensed for #{license.nodes} active #{pluralize 'node', license.nodes}."
  end

  def append_start_date
    append "Your support and maintenance agreement starts on #{license.start_date}"
  end

  def append_end_date
    append "Your support and maintenance agreement ends on #{license.end_date}"
  end

  def append_blank
    append ""
  end

  def append_licensee
    append "This Puppet Enterprise distribution is licensed to:"
    append license.to
    append ""
  end

  def append_complimentary
    append "You are using a complimentary ten node license provided free by Puppet Labs."
    append ""
    append "Your complimentary license does not include Support & Maintenance. If you"
    append "would like to obtain official Support & Maintenance, please contact us"
    append "for pricing, and to find out about volume discounts."
    append ""
  end

  def append_exceeded_node_count
    error = true

    over = active_nodes - license.nodes

    append "You have exceeded the node count in your license by #{over} active #{pluralize 'node', over}!"
    append ""
    append "You should make sure that all inactive nodes are deactivated in PuppetDB."
    append "You can use the command `puppet node status <node>` to check the status of a"
    append "node, and `puppet node deactivate <node>` to deactivate unused nodes."
    append ""
    append "Please contact Puppet Labs to obtain additional licenses to bring your network"
    append "back in compliance."
    append ""
  end

  def append_expired
    append "Your Support & Maintenance agreement expired on #{license.end_date}!"
    append "You have run for #{license.days_exceeded} #{pluralize 'day', license.days_exceeded} without a support agreement; please contact"
    append "Puppet Labs urgently to renew your Support & Maintenance agreement."
    append ""
  end

  def append_expiring_soon
    append "Your Support & Maintenance term expires on #{license.end_date}."
    append "You have #{license.days_remaining} #{pluralize 'day', license.days_remaining} remaining under that agreement; please contact"
    append "Puppet Labs to renew your Support & Maintenance agreement:"
    append ""
  end

  def append_contact_details
    append "You can reach Puppet Labs for sales, support, or maintenance agreements"
    append "by email to #{@email}, on #{@phone}, or visit us on"
    append "the web at #{@url} for more information."
  end
end
