require 'pp'
require 'json'
require 'fileutils'
require 'tmpdir'
require 'pe_backup_tools/utils'
require 'pe_backup_tools/utils/analytics_client'
require 'open3'
require 'digest'

module PeBackupTools
  # This module provides functionality for the create command
  # create: Create a backup
  module Backup
    def self.gpg_check(gpgkey)
      gpg_cmd = "echo 'test' | gpg --trust-model always --recipient #{gpgkey} --encrypt > /dev/null"
      success = system(gpg_cmd)
      raise _('Error: gpg key check failed. Make sure gpg is installed and your gpg key is imported on this node to use the encryption option') unless success
    end

    def self.encrypt_archive(backup_dir, archive, gpgkey)
      backup_path = File.join(backup_dir, archive)
      encrypted_output = backup_path + '.gpg'
      gpg_cmd = "gpg --trust-model always --recipient #{gpgkey} --output #{encrypted_output} --encrypt #{backup_path}"
      success = system(gpg_cmd)
      if success && File.exist?(encrypted_output)
        size = File.size(encrypted_output)
        FileUtils.remove_entry(backup_path) if File.exist?(backup_path)
        return [encrypted_output, size]
      else
        raise _("Error: gpg encryption of the archive failed. Unencrypted archive is available at #{backup_path}")
      end
    end

    def self.gzip_local_backup(dir, backup_dir, archive)
      backup_path = File.join(backup_dir, archive)
      success = system("gzip -c #{dir}/#{archive} > #{backup_path}")
      raise _('Failed to compress archive with gzip.') unless success

      archive_sha = Digest::SHA256.file(backup_path).hexdigest
      archive_size = File.size(backup_path)

      return [archive_sha, archive_size]
    end

    def self.puppet_infrastructure_recover(environment, logfile)
      loop do
        `/opt/puppetlabs/bin/puppet-infrastructure recover_configuration --pe-environment #{environment} >> #{logfile} 2>&1`
        exitstatus = $CHILD_STATUS.exitstatus
        break if exitstatus.zero?
        if exitstatus == 17
          puts _('Puppet agent run currently in progress. Waiting 30 seconds before trying again.')
          sleep(30)
        else
          raise _('Command `puppet infrastructure recover_configuration` failed.') unless $CHILD_STATUS.exitstatus.zero?
        end
      end
    end

    def self.validate_backup_location(backup_dir, backup_filename)
      raise _('Specified backup directory %{dir} does not exist.') % { dir: backup_dir } unless Dir.exist?(backup_dir)
      raise _('Specified backup directory %{dir} is not writable.') % { dir: backup_dir } unless File.writable?(backup_dir)
      raise _('Specified backup name %{filename} cannot be a file path.') % { filename: backup_filename } if backup_filename != File.basename(backup_filename)
      backup_location = File.join(backup_dir, backup_filename)
      raise _('File already exists at desired location %{filepath}.') % { filepath: backup_location } if File.exist?(backup_location)
    end

    def self.validate_postgres_writable(dir)
      raise _('Staging directory %{dir} inside the backup directory is not writable by pe-postgres user.') % { dir: dir } unless system("su - pe-postgres -s /bin/bash -c 'if [ -w #{dir} ] ; then exit 0; else exit 1; fi'")
    end

    def self.write_metadata_file(dir, archive, backup_metadata)
      bytes_written = File.write(File.join(dir, 'backup_metadata.json'), JSON.pretty_generate(backup_metadata))
      success = system("tar -P -C '#{dir}' -rf #{dir}/#{archive} backup_metadata.json") if bytes_written.positive?
      raise _('Failed to add metadata file to backup.') unless success
    end

    def self.global_excluded_paths
      %w[
        /opt/puppetlabs/puppet
        /opt/puppetlabs/installer
        /opt/puppetlabs/server/pe_build
        /opt/puppetlabs/server/data/packages
        /opt/puppetlabs/server/apps
        /opt/puppetlabs/server/data/postgresql
        /opt/puppetlabs/server/data/puppetserver/yaml/facts
        /opt/puppetlabs/server/data/code-manager/worker-caches
        /opt/puppetlabs/server/data/environments/enterprise/modules
        /opt/puppetlabs/server/data/puppetserver/vendored-jruby-gems
        /opt/puppetlabs/bin
        /opt/puppetlabs/client-tools
        /opt/puppetlabs/puppet-metrics-collector
        /opt/puppetlabs/server/share
        /etc/puppetlabs/nginx/conf.d/http_redirect.conf
        /etc/puppetlabs/nginx/conf.d/proxy.conf
        /etc/puppetlabs/client-tools/services.conf
      ]
    end

    def self.get_excluded_paths(scope)
      excluded_paths = {
        code: %w[],
        config: %w[
          /etc/puppetlabs/code
          /etc/puppetlabs/code-staging
          /etc/puppetlabs/console-services/conf.d/secrets
          /etc/puppetlabs/puppet/ssl
          /etc/puppetlabs/orchestration-services/conf.d/secrets
          /opt/puppetlabs/server/data/puppetserver/filesync/storage
          /opt/puppetlabs/server/data/puppetserver/filesync/client
          /opt/puppetlabs/server/data/puppetdb
        ],
        certs: %w[],
        puppetdb: %w[
          /opt/puppetlabs/server/data/puppetdb/stockpile
        ]
      }

      included_paths = scope == ['config'] ? [] : get_included_paths(scope.reject { |s| s == 'config' })
      excluded_paths.select { |k, _| scope.include? k.to_s }
                    .values
                    .concat(global_excluded_paths)
                    .flatten
                    .reject { |path| included_paths.include? path }
    end

    def self.get_included_paths(scope)
      included_paths = {
        code: %w[
          /etc/puppetlabs/code
          /etc/puppetlabs/code-staging
          /opt/puppetlabs/server/data/puppetserver/filesync/storage
        ],
        config: %w[
          /etc/puppetlabs
          /opt/puppetlabs
        ],
        certs: %w[
          /etc/puppetlabs/puppet/ssl
        ],
        puppetdb: %w[
          /opt/puppetlabs/server/data/puppetdb
        ]
      }

      # If the scope includes 'config' all the other paths just end up being
      # redundant
      functional_scope = scope.include?('config') ? ['config'] : scope
      included_paths.select { |k, _| functional_scope.include? k.to_s }
                    .values
                    .flatten
    end

    def self.get_included_dbs(scope)
      included_dbs = {
        config: %w[
          pe-orchestrator
          pe-rbac
          pe-classifier
          pe-activity
          pe-inventory
        ],
        puppetdb: %w[
          pe-puppetdb
        ]
      }
      included_dbs.select { |k, _| scope.include? k.to_s }.values.flatten
    end

    def self.format_excluded_paths(excluded_paths)
      excluded_paths.map { |excluded_path| "--exclude #{excluded_path}" if File.exist? excluded_path }.join(' ')
    end

    def self.format_included_paths(included_paths)
      included_paths.map { |included_path| included_path.to_s if File.exist? included_path }.join(' ')
    end

    def self.get_fs_size(scope)
      included_paths_args = format_included_paths(get_included_paths(scope))
      excluded_paths_args = format_excluded_paths(get_excluded_paths(scope))

      stdout, stderr, status = Open3.capture3("du --bytes --max-depth=0 #{included_paths_args} #{excluded_paths_args}")
      raise stderr unless status.success?
      stdout
    end

    def self.get_db_size(db)
      raise _('Failed to get database size: database not specified.') unless db
      psql = "SELECT pg_database_size(datname) FROM pg_database WHERE datname = '#{db}'"
      psql_command = %(/opt/puppetlabs/server/bin/psql --tuples-only --no-align  --quiet --no-psqlrc -c \\"#{psql}\\")
      bash_command = %(su - pe-postgres -s /bin/bash -c "#{psql_command}")
      stdout, stderr, status = Open3.capture3(bash_command)
      raise stderr unless status.success?
      stdout.to_i
    end

    def self.estimate_backup_size(scopes)
      backup_size = { total_size: 0 }

      scopes.each do |scope|
        backup_size[scope] = { total_size: 0 }
        backup_size[scope][:filesystem] = get_fs_size([scope]).each_line.map do |row|
          size, dir = row.chomp.split(' ')
          size = size.to_i
          backup_size[:total_size] += size
          backup_size[scope][:total_size] += size
          { dir: dir, size: size }
        end
        included_dbs = get_included_dbs([scope])
        backup_size[scope][:database] = included_dbs.map do |db|
          db_size = get_db_size(db)
          backup_size[:total_size] += db_size
          backup_size[scope][:total_size] += db_size
          { db: db, size: db_size }
        end
      end
      backup_size
    end

    def self.backup_filesystem(dir, archive, scope_config, logger)
      included_paths = format_included_paths(scope_config[:included_paths])
      excluded_paths = format_excluded_paths(scope_config[:excluded_paths])

      timeout = 2
      8.times do
        output, = Open3.capture2e("tar -P #{excluded_paths} -cf #{dir}/#{archive} #{included_paths}")
        if %r{file changed as we read it}.match?(output)
          logger.warn("A file changed as we read it while backing up the filesystem, waiting #{timeout} seconds before trying again")
          sleep timeout
          timeout *= 2
        else
          return
        end
      end

      raise _('Failed to create file system backup.')
    end

    def self.backup_database(db, dir, archive)
      success = system("su - pe-postgres -s /bin/bash -c \"/opt/puppetlabs/server/bin/pg_dump -Fd -Z3 -j4 #{db} -f #{dir}/#{db}.bin\"")
      success = system("tar -P -C '#{dir}' -rf #{dir}/#{archive} #{db}.bin") if success
      raise _('Failed to backup PE database %{db}.') % { db: db } unless success
    ensure
      # Remove the binary database backup now that it's just taking up space
      FileUtils.remove_entry(File.join(dir, "#{db}.bin")) if Dir.exist?(File.join(dir, "#{db}.bin"))
    end

    def self.generate_scope_config(options)
      included_paths = get_included_paths(options[:scope])
      excluded_paths = get_excluded_paths(options[:scope])
      included_dbs = get_included_dbs(options[:scope])

      {
        scope: options[:scope],
        included_paths: included_paths,
        excluded_paths: excluded_paths,
        included_dbs: included_dbs
      }
    end

    def self.compute_total_steps(scope, skip_encryption)
      steps = 10
      # recover_configuration isn't required unless :config is specified
      # 5 databases aren't included unless 'config' is specified
      steps -= 6 unless scope.include?('config')
      steps -= 1 unless scope.include?('puppetdb')
      steps -= 1 if skip_encryption
      steps
    end

    def self.backup(options)
      raise _('Error: This command is limited to only the primary node.') unless PeBackupTools::Utils.primary_master?

      scope_config = generate_scope_config(options)

      if scope_config[:scope].include?('config') || scope_config[:scope].include?('puppetdb')
        raise _('Error: This command cannot backup up the `certs` or `code` scopes on split installations.') unless PeBackupTools::Utils.monolithic?
      end

      timestamp = Time.now.utc.strftime('pe_backup-%Y-%m-%d_%H.%M.%S_UTC')
      logdir = File.expand_path(options[:logdir] || '/var/log/puppetlabs/pe-backup-tools')
      logfilename = timestamp + '.log'
      logger, logfile = PeBackupTools::Utils.configure_logging(logdir, logfilename)

      gpg_check(options[:gpgkey]) unless options[:gpgkey].nil?

      # '/var/puppetlabs/backups' dir get created in the pe-backup-tools-vanagon packaging
      backup_dir = File.expand_path(options[:backup_dir] || '/var/puppetlabs/backups')
      archive = options[:backup_name] || timestamp + '.tgz'
      timings = {}
      step_printer = PeBackupTools::Utils::StepPrinter.new(compute_total_steps(scope_config[:scope], options[:gpgkey].nil?))

      validate_backup_location(backup_dir, archive)

      dir = Dir.mktmpdir('staging', backup_dir)
      # PostgreSQL needs permissions to write to the tmpdir
      FileUtils.chown_R('pe-postgres', 'pe-postgres', dir)

      validate_postgres_writable(dir) if scope_config[:scope].include?('config') || scope_config[:scope].include?('puppetdb')

      puts _('Starting PE backup process.') + ' ' + _('Note: The backup process does not disrupt Puppet Enterprise services.')
      puts ''
      puts _('Checking disk space for backup.') + ' ' + _('This calculation may take a few minutes.')
      puts ''
      backup_size = estimate_backup_size(scope_config[:scope])
      backup_size[:backup_name] = archive

      unless options[:force]
        timings['backup_space_comparison'] = PeBackupTools::Utils.benchmark do
          free_space, mount_point = PeBackupTools::Utils.extract_available_space_and_mount_point(backup_dir)
          # Since we can't easily estimate the compressed size of the archive before actually creating
          # the archive and because the archive file and compressed file will exist next to each other
          # for a brief moment, we will multiply the total size by 2 (this will always be an overestimate).
          working_space_needed = backup_size[:total_size] * 2
          humanized_free_space = PeBackupTools::Utils.humanize_size(free_space)
          humanized_working_space_needed = PeBackupTools::Utils.humanize_size(working_space_needed)

          puts _('Estimated backup size: %{backup_size}') % { backup_size: PeBackupTools::Utils.humanize_size(backup_size[:total_size]) }
          puts _('Estimated space needed to back up: %{space_needed}') % { space_needed: humanized_working_space_needed }
          puts _('Disk space available: %{free_space}') % { free_space: humanized_free_space }
          puts ''
          if free_space < working_space_needed
            error_description = [
              _('There is not enough space in \'%{mount_point}\' to create the backup.') % { mount_point: mount_point },
              _('The directory has %{free_space} available, but the space needed to create the backup is %{space_needed}.') % { free_space: humanized_free_space, space_needed: humanized_working_space_needed },
              _('Free up some space in \'%{mount_point}\' or create your backup in a different location.') % { mount_point: mount_point }
            ].join(' ')
            raise error_description
          end
        end
      end

      gpg_extension = options[:gpgkey].nil? ? '' : '.gpg'
      puts _('Creating backup at %{backup_path}') % { backup_path: File.join(backup_dir, archive) + gpg_extension }
      puts ''
      puts _('Log messages will be saved to %{logfile}') % { logfile: logfile }
      puts ''

      if scope_config[:scope].include?('config')
        timings['recover_configuration'] = PeBackupTools::Utils.benchmark do
          step_printer.step(_('Backing up PE related classification'))
          environment = options[:pe_environment] || Puppet[:environment]
          puppet_infrastructure_recover(environment, logfile)
        end
      end

      timings['file_system_backup'] = PeBackupTools::Utils.benchmark do
        step_printer.step(_('Creating file system backup'))
        backup_filesystem(dir, archive, scope_config, logger)
      end

      scope_config[:included_dbs].each do |db|
        timings["#{db.tr('-', '_')}_backup"] = PeBackupTools::Utils.benchmark do
          step_printer.step(_('Backing up the %{db} database') % { db: db })
          backup_database(db, dir, archive)
        end
      end

      certname = PeBackupTools::Utils.get_certname
      backup_metadata = {
        uncompressed_size: backup_size,
        scope_config: scope_config,
        certname: certname,
        pe_version: PeBackupTools::Utils.get_pe_version
      }
      write_metadata_file(dir, archive, backup_metadata)

      archive_sha = nil
      archive_size = nil
      timings['gzip_backup'] = PeBackupTools::Utils.benchmark do
        step_printer.step(_('Compressing archive file to backup directory'))
        archive_sha, archive_size = gzip_local_backup(dir, backup_dir, archive)
      end

      timings = timings.merge(total: PeBackupTools::Utils.get_total_runtime(timings))

      certname = PeBackupTools::Utils.get_certname
      cert_conf = PeBackupTools::Utils.get_agent_cert_conf(certname)
      analytics_url = "https://#{certname}:8140/analytics/v1"
      PeBackupTools::Utils::AnalyticsClient.store_event(analytics_url, cert_conf, 'puppet-backup.create',
                                                        backup_dir: options[:backup_dir].nil? ? false : true,
                                                        backup_name: options[:backup_name].nil? ? false : true,
                                                        scope: scope_config[:scope],
                                                        force: options[:force] ? true : false,
                                                        timings: timings)

      backup_path = File.join(backup_dir, archive)

      encryption = 'no'
      if options[:gpgkey].nil?
        logger.info(_('Backup creation succeeded (size: %{size}, scope: %{scope}, time: %{time}, sha256: %{sha}, filename: %{backup_path}, gpg_encryption: %{encryption})') % { size: PeBackupTools::Utils.humanize_size(archive_size), scope: scope_config[:scope], time: PeBackupTools::Utils.humanize_time(timings[:total]), sha: archive_sha, backup_path: backup_path, encryption: encryption })
      else
        encrypted_file = nil
        encrypt_size   = nil
        timings['encrypt_backup'] = PeBackupTools::Utils.benchmark do
          step_printer.step(_('Encypting archive file using gpg key '))
          encrypted_file, encrypt_size = encrypt_archive(backup_dir, archive, options[:gpgkey])
        end
        encryption = 'yes'
        backup_path = encrypted_file
        timings = timings.merge(total: PeBackupTools::Utils.get_total_runtime(timings))
        logger.info(_('Backup creation succeeded (size: %{size}, scope: %{scope}, time: %{time}, sha256: %{sha}, filename: %{encrypted_file}, gpg_encryption: %{encryption})') % { size: PeBackupTools::Utils.humanize_size(encrypt_size), scope: scope_config[:scope], time: PeBackupTools::Utils.humanize_time(timings[:total]), sha: archive_sha, encrypted_file: encrypted_file, encryption: encryption })
      end

      puts _('Warning: This backup does not include the secret key used to encrypt and decrypt sensitive data stored in the inventory service. You must back up this information separately. /etc/puppetlabs/orchestration-services/conf.d/secrets')
      puts _('Warning: This backup does not include the secret key used to encrypt and decrypt sensitive data stored in the inventory service. You must back up this information separately. /etc/puppetlabs/console-services/conf.d/secrets')
      puts ''
      {
        command: 'create',
        status: 'success',
        filename: backup_path,
        digest: archive_sha,
        size: archive_size,
        scope: scope_config[:scope],
        runtime: timings
      }
    rescue RuntimeError => e
      logger.error(_('Backup creation failed with: %{error}') % { error: e.message }) if logger
      {
        command: 'create',
        status: 'failed',
        error_description: e.message,
        logfile: logfile
      }
    ensure
      FileUtils.remove_entry(dir) if dir
    end
  end
end
