require 'pe_backup_tools/restore'
require 'pe_backup_tools/backup'

module PeBackupTools
  # This module provides functionality for interacting with the CLI commands
  class CLI
    attr_reader :options, :subcommands

    def peenv_flag(opts, options)
      opts.on('--pe-environment=ENVIRONMENT', _('The environment of the PE Infrastructure.') + ' ' + _('Defaults to production')) do |pe_environment|
        options[:pe_environment] = pe_environment
      end
    end

    def force_flag(opts, options)
      opts.on('--force', _('Bypass validation checks and ignore warnings.')) do |force|
        options[:force] = force
      end
    end

    def tempdir_flag(opts, options)
      opts.on('--tempdir=PATH', _('The directory for restore activities, defaults to $TMPDIR if set or /tmp if not')) do |tempdir|
        options[:tempdir] = tempdir
      end
    end

    def scope_flag(opts, options)
      msg = [
        _('Scope of backup to create or restore. ex: --scope=config,certs'),
        _('Select \'all\', or any combination of the other scope options in a comma-separated list.'),
        '  all: (' + _('default') + ').',
        '  code: ' + _('Puppet code deployed to your codedir at backup time.'),
        '  config: ' + _('PE Configuration including license, classification & RBAC settings.') + ' ' + _('Does NOT include puppet_gems or puppetserver_gems.'),
        '  puppetdb: ' + _('PuppetDB data, including facts, catalogs and historical reports.'),
        '  certs: ' + _('PE CA Certificates and full ssl directory.')
      ]
      opts.on('--scope=SCOPE', *msg) do |scope|
        options[:scope] = scope.split(',').map(&:chomp).map(&:downcase)
      end
    end

    def global_options
      options = {}
      OptionParser.new do |opts|
        opts.banner = 'Global options:'
        force_flag(opts, options)
        tempdir_flag(opts, options)
        peenv_flag(opts, options)
        scope_flag(opts, options)
      end
    end

    def get_valid_scopes
      %w[
        code
        puppetdb
        config
        certs
      ]
    end

    def validate_scope_options(scopes)
      if scopes.empty?
        warn(_('Error: Flag values cannot be empty'))
        warn ''
        exit(1)
      end

      invalid = scopes.reject { |scope| get_valid_scopes.include? scope }

      unless invalid.empty?
        warn(n_('Error: Invalid scope option: %{invalid}', 'Invalid scope options: %{invalid}', invalid.length) % { invalid: invalid.join(', ').chomp })
        warn ''
        exit(1)
      end
    end

    def initialize
      @options = {}
      @subcommands = {
        'create' => OptionParser.new do |opts|
          opts.banner = 'Usage: puppet-backup create [options]'
          opts.on('--dir=BACKUP_DIR',
                  _('Directory to save backup file.') + ' ' + _('Defaults to /var/puppetlabs/backups/')) do |backup_dir|
            options[:backup_dir] = backup_dir
          end
          opts.on('--name=BACKUP_NAME',
                  _('Name to give to backup file.') + ' ' + _('Defaults to pe_backup-<timestamp>.tgz')) do |backup_name|
            options[:backup_name] = backup_name
          end
          opts.on('--gpgkey=GPG_KEY_ID',
                  _('GPG key ID to be used for encryption.') + ' ' + \
                  _(' Make sure to import the public key before using this option')) do |gpgkey|
            options[:gpgkey] = gpgkey
          end
          force_flag(opts, options)
          tempdir_flag(opts, options)
          peenv_flag(opts, options)
          scope_flag(opts, options)
        end,
        'restore' => OptionParser.new do |opts|
          opts.banner = 'Usage: puppet-backup restore <backup-filepath> [options]'
          force_flag(opts, options)
          tempdir_flag(opts, options)
          peenv_flag(opts, options)
          scope_flag(opts, options)
        end
      }
    end

    def command?(cmd)
      subcommands.key?(cmd)
    end

    def command(cmd)
      subcommands[cmd].order!
      if options.find { |_, value| value.chomp == '' if value.is_a?(String) }
        warn(_('Error: Flag values cannot be empty'))
        warn ''
        exit(1)
      end
      case cmd
      when 'restore'
        if ARGV.empty?
          warn(_('Error: Must supply a backup location.'))
          warn ''
          warn(subcommands['restore'])
          exit(1)
        else
          backup = ARGV.shift
          subcommands[cmd].order!
          unless ARGV.empty?
            warn(_('Error: Only one backup location can be specified.'))
            warn ''
            warn(subcommands['restore'])
            exit(1)
          end
          options[:backup] = File.expand_path(backup)
        end
        if options[:scope] == ['all'] || !options[:scope]
          # By default the scope of the restore is everything in
          # the backup
          options[:scope] = 'all'
        else
          validate_scope_options(options[:scope])
        end
        info = PeBackupTools::Restore.restore(options)
        PeBackupTools::Utils::CommandSummary.summarize_restore(info)
        info[:status] == 'success' ? exit : exit(1)
      when 'create'
        unless ARGV.empty?
          warn(_('Error: Positional arguments are not allowed.'))
          warn ''
          warn(subcommands['create'])
          exit(1)
        end
        if options[:scope] == ['all'] || !options[:scope]
          options[:scope] = get_valid_scopes
        else
          validate_scope_options(options[:scope])
        end
        info = PeBackupTools::Backup.backup(options)
        PeBackupTools::Utils::CommandSummary.summarize_create(info)
        info[:status] == 'success' ? exit : exit(1)
      else
        raise _('Error: Unrecognized command.')
      end
    rescue OptionParser::ParseError => e
      warn e.message
      warn ''
      warn subcommands[cmd]
      exit(1)
    end
  end
end
