require 'hocon'
require 'json'
require 'net/https'
require 'uri'
require 'openssl'
require 'puppet'
require 'benchmark'
require 'base64'
require 'pe_backup_tools/utils/command_summary'
require 'facter'
require 'open3'

module PeBackupTools
  # This module provides utility functions related to a variety
  # of things  (passwords, GET/POST, certificates) for our instance commands
  module Utils
    Puppet.initialize_settings

    def self.get_certname
      Puppet[:certname]
    end

    def self.get_fqdn
      Facter.value('fqdn')
    end

    def self.get_pe_version
      # TODO: Get this from the Facter fact maybe?
      File.read('/opt/puppetlabs/server/pe_version').chomp
    end

    def self.get_cert_conf(cert_dir, hostname)
      {
        'ca_cert' => '/etc/puppetlabs/puppet/ssl/certs/ca.pem',
        'private_key' => "#{cert_dir}/#{hostname}.private_key.pem",
        'cert' => "#{cert_dir}/#{hostname}.cert.pem"
      }
    end

    def self.get_agent_cert_conf(certname)
      {
        'ca_cert' => '/etc/puppetlabs/puppet/ssl/certs/ca.pem',
        'private_key' => "/etc/puppetlabs/puppet/ssl/private_keys/#{certname}.pem",
        'cert' => "/etc/puppetlabs/puppet/ssl/certs/#{certname}.pem"
      }
    end

    def self.get_pe_conf
      pe_conf_path = '/etc/puppetlabs/enterprise/conf.d/pe.conf'
      pe_conf = Hocon.load(pe_conf_path)
      raise _('Failed to find `puppet_master_host` in `%{pe_conf_path}`') % { pe_conf_path: pe_conf_path } unless pe_conf['puppet_enterprise::puppet_master_host']
      pe_conf
    end

    def self.primary_master?
      pe_conf = get_pe_conf
      puppet_master_host = pe_conf['puppet_enterprise::puppet_master_host']
      puppet_master_host = get_certname if puppet_master_host == '%{::trusted.certname}'
      (puppet_master_host == get_certname)
    end

    def self.monolithic?
      pe_conf = get_pe_conf
      puppet_master_host = pe_conf['puppet_enterprise::puppet_master_host']
      puppet_master_host = get_certname if puppet_master_host == '%{::trusted.certname}'
      console_host       = pe_conf['puppet_enterprise::console_host']  || puppet_master_host
      puppetdb_host      = pe_conf['puppet_enterprise::puppetdb_host'] || puppet_master_host
      database_host      = pe_conf['puppet_enterprise::database_host'] || puppet_master_host
      (console_host == puppet_master_host) && (puppetdb_host == puppet_master_host) && (database_host == puppet_master_host)
    end

    def self.build_auth(uri, cert_conf)
      https = Net::HTTP.new(uri.host, uri.port)
      https.use_ssl = true
      https.ca_file = cert_conf['ca_cert']
      https.key = OpenSSL::PKey::RSA.new(File.read(cert_conf['private_key']))
      https.cert = OpenSSL::X509::Certificate.new(File.read(cert_conf['cert']))
      https.verify_mode = OpenSSL::SSL::VERIFY_PEER
      https
    end

    def self.configure_logging(logdir, logfilename)
      logfile = logdir + '/' + logfilename
      logger = Logger.new(logfile)
      logger.level = Logger::DEBUG
      [logger, logfile]
    rescue Errno::EACCES => _e
      raise _('Unable to access log file `%{logfile}` to configure logging.') % { logfile: logfile }
    end

    def self.benchmark(&block)
      Benchmark.realtime do
        yield block
      end
    end

    def self.get_total_runtime(timings)
      timings.inject(0) do |sum, (_, v)|
        sum + v
      end
    end

    def self.post_response(url, cert_conf, request_body = nil)
      uri = URI.parse(url)
      https = build_auth(uri, cert_conf)

      request = Net::HTTP::Post.new(uri.request_uri)
      request['Content-Type'] = 'application/json'
      request.body = request_body if request_body

      res = https.request(request)
      case res
      when Net::HTTPSuccess then
        res
      else
        raise _('An error occurred posting to %{url}: HTTP %{http_code}, %{metadata}, %{body}.') % { url: url, http_code: res.code, metadata: res.to_hash.inspect, body: res.body }
      end
    end

    def self.get_response(url, cert_conf)
      uri = URI.parse(url)
      https = build_auth(uri, cert_conf)

      request = Net::HTTP::Get.new(uri.request_uri)
      request['Content-Type'] = 'application/json'

      res = https.request(request)
      case res
      when Net::HTTPSuccess then
        res
      else
        raise _('An error occurred getting a response from %{url}: HTTP %{http_code}, %{metadata}, %{body}.') % { url: url, http_code: res.code, metadata: res.to_hash.inspect, body: res.body }
      end
    end

    def self.puppet_node_deactivate(certname, logfile)
      success = system("/opt/puppetlabs/puppet/bin/puppet node deactivate #{certname} >>#{logfile} 2>&1")
      raise _('Received unknown exit code while deactivating the old primary node.') unless success
    end

    def self.puppet_infrastructure_configure(environment, logfile)
      `/opt/puppetlabs/bin/puppet-infrastructure configure --debug --detailed-exitcodes --no-noop --no-recover --pe-environment #{environment} >> #{logfile} 2>&1`
      case $CHILD_STATUS.exitstatus
      when 0, 2
        return
      when 1
        raise _('Failed to configure PE.')
      when 4, 6
        raise _('Configuring PE completed but with failures.')
      else
        raise _('Received unknown exit code while configuring PE.')
      end
    end

    def self.run_df_byte_size_portability(dir)
      stdout, stderr, status = Open3.capture3("df --block-size=1 --portability #{dir}")
      raise stderr unless status.success?
      return stdout
    end

    def self.extract_available_space_and_mount_point(dir)
      last_line = run_df_byte_size_portability(dir).split("\n").last
      free_space = last_line.split(' ')[3].to_i
      mount_point = last_line.split(' ')[5]
      return [free_space, mount_point]
    end

    def self.humanize_time(secs)
      if secs < 60
        _('%{seconds} sec') % { seconds: secs.round }
      else
        minutes, seconds = secs.divmod(60)
        if minutes < 60
          _('%{minutes} min, %{seconds} sec') % { minutes: minutes, seconds: seconds.round }
        else
          hours, minutes = minutes.divmod(60)
          _('%{hours} hr, %{minutes} min, %{seconds} sec') % { hours: hours, minutes: minutes, seconds: seconds.round }
        end
      end
    end

    def self.humanize_size(bytes)
      bytes = bytes.to_i
      units = %w[B KB MB GB TB].freeze

      if bytes < 1024
        exponent = 0
        num = bytes
      else
        max_exp  = units.size - 1

        exponent = (Math.log(bytes) / Math.log(1024)).to_i # convert to base
        exponent = max_exp if exponent > max_exp # we need this to avoid overflow for the highest unit
        num = (bytes / 1024.0**exponent).round(2)
      end
      "#{num} #{units[exponent]}"
    end

    # Helper class for printing the steps of a backup/restore
    class StepPrinter
      def initialize(total_steps)
        @step_counter = 1
        @total_steps = total_steps
      end

      def step(msg)
        puts "Step #{@step_counter} of #{@total_steps}: #{msg}"
        @step_counter += 1
      end
    end
  end
end
