# frozen_string_literal: true

module PeInstaller

  # Raised when a inventory does not allow us to evaluate architecture.
  class ArchitectureError < PeInstaller::Error; end

  # Raised with validation errors.
  class InvalidArchitecture < ArchitectureError
    attr_accessor :architecture, :errors

    def initialize(architecture, errors)
      self.architecture = architecture
      self.errors = errors
      super("#{architecture}. Errors:\n#{errors.join("\n")}")
    end
  end

  # Abstraction for the type of PE installation.
  # https://puppet.com/docs/pe/latest/choosing_an_architecture.html
  #
  # Standard: master
  # Large: master + compilers
  #
  # Either Standard or Large may have a replica set up.
  #
  # Extra Large: master + database + compilers
  #
  # Replica are not supported in Extra Large (but see below).
  #
  # There is a complication in that 'extra large' for puppetlabs-peadm
  # is different architecture with postgresql on both the master and the database,
  # node group changes, managed load balancers and the ability to set up a replica.
  class Architecture
    attr_accessor :inventory, :errors

    # @param targets [Bolt::Target,Array<Bolt::Target>]
    # @return [Array<String>] converts targets to a list of name or uri strings.
    def self.as_certs(targets)
      Array(targets).map { |t| t.name || t.uri }
    end

    # @param inventory [PeInstaller::Inventory]
    def initialize(inventory)
      self.inventory = inventory
    end

    # @return [Symbol] :standard, :large, :extra_large
    # @raise [PeInstaller::ArchitectureError] if there is no master.
    def size
      raise(PeInstaller::ArchitectureError, 'No master node.') if master.nil?
      if !split?
        compilers.empty? ? :standard : :large
      else
        :extra_large
      end
    end

    # @return [Boolean] true if installation architecture is +standard+.
    def standard?
      size == :standard
    end

    # @return [Boolean] true if installation architecture is +large+.
    def large?
      size == :large
    end

    # @return [Boolean] true if installation architecture is +extra_large+.
    def extra_large?
      size == :extra_large
    end

    # @return [Bolt::Target] master target.
    def master
      inventory.masters.first
    end

    # This must be one.
    #
    # @return [Integer] number of masters.
    def master_count
      inventory.masters.size
    end

    # @return [Bolt::Target] database target.
    def database
      inventory.databases.first
    end

    # This must be one or zero.
    #
    # @return [Integer] number of databases.
    def database_count
      inventory.databases.size
    end

    # These are the primary infrastructure nodes which must exist for all
    # core PE services to be present, and for PE to be able to manage itself and
    # extend itself with additional compilers, replica, agents.
    #
    # @return [Array[Bolt::Target]] array of unique core target(s) (master and database).
    def core
      [master, database].compact.uniq
    end

    # @return [Boolean] true if there is more than one core service node.
    # @raise [PeInstaller::ArchitectureError] if there is no master.
    def split?
      raise(PeInstaller::ArchitectureError, 'No master node.') if master.nil?
      core.size > 1
    end

    # @return [Array<Bolt::Target>] array of replica targets.
    def replicas
      inventory.replicas
    end

    # @return [Boolean] true if installation has High Availability set up.
    def ha?
      !replicas.empty?
    end

    # @return [Array<Bolt::Target>] array of compiler targets.
    def compilers
      inventory.compilers
    end

    # @return [Array<Bolt::Target>] array of all core, replica and compiler targets.
    def targets
      [core, replicas, compilers].flatten
    end

    def valid?
      validate if @errors.nil?
      @errors.empty?
    end

    # Tests for a valid Architecture instance.
    VALIDATORS = {
      must_have_master: {
        test: ->(a) { !a.master.nil? },
        message: 'No master node was found.',
      }.freeze,

      must_have_single_master: {
        test: ->(a) { a.master_count == 1 },
        message: lambda do |a|
          masters = Architecture.as_certs(a.inventory.masters).join(', ')
          "There must be a single master. Found: #{masters}."
        end,
        skip_if: :must_have_master,
      }.freeze,

      must_not_have_more_than_one_database: {
        test: ->(a) { a.database_count <= 1 },
        message: lambda do |a|
          databases = Architecture.as_certs(a.inventory.databases).join(', ')
          "There must be at most one database. Found: #{databases}."
        end,
      }.freeze,

      must_not_intersect_core_and_replica_targets: {
        test: ->(a) { (a.core & a.replicas).empty? },
        message: lambda do |a|
          overlap = Architecture.as_certs(a.core & a.replicas).join(', ')
          "The master (and database) cannot overlap replica targets. Found: #{overlap} in both."
        end,
      }.freeze,

      must_not_intersect_core_and_compiler_targets: {
        test: ->(a) { (a.core & a.compilers).empty? },
        message: lambda do |a|
          overlap = Architecture.as_certs(a.core & a.compilers).join(', ')
          "The master (and database) cannot overlap compiler targets. Found: #{overlap} in both."
        end,
      }.freeze,

      must_not_intersect_compiler_and_replica_targets: {
        test: ->(a) { (a.compilers & a.replicas).empty? },
        message: lambda do |a|
          overlap = Architecture.as_certs(a.compilers & a.replicas).join(', ')
          "Replica and compiler targets cannot overlap. Found: #{overlap} in both."
        end,
      }.freeze,

      must_not_have_replica_if_extra_large: {
        test: ->(a) { !a.extra_large? || !a.ha? },
        message: lambda do |a|
          master = Architecture.as_certs(a.master).first
          database = Architecture.as_certs(a.database).first
          replicas = Architecture.as_certs(a.replicas).join(', ')
          [
            'An extra_large installation cannot have replica.',
            "Found: master: #{master}, database: #{database}, replicas: #{replicas}.",
          ].join(' ')
        end,
        skip_if: :must_have_master,
      }.freeze,
    }.freeze

    # @return [Array<String>] an array of error messages produced by validating
    # the architecture. If empty, the architecture is valid.
    def validate
      validator = lambda do |validators, subject|
        errors = {}
        validators.each do |key, rule|
          skip_if = rule[:skip_if] || :_no_skip_
          next if errors.keys.include?(skip_if)

          test = rule[:test]
          passed = test.call(subject)
          next if passed

          err_template = rule[:message]
          err =
            case err_template
            when String
              err_template
            else
              err_template.call(subject)
            end
          errors[key] = err
        end
        errors.values
      end

      @errors = validator.call(VALIDATORS, self)
    end

    # Validate and raise error if invalid.
    # @return [Boolean] true if valid.
    # @raise [PeInstaller::InvalidArchitecture] with the errors if not.
    def validate!
      errors = validate
      raise(PeInstaller::InvalidArchitecture.new(self, errors)) if !errors.empty?
      true
    end

    def to_s
      base = format("#<#{self.class}:0x%<id>0.16x", { id: (__id__ * 2) })
      targets = []
      targets << "master: #{Architecture.as_certs(inventory.masters)}" if master_count.positive?
      targets << "database: #{Architecture.as_certs(inventory.databases)}" if database_count.positive?
      targets << "compilers: #{Architecture.as_certs(inventory.compilers)}" if !compilers.empty?
      targets << "replicas: #{Architecture.as_certs(inventory.replicas)}" if !replicas.empty?
      "#{base}: #{targets.join(' ')}>"
    rescue StandardError
      "#{base}:Invalid>"
    end
  end
end
