# frozen_string_literal: true

require 'thor'
require 'pe_installer/cli_ext/setup'
require 'pe_installer/config'
require 'pe_installer/logger'
require 'pe_installer/manager'

module PeInstaller

  # The command-line structure.
  class CLI < Thor

    # Exit non-zero if an error is raised.
    def self.exit_on_failure?
      true
    end

    ################
    # Global options

    class_option(:verbose,
                 type: :boolean,
                 desc: 'Show plan details during execution.')
    class_option(:debug,
                 type: :boolean,
                 desc: 'Output debugging information.')
    class_option(:trace,
                 type: :boolean,
                 desc: 'Add stacktrace to error messages.')
    class_option(:quiet,
                 type: :boolean,
                 desc: 'Suppress output unless erroring.')
    class_option(:format,
                 type: :string,
                 enum: %w[human json],
                 desc: 'Output format.')

    # @return [Array<String>] of the global class options for the tool.
    def self.global_option_keys
      class_options.keys.map(&:to_s)
    end

    include PeInstaller::CliExt::Setup
    include PeInstaller::Logger

    ##################
    # Install Commands

    desc('install', 'Install Puppet Enterprise')
    long_desc(
      <<~LONG
        The install command will install a version of Puppet Enterprise for the
        first time.

        It must be passed a path to a PE tarball, and either a direct reference
        to, minimally, a master fqdn/certname, or a pe.conf file that contains the
        same.
      LONG
    )
    option(:tarball,
           type: :string,
           required: true,
           desc: 'Path to the Puppet Enterprise tarball to install.')
    option(:master,
           type: :string,
           desc: 'The FQDN certname of the master node (defaults to this node).')
    option(:database,
           type: :string,
           desc: 'The FQDN certname of the database node (defaults to the master node).')
    option(:compilers,
           type: :array,
           desc: 'A list of FQDN certnames of compiler nodes (empty by default).')
    option(:replicas,
           type: :array,
           desc: 'A list of FQDN certnames of replica nodes (empty by default).')
    option(:pe_conf,
           type: :string,
           desc: 'Path to a pe.conf file with puppet_enterprise parameters.',
           long_desc: <<~LONG)
             Additional Puppet configuration may be in this HOCON file, if required.
             The core puppet_master_host and datbase_host properties will be picked up
             if set, but must not conflict with --master or --database.
           LONG

    def install
      config = setup(:install, options)
      info('Executing install')
      PeInstaller::Manager.manage(config: config)
    end

    desc('local-install', 'Install Puppet Enterprise services on this node.')
    long_desc(
      <<~LONG
        The local-install command is used internally to set up Puppet
        Enterprise services on the current node (based on its role) over the
        localhost adapter.

        In general, this command does not need to be run manually. The plan
        invoked by the install command will call this command on each install
        node as required.

        This command assumes an unpacked PE tarball is already present on the
        node.
      LONG
    )
    option(:pe_tarball_dir,
           type: :string,
           required: true,
           desc: 'Path to the unpacked Puppet Enterprise tarball directory.')
    option(:pe_conf,
           type: :string,
           default: '/etc/puppetlabs/enterprise/conf.d/pe.conf',
           desc: 'Path to the pe.conf file controlling configuration.')
    option(:profile,
           type: :string,
           enum: %w[all ca],
           default: 'all',
           desc: 'Which profile to apply.',
           long_desc: <<~LONG)
             The 'ca' profile is used when bootstrapping a split install to
             prepare just the certifacte authority on the master prior to
             installing the database on a separate node and then completing
             the rest of the installation on the master.
           LONG

    def local_install
      config = setup(:local_install, options)
      info("Executing local-install profile: #{options['profile']}")
      PeInstaller::Manager.manage(config: config)
    end

    #################
    # Upgrade Command

    desc('upgrade', '(TODO) Upgrade Puppet Enterprise')
    long_desc(
      <<-LONG
        Upgrade an existing PE installation to a new version.

        TODO
      LONG
    )
    def upgrade
      puts('Not yet implemented.')
    end

    #################
    # Repair Command

    desc('repair', '(TODO) Repair Puppet Enterprise')
    long_desc(
      <<-LONG
        Repair a broken PE installation.

        TODO
      LONG
    )
    def repair
      puts('Not yet implemented.')
    end

    ##################
    # Validate Command

    desc('validate', '(TODO) Validate Puppet Enterprise State')
    long_desc(
      <<-LONG
        Validate the working state of a PE Installation

        TODO
      LONG
    )
    def validate
      puts('Not yet implemented.')
    end

    #################
    # Support Command

    desc('support', '(TODO) Gather Support Script Details')
    long_desc(
      <<-LONG
        Run the support script on the nodes of a PE installtion.

        TODO
      LONG
    )
    def support
      puts('Not yet implemented.')
    end
  end
end
