# frozen_string_literal: true

require 'pe_installer/architecture'
require 'pe_installer/bolt_interface'
require 'pe_installer/inventory'
require 'pe_installer/reporter'

module PeInstaller

  # Errors raised while managing a plan.
  class ManagerError < PeInstaller::Error; end

  # The engine for managing Puppet Enterprise.
  class Manager
    attr_accessor :bolt
    attr_accessor :config
    attr_accessor :inventory
    attr_accessor :reporter

    # Initialize a Manager with a working Bolt interface and inventory.
    #
    # @return [PeInstaller::Manager]
    def self.prepare(config:, **kwargs)
      bolt = kwargs[:bolt] || PeInstaller::BoltInterface.instance

      inventory = kwargs[:inventory]
      if inventory.nil?
        seed_roles = PeInstaller::Inventory::NodeRoles.from_config(config)
        inventory = PeInstaller::Inventory.with_seed_roles(bolt.inventory, seed_roles)
      end

      reporter = kwargs[:reporter] || PeInstaller::Reporter.new(config)

      new(
        config: config,
        bolt: bolt,
        inventory: inventory,
        reporter: reporter
      )
    end

    # Prepare a Manager instance and dispatch.
    def self.manage(config:, **kwargs)
      manager = prepare(config: config, **kwargs)
      manager.send(config.command)
    end

    def initialize(config:, **kwargs)
      self.config = config
      self.bolt = kwargs[:bolt]
      self.inventory = kwargs[:inventory]
      self.reporter = kwargs[:reporter]
    end

    def install
      architecture = PeInstaller::Architecture.new(inventory)
      architecture.validate!

      tarball = config.arguments['tarball']
      platform = config.bootstrap_metadata['platform_tag']

      # TODO: this is not a good place for this side effect...
      # Resolve this inside architecture? Or PeInstaller::Inventory?
      # Is it always only relevant for master?
      PeInstaller::Util.resolve_local_target(architecture.master)

      plan = 'enterprise_tasks::testing::pre_check'
      parameters = {
        'targets'          => architecture.targets,
        'tarball_platform' => platform,
      }
      manage(plan, parameters)

      plan = 'enterprise_tasks::testing::install'
      parameters = {
        'tarball'   => tarball,
        'master'    => architecture.master,
        'database'  => architecture.database,
        'replicas'  => architecture.replicas,
        'compilers' => architecture.compilers,
        'other_pe_conf_parameters' => config.pe_conf_hash,
        'legacy_installer'         => false,
      }
      manage(plan, parameters)
    end

    def local_install
      plan = 'enterprise_tasks::testing::install_core_node'

      pe_build = config.bootstrap_metadata['pe_build_version']
      pe_install_parameters = {
        'bootstrap_to' => pe_build,
      }
      profile = config.arguments['profile']
      pe_install_parameters['ca_only'] = true if profile == 'ca'

      classes = [
        # Sadly, order is important here, as pe_install includes
        # puppet_enterprise::packages, so we need to declare packages first or
        # face a duplicate declaration error.
        ['puppet_enterprise::packages', { 'installing' => true }],
        ['pe_install', pe_install_parameters],
      ]

      parameters = {
        'target'             => 'localhost',
        'classes'            => classes,
        'pe_tarball_dir'     => config.arguments['pe_tarball_dir'],
        'pe_conf'            => config.arguments['pe_conf'],
        'manage_permissions' => true,
      }
      manage(plan, parameters)
    end

    def manage(plan, parameters)
      result = bolt.run_plan(plan, parameters)
      if !result.ok?
        # TODO: this isn't the right spot for this, but I need a non-zero exit for now
        # for pe_run_installer task script to report
        exit 1
      end
      result
    rescue Bolt::Error => e
      reporter.step_error(e)
    end
  end
end
