# Ensure that the executing block does so within the context
# of the given RBAC user logged in for the given lifetime.
#
# By default, assumes 'pe-enterprise-tasks-user' which is a
# temporary account specifically for these plans. The function
# will ensure that this user is created with a random password,
# and that the user is deleted before the function exits.
#
# Provides the path to the token file on the primary node, and
# the token itself to the block.
#
# @param primary [Target] the PE primary node.
#
# @param options [Hash] (see below)
#
# @option [String] options['account'] the RBAC account to log in as.
#
# @option [Enterprise_tasks::Absolute_path] options['token_file'] absolute path
# to the RBAC token file on the primary.
#
# @option [Optional[Variant[Sensitive[String],String]] options['password']
# password for the RBAC account. Will be generated if not given; useful if
# account is not given and we are generating a temporary account.
#
# @option [Optional[Enterprise_tasks::Lifetime]] options['lifetime'] lifetime of
# the RBAC token the login will generate. Defaults to '5m'.
Puppet::Functions.create_function('enterprise_tasks::with_puppet_access') do
  dispatch :with_access do
    param 'Target', :primary
    optional_param 'Optional[Hash]', :options
    block_param 'Callable', :_block
  end

  def with_access(primary, options = {}, &_block)
    token_file = '/opt/puppetlabs/installer/share/enterprise-tasks-token'.freeze
    et_user = 'pe-enterprise-tasks-user'.freeze

    lifetime = options['lifetime'] || '5m'
    account = options['account'] || et_user
    token_file = options['token_file'] || token_file

    password = if options['password'].nil?
                 call_function('enterprise_tasks::generate_random_password', 16)
               elsif options['password'].respond_to?(:unwrap)
                 options['password'].unwrap
               else
                 options['password']
               end

    if account == et_user
      # Generate the pe-enterprise-tasks-user account
      call_function('run_task', 'enterprise_tasks::create_enterprise_tasks_rbac_user', primary, 'password' => password)
    end

    # Log in
    call_function('enterprise_tasks::message', 'puppet_access', "Obtaining a token for '#{account}' with a #{lifetime} lifetime.")
    result = call_function(
      'run_task',
      'enterprise_tasks::puppet_access',
      primary,
      'account'    => account,
      'token_file' => token_file,
      'password'   => password,
      'lifetime'   => lifetime,
    ).first

    yield(token_file, result['token'])
  ensure
    # Ensure that we clean up a generated pe-enterprise-tasks-user
    call_function('run_task', 'enterprise_tasks::delete_enterprise_tasks_rbac_user', primary) if account == et_user
  end
end
