# This is run with Bolt over SSH.  Run from a primary that is a promoted replica,
# and turns the old primary into a replica. The dns_alt_names parameter
# must be supplied if the old primary has alt names set in puppet.conf.
#
# Steps this plan takes:
# 1.  If dns_alt_names is defined, verify that puppetserver conf files in /etc/puppetlabs/puppetserver/conf.d have certificate-authority.allow-subject-alt-names set to true.
#     This should be managed to true by default.
# 2.  Disable the agent on the current and old primaries with puppet agent --disable.
# 3.  Stop all PE services on the old primary, and remove the puppet infra recover_configuration cron job.
# 4.  On the old primary, rm -rf /etc/puppetlabs/enterprise/*
# 5.  On the current primary, ensure that the previous primary's certs are removed
# 6.  On the old primary, make sure pe-postgresql is running, then run the
#     following with: su -s /bin/bash - pe-postgres -c “/opt/puppetlabs/server/bin/psql”
#         a. SELECT pg_drop_replication_slot(slot_name) FROM pg_replication_slots;
#         b. UPDATE pg_database SET datallowcon = ‘false’ WHERE datname IN
#            (‘pe-activity’, ‘pe-classifier’, ‘pe-orchestrator’, ‘pe-rbac’);
#         c. SELECT pg_terminate_backend(pid) FROM pg_stat_activity WHERE datname
#            IN (‘pe-activity’, ‘pe-classifier’, ‘pe-orchestrator’, ‘pe-rbac’);
#         d. DROP DATABASE IF EXISTS “pe-activity”;
#         e. DROP DATABASE IF EXISTS “pe-classifier”;
#         f. DROP DATABASE IF EXISTS “pe-orchestrator”;
#         g. DROP DATABASE IF EXISTS “pe-rbac”;
# 7.  On the old primary, stop pe-postgresql.
# 8.  On the old primary, ensure /etc/puppetlabs/puppet/ssl and /etc/puppetlabs/puppetserver/ca and removed
# 9.  If dns_alt_names is defined, run puppet config set --section main dns_alt_names “<dns_alt_names value>” on the old primary.
# 10. On the old primary, run puppet agent -t --server_list <current_primary>:8140. This will generate a CSR.
# 11. On the current primary, run puppetserver ca sign --certname <old_primary>. Before we do this, we check the CSR with puppetserver ca list --certname <node>.
#     If dns_alt_names was not specified, we verify that no subject alt names appear in the CSR.
# 12. Copy /etc/puppetlabs/enterprise/hiera.yaml from the current primary to the old primary.
# 13. Run puppet infra provision replica <old_primary> --enable.
# 14. Enable the agent on the current and old primaries with puppet agent --enable.
plan enterprise_tasks::enable_ha_failover(
  TargetSpec $host,
  Enum['mono', 'mono-with-compile'] $topology,
  Optional[TargetSpec] $primary               = 'localhost',
  Optional[Integer] $replication_timeout_secs = 1800,
  Optional[Boolean] $skip_agent_config        = undef,
  Optional[String] $agent_server_urls         = undef,
  Optional[String] $pcp_brokers               = undef,
  Optional[String] $dns_alt_names             = undef,
  Optional[Boolean] $force                    = false,
) {
  $prev_primary = get_target($host)
  $curr_primary = get_target($primary)
  enterprise_tasks::test_connection([$prev_primary, $curr_primary])

  # Fail when either agent_server_urls or pcp_brokers is not supplied, when they need to be
  if ($topology == 'mono-with-compile' and !$skip_agent_config and (!$agent_server_urls or !$pcp_brokers)) {
    fail_plan(@(EOT/L))
      agent_server_urls and pcp_brokers are required parameters for the \
      'mono-with-compile' topology, unless skip_agent_config is set to true
      |-EOT
  }

  $certname_cmd = "${constants()['puppet_bin']} config print certname --section agent"
  $curr_primary_certname = run_command($certname_cmd, $curr_primary).first['stdout'].strip
  $prev_primary_certname = run_command($certname_cmd, $prev_primary).first['stdout'].strip

  enterprise_tasks::verify_node($curr_primary_certname, 'primary', $force)

  if $dns_alt_names{
    run_plan(enterprise_tasks::is_subject_alt_names_allowed, primary => $primary, force => $force)
  }
  $allow_subject_alt_names = $dns_alt_names ? {
    undef   => false,
    default => true
  }

  enterprise_tasks::with_agent_disabled([$prev_primary, $curr_primary]) || {
    run_task('enterprise_tasks::pe_services', $prev_primary,
      role  => 'primary',
      state => 'stopped',
    )
    apply($prev_primary) {
      cron { 'puppet infra recover_configuration':
        ensure => absent,
      }
    }
    enterprise_tasks::message('enable_ha_failover','Removing files, dirs and artifacts...')
    run_command('rm -rf /etc/puppetlabs/enterprise/*', $prev_primary)
    run_task('enterprise_tasks::drop_pglogical_databases', $prev_primary)

    apply($curr_primary) {
      file { ["/etc/puppetlabs/puppet/ssl/ca/signed/${prev_primary_certname}.pem",
              "/etc/puppetlabs/puppet/ssl/ca/requests/${prev_primary_certname}.pem",
              "/etc/puppetlabs/puppet/ssl/certificate_requests/${prev_primary_certname}.pem",
              "/etc/puppetlabs/puppet/ssl/certs/${prev_primary_certname}.pem",
              "/etc/puppetlabs/puppet/ssl/private_keys/${prev_primary_certname}.pem",
              "/etc/puppetlabs/puppet/ssl/public_keys/${prev_primary_certname}.pem",
              "/etc/puppetlabs/puppetserver/ca/signed/${prev_primary_certname}.pem",
              "/etc/puppetlabs/puppetserver/ca/requests/${prev_primary_certname}.pem"]:
        ensure => absent,
        force  => true,
      }
    }

    apply($prev_primary) {
      file { '/etc/puppetlabs/puppet/ssl':
        ensure  => absent,
        recurse => true,
        force   => true,
      }

      file { '/etc/puppetlabs/puppetserver/ca':
        ensure  => absent,
        recurse => true,
        force   => true,
      }
    }

    if $dns_alt_names {
      enterprise_tasks::message('enable_ha_failover', 'Setting dns_alt_names in puppet.conf')
      run_command("${constants()['puppet_bin']} config set --section main dns_alt_names \"${dns_alt_names}\"", $prev_primary)
    }
    # This puppet run will exit 1, but will generate a CSR in the process.
    # If autosign is enabled, will exit 2.
    run_task('enterprise_tasks::run_puppet', $prev_primary,
      alternate_host => $curr_primary_certname,
      exit_codes     => [1,2],
    )

    enterprise_tasks::message('enable_ha_failover','Signing cert...')
    run_task('enterprise_tasks::puppetserver_ca_sign', $curr_primary,
      host                    => $prev_primary_certname,
      allow_subject_alt_names => $allow_subject_alt_names,
    )
    run_task('enterprise_tasks::run_puppet', $prev_primary,
      alternate_host => $curr_primary_certname,
    )
    upload_file('/etc/puppetlabs/enterprise/hiera.yaml', '/etc/puppetlabs/enterprise/hiera.yaml', $prev_primary)
  }

  # Provision replica and running puppet through orchestrator requires the agent
  # to be enabled, so this is not included in the enterprise_tasks::with_agent_disabled block.
  # Should be safe to run this outside the block at this point.
  enterprise_tasks::message('enable_ha_failover','Provisioning old primary as new replica...')
  $provision_result = catch_errors() || {
    run_task('enterprise_tasks::provision_replica', $curr_primary,
      host                => $prev_primary_certname,
      replication_timeout => $replication_timeout_secs,
      streaming           => false,
    )
  }
  if $provision_result =~ Error {
    enterprise_tasks::message("${provision_result.details['object']}", "${provision_result.details['result_set'].first.error.details()}")
    fail_plan('ERROR: Failed to provision replica')
  }
  enterprise_tasks::message('enable_ha_failover','Enabling new replica...')
  $enable_result = catch_errors() || {
    run_task('enterprise_tasks::enable_replica', $curr_primary,
      host              => $prev_primary_certname,
      topology          => $topology,
      skip_agent_config => $skip_agent_config,
      agent_server_urls => $agent_server_urls,
      pcp_brokers       => $pcp_brokers,
    )
  }
  if $enable_result =~ Error {
    enterprise_tasks::message("${enable_result.details['object']}", "${enable_result.details['result_set'].first.error.details()}")
    fail_plan('ERROR: Failed to enable replica')
  }

  # If skipping agent config, there is no need to run Puppet everywhere
  unless ($skip_agent_config) {
    enterprise_tasks::message('enable_ha_failover',@("EOT"/L))
    ${host} has been enabled for your Puppet Enterprise
    infrastructure. Agent configuration is now managed with Puppet, and will
    be updated the next time each agent runs to enable replica failover.

    If you wish to immediately run Puppet on all your agents, you can do so
    with this command:

      puppet job run --no-enforce-environment --query 'nodes {deactivated is null and expired is null}'
    |-EOT
  }
}
