# Performs all the steps to install either a PE primary or database node
# once a PE tarball has been unpacked on the node and its pe-installer
# package has been installed.
#
# Assumes that the given target will be using localhost transport.
# This allows the apply to work with the Puppet provided by the pe-installer
# package. This is a requirement for initial install, since we have no other
# Puppet/Ruby to apply with, and is probably best for upgrade as well
# since we're switching the agent version.
#
# Gets additional version information from the tarball's bootstrap-metadata file.
#
# @param target
#   The node we are installing PE on; expected to be localhost,
#   or have localhost as the transport.
# @param pe_tarball_dir
#   Absolute path to the unpacked PE tarball directory.
# @param classes
#   A Hash of the pe_install class(es) and parameters to apply.
# @param pe_conf
#   Absolute path to the pe.conf file.
#   If not given, it's assumed to already be in place at
#   /etc/puppetlabs/enterprise/conf.d
# @param manage_permissions
#   Whether or not to manage permissions of files copied from the tarball.
#   Set true during initial install; false otherwise.
plan enterprise_tasks::testing::install_core_node(
  Variant[String,Target] $target,
  Enterprise_tasks::Absolute_path $pe_tarball_dir,
  Array[Tuple[String,Hash]] $classes,
  Boolean $manage_permissions = false,
  Optional[Enterprise_tasks::Absolute_path] $pe_conf = undef,
) {
  $_target = get_target($target)

  enterprise_tasks::message('install_core_node', 'Obtain core facts for bootstrap.')
  $_facts = run_task('enterprise_tasks::installer_facter', $_target).first().value()
  add_facts($_target, $_facts)

  enterprise_tasks::message('install_core_node', 'Obtain metadata from PE tarball.')
  $bootstrap_metadata = run_task(
    'enterprise_tasks::get_bootstrap_metadata',
    $_target,
    'pe_dir' => $pe_tarball_dir
  ).first().value()
  $effective_date = enterprise_tasks::first_defined($bootstrap_metadata['effective_date'], $bootstrap_metadata['auto_effective_date'])

  enterprise_tasks::message('install_core_node', 'Bootstrap package repository and install puppet-agent.')
  $bootstrap_result = apply($_target, '_catch_errors' => true) {
    class { 'pe_install::bootstrap':
      pe_tarball_dir      => $pe_tarball_dir,
      pe_build_version    => $bootstrap_metadata['pe_build_version'],
      platform_tag        => $bootstrap_metadata['platform_tag'],
      aio_agent_version   => $bootstrap_metadata['aio_agent_version'],
      aio_install_version => $bootstrap_metadata['aio_install_version'],
      effective_date      => $effective_date,
      manage_permissions  => $manage_permissions,
      pe_conf             => $pe_conf,
    }
  }.first()
  enterprise_tasks::apply_report('install_core_node', $bootstrap_result, true)

  # We're not using the Puppet in the puppet-agent installation, but probably still
  # want it disabled to keep it from interfering. Not an issue for install, but is
  # for upgrades.
  $install_result = enterprise_tasks::with_agent_disabled($_target) || {
    # The agent is installed, but we still want to gather custom facts.
    enterprise_tasks::message('install_core_node', 'Obtain custom facts for install.')
    apply_prep($_target)

    $class_names = $classes.map() |$i| { $i[0] }
    enterprise_tasks::message('install_core_node', "Apply installation classes: ${class_names}")
    debug("enterprise_tasks::testing::install_core_node applying ${classes} to ${_target}")
    apply($_target, '_catch_errors' => true) {
      $classes.each |$entry| {
        $class = $entry[0]
        $parameters = $entry[1]
        class { $class:
          * => $parameters,
        }
      }
    }.first()
  }
  enterprise_tasks::apply_report('install_core_node', $install_result, true)

  return {
    bootstrap_result => $bootstrap_result,
    install_result   => $install_result,
  }
}
