# Orchestrates an upgrade of all PE infrastructure.
#
# This includes the initial primary and primary/database upgrade, along with
# subsequent replica, complier, load balancers, razor and agent upgrades.
#
# TODO:
#  * compilers
#  * load balancers
#  * razor
#  * agents
#
# @param primary [Variant[Target,String]] the primary PE node.
# @param version [Optional[Enterprise_tasks::Pe_version_or_family]]
#   a version string indicating either the family line (2019.1, 2019.2, etc.)
#   or an exact version of PE to download and upgrade to. If just the family is
#   given, the latest development build from that line will be installed.
# @param tarball [Optional[Enterprise_tasks::Absolute_path]] alternately you may
#   supply an absolute path to a PE tarball on the localhost which you wish
#   installed.
# @param $use_tempdirs [Boolean] if true, generate proper tempdirs as workdirs
#   for uploading PE and pe.conf files. Otherwise use /root. Defaults to true.
#   Set to false for manual testing where you want the simplicity of using /root
#   as the workdir.
plan enterprise_tasks::testing::upgrade(
  Variant[TargetSpec,String] $primary,
  Optional[Enterprise_tasks::Pe_version_or_family] $version = undef,
  Optional[Enterprise_tasks::Absolute_path] $tarball = undef,
  Boolean $use_tempdirs = true,
) {

  enterprise_tasks::message('upgrade', 'Upgrading the primary PE infrastructure.')

  enterprise_tasks::message('upgrade', "Obtain latest user_data.conf information from ${primary}")
  run_task('enterprise_tasks::puppet_infra_recover_configuration', $primary)
  $primary_target = get_targets($primary)[0]
  enterprise_tasks::get_facts_on($primary_target)

  $current_pe_version = $primary_target.facts['pe_server_version']
  $enterprise_data_conf = (versioncmp($current_pe_version, '2018.1.3') == -1) ? {
    true    => '/etc/puppetlabs/enterprise/conf.d/pe.conf',
    default => '/etc/puppetlabs/enterprise/conf.d/user_data.conf',
  }
  $pe_keys = ['puppet_enterprise::ha_enabled_replicas']
  $user_data_hash = run_task(enterprise_tasks::get_conf_values,
    $primary,
    'file'          => $enterprise_data_conf,
    'keys'          => $pe_keys,
  ).first.value
  $replicas = $user_data_hash['puppet_enterprise::ha_enabled_replicas']

  # upgrade_pe
  run_plan('enterprise_tasks::testing::upgrade_pe',
    'primary'      => $primary,
    'version'      => $version,
    'tarball'      => $tarball,
    'pe_conf'      => $enterprise_data_conf,
    'use_tempdirs' => $use_tempdirs,
  )

  # upgrade_replica
  if !$replicas.empty() {
    run_plan('enterprise_tasks::testing::upgrade_replica',
      'primary'    => $primary,
      'replicas'   => $replicas,
    )
  }
}
