# Plan to assist in manul testing of upgrade workflows, first by installing a specific
# version, and then by upgrading to another version.
#
# At the moment this only handles monolithic and primary/database layouts of PE.
#
# If a single target is given in nodes, this will be the monolothic primary.
# If an array of two targets is given, the first will be the primary, and the second the database node.
#
# @param nodes [TargetSpec] the target nodes to install/upgrade PE on.
# @param upgrade_from [Enterprise_tasks::Pe_version_or_family] the
#   version or family of PE to install initially.
# @param upgrade_to_tarball [Optional[Enterprise_tasks::Absolute_path]] local
#   path to a PE tarball we will upgrade to.
# @param upgrade_to_version [Optional[Enterprise_tasks::Pe_version_or_family]] pe family
#   or version to upgrade from. Either this or upgrade_to_tarball must be
#   specified.
# @param update_pe_conf [Optional[Hash]] a hash of additional parameters to be
#   added to pe.conf *before* the upgrade begins. For example:
#     '{ "puppet_enterprise::postgres_version_override": "11" }'
#   could be set in order to test database migration from 9.6 -> 11 before lovejoy had
#   postgresql 11 as a default...
plan enterprise_tasks::testing::upgrade_workflow(
  TargetSpec $nodes,
  Enterprise_tasks::Pe_version_or_family $upgrade_from,
  Optional[Enterprise_tasks::Absolute_path] $upgrade_to_tarball = undef,
  Optional[Enterprise_tasks::Pe_version_or_family] $upgrade_to_version = undef,
  Optional[Hash] $update_pe_conf = {},
) {

  if ($upgrade_to_tarball == undef and $upgrade_to_version == undef) or ($upgrade_to_tarball != undef and $upgrade_to_version != undef) {
    fail_plan("You must specify either upgrade_to_version or upgrade_to_tarball, but not both. Received version: '${upgrade_to_version}' and tarball: '${upgrade_to_tarball}'")
  }

  $targets = get_targets($nodes)

  $primary = $targets[0]
  $database = case size($targets) {
    1: {
      $primary
    }
    2: {
      $targets[1]
    }
    default: {
      fail_plan("Expected at most two targets, a primary and a database host, received: ${targets}")
    }
  }

  $_upgrade_from_version = enterprise_tasks::lookup_pe_version($upgrade_from)

  run_plan('enterprise_tasks::testing::install_pe',
    'primary'                => $primary,
    'database'               => $database,
    'version'                => $_upgrade_from_version,
    'console_admin_password' => versioncmp($_upgrade_from_version, '2019.0') == -1 ? {
      true    => 'password', # upgrade_from is < 2019.0 so we need a password
      default => undef,
    },
    'use_tempdirs'           => false,
  )

  ##############################################
  # UPGRADE

  if !empty($update_pe_conf) {
    enterprise_tasks::message('upgrade', "Add the following additional parameters to pe.conf before upgrade: ${update_pe_conf}")
    run_task(enterprise_tasks::add_modify_conf_keys, $targets,
      'file'    => '/etc/puppetlabs/enterprise/conf.d/pe.conf',
      'hash'    => $update_pe_conf,
    )
  }

  if $upgrade_to_tarball != undef {
    run_plan('enterprise_tasks::testing::prep_test_package_gpg', 'nodes' => $targets)
  }

  run_plan('enterprise_tasks::testing::upgrade_pe',
    'primary'        => $primary,
    'tarball'        => $upgrade_to_tarball,
    'version'        => $upgrade_to_version,
    'use_tempdirs'   => false,
    'upgrading_from' => $_upgrade_from_version,
  )
}
