#!/opt/puppetlabs/puppet/bin/ruby
# frozen_string_literal: true

require_relative '../files/enterprise_task_helper.rb'
require 'open3'

class PgBaseBackup < EnterpriseTaskHelper
  attr_accessor :pe_server_dir, :pg_bin, :pg_home, :primary, :pg_version
  attr_reader :tablespaces, :backup_dir, :cert_dir

  PE_SERVER_DIR = '/opt/puppetlabs/server'
  def initialize
    @tablespaces = [
      'activity',
      'classifier',
      'inventory',
      'orchestrator',
      'puppetdb',
      'rbac',
    ]
    @backup_dir = 'pe-basebackup'
    @pe_server_dir = PE_SERVER_DIR
    @cert_dir = 'certs'
    @pg_bin = "#{pe_server_dir}/bin"
    @pg_home = "#{pe_server_dir}/data/postgresql"
    @primary
    @pg_version
  end

  def run_cmd(command, error, errcode)
    cmd = Array(command)
    stdout, stderr, status = Open3.capture3(*cmd)
    result = stdout.strip
    if !status.success?
      raise EnterpriseTaskHelper::Error.new(
        error,
        errcode.to_s,
        {
          command: cmd.join(' '),
          exitcode: status.exitstatus,
          stdout: stdout,
          stderr: stderr,
        },
      )
    end
    return result
  end

  def create_basebackup
    connection_string = [
      "host=#{primary}",
      'port=5432',
      "sslcert=#{pg_home}/#{cert_dir}/_local.cert.pem",
      "sslkey=#{pg_home}/#{cert_dir}/_local.private_key.pem",
      'sslrootcert=/etc/puppetlabs/puppet/ssl/certs/ca.pem',
      'user=pe-ha-replication',
      'sslmode=verify-full',
    ]
    backup_command = [
      "#{pg_bin}/pg_basebackup",
      "-D #{pg_home}/#{backup_dir}",
      "--dbname=\"#{connection_string.join(' ')}\"",
      '--checkpoint=fast',
      '--wal-method=stream',
    ].join(' ')
    su_command = ['su', '-s', '/bin/bash', 'pe-postgres', '-c']
    command = su_command << backup_command
    run_cmd(
      command,
      'pg_basebackup failed',
      'puppetlabs.pg-basebackup/basebackup-failed',
    )
  end

  def task(primary:, postgresql_version: 11, **_kwargs)
    @primary = primary
    @pg_version = postgresql_version

    run_cmd(
      '/opt/puppetlabs/bin/puppet resource service pe-postgresql ensure=stopped',
      'Failed to stop pe-postgresql service',
      'puppetlabs.pg-basebackup/stop-postgres-failed',
    )

    # remove existing tablespaces to make room for those copied by basebackup
    tablespaces.each do |ts|
      run_cmd(
        "rm -rf #{pg_home}/#{ts}",
        "Unable to remove #{ts} tablespace",
        'puppetlabs.pg-basebackup/remove-tablespace-failed',
      )
    end

    # Save replica's certs off to the side before removing PGDATA
    if Dir.exist?("#{pg_home}/#{pg_version}/data/certs")
      run_cmd(
        "rm -rf #{pg_home}/#{cert_dir} && mv #{pg_home}/#{pg_version}/data/certs #{pg_home}/#{cert_dir}",
        'Unable to grab the replica\'s pg cert dir',
        'puppetlabs.pg-basebackup/copy-replica-cert-dir-failed',
      )
    end

    # If the pg cert dir didn't exist above, we're rerunning this task to fix a failed
    # pg_basebackup. If we also don't have #{pg_home}/#{cert_dir} from that previous run,
    # check if we have #{pg_home}/#{backup_dir}/certs, which would be from after a
    # successful pg_basebackup but something went wrong restoring the data dir
    # and turning postgres back on.
    if !Dir.exist?("#{pg_home}/#{cert_dir}")
      if Dir.exist?("#{pg_home}/#{backup_dir}/certs")
        run_cmd(
          "mv #{pg_home}/#{backup_dir}/certs #{pg_home}/#{cert_dir}",
          'Unable to grab the replica\'s pg cert dir',
          'puppetlabs.pg-basebackup/copy-replica-cert-dir-failed',
        )
      else
        raise EnterpriseTaskHelper::Error.new(
          'Unable to find postgres certs directory. Please try running puppet on the replica to restore certs, then try this action again.',
          'puppetlabs.pg-basebackup/pg-certs-dir-missing',
          {
            paths: "Could not find certs at #{pg_home}/#{pg_version}/data/certs, #{pg_home}/#{cert_dir}, or #{pg_home}/#{backup_dir}/certs",
          },
        )
      end
    end

    # remove PGDATA before running pg_basebackup to leave PGHOME in a state that
    # puppet knows how to fix on the next puppet run in case the basebackup fails
    run_cmd(
      "rm -rf #{pg_home}/#{pg_version}/data/",
      'Unable to remove replica\'s original pg data dir',
      'puppetlabs.pg-basebackup/remove-replica-pg-data-dir-failed',
    )

    # Clean up backup dir before running pg_basebackup so that the task can be re-run
    # upon failure
    run_cmd(
      "rm -rf #{pg_home}/#{backup_dir}",
      'Unable to clean backup dir',
      'puppetlabs.pg-basebackup/remove-backup-dir-failed',
    )
    create_basebackup

    # replace the certs in the dir created by pg_basebackup with those saved
    # from the replica, this dir is a copy of the primary's PGDATA dir
    run_cmd(
      "rm -rf #{pg_home}/#{backup_dir}/#{cert_dir} && mv #{pg_home}/#{cert_dir} #{pg_home}/#{backup_dir}",
      'Unable to move certs into basebackup data dir',
      'puppetlabs.pg-basebackup/move-replica-certs-into-place',
    )

    run_cmd(
      "mv #{pg_home}/#{backup_dir} #{pg_home}/#{pg_version}/data/",
      'Unable to move basebackup data dir into place',
      'puppetlabs.pg-basebackup/move-backup-data-failed',
    )

    run_cmd(
      '/opt/puppetlabs/bin/puppet resource service pe-postgresql ensure=running',
      'Failed to start pe-postgresql service',
      'puppetlabs.pg-basebackup/start-postgres-failed',
    )

    {
      success: true,
    }.to_json
  end
end
PgBaseBackup.run if __FILE__ == $PROGRAM_NAME
