require 'puppet/http/client'
require 'json'

class Puppet::Node::HieraNodeAdapter < Puppet::Pops::Adaptable::Adapter
  attr_accessor :hiera_data
end

class Puppet::Node::Classifier < Puppet::Indirector::Code
  AgentSpecifiedEnvironment = "agent-specified"
  ClassificationConflict = 'classification-conflict'
  KEY_HIERA_DATA = 'config_data'

  def self.load_config
    config_path = File.join(Puppet[:confdir], 'classifier.yaml')

    config = nil
    if File.exists?(config_path)
      config = YAML.load_file(config_path)
    else
      Puppet.warning("Classifier config file '#{config_path}' does not exist, using defaults")
      config = {}
    end

    if config.respond_to?(:to_ary)
      config.map do |service|
        merge_defaults(service)
      end
    else
      service = merge_defaults(config)
      [service]
    end
  end

  def adapt_node_with_hiera_data(node, hiera_data)
    hiera_data ||= {}
    flattened_hiera_data = {}
    hiera_data.each do |scope, kv|
      kv.each do |k, v|
        flattened_hiera_data["#{scope}::#{k}"] = v
      end
    end

    Puppet::Node::HieraNodeAdapter.adapt(node).hiera_data = flattened_hiera_data
    return node
  end

  def find(request)
    name = request.key
    facts = if request.options[:facts].is_a?(Puppet::Node::Facts)
              request.options[:facts]
            else
              Puppet::Node::Facts.indirection.find(name, :environment => request.environment)
            end

    fact_values = if facts.nil?
                    {}
                  else
                    facts.sanitize
                    facts.to_data_hash['values']
                  end

    trusted_data = Puppet.lookup(:trusted_information) do
      # This block contains a default implementation for trusted
      # information. It should only get invoked if the node is local
      # (e.g. running puppet apply)
      temp_node = Puppet::Node.new(name)
      temp_node.parameters['clientcert'] = Puppet[:certname]
      Puppet::Context::TrustedInformation.local(temp_node)
    end

    trusted_data_values = if trusted_data.nil?
                            {}
                          else
                            trusted_data.to_h
                          end

    facts_for_request = {"fact" => fact_values,
             "trusted" => trusted_data_values}

    if request.options.include?(:transaction_uuid)
      facts_for_request["transaction_uuid"] = request.options[:transaction_uuid]
    end

    requested_environment = request.options[:configured_environment] || request.environment

    services.each do |service|
      result = retrieve_classification(name, facts_for_request, requested_environment, service)
      if result.is_a? Puppet::Node
        # Puppet 5.0 and later supports sending facts along with #fact_merge, to
        # avoid extra indirection calls
        if result.method(:fact_merge).arity.zero?
          result.fact_merge
        else
          result.fact_merge(facts)
        end
        return result
      elsif result.respond_to?(:[]) and result['kind'] == ClassificationConflict
        # got a classification conflict
        msg = result['msg']
        Puppet.err(msg)
        raise Puppet::Error, msg
      end
    end

    # got neither a valid classification nor a classification conflict, so all the services are
    # unreachable or having unforeseen problems
    msg = "Classification of #{name} failed due to a Node Manager service error. Please check /var/log/puppetlabs/console-services/console-services.log on the node(s) running the Node Manager service for more details."
    Puppet.err(msg)
    raise Puppet::Error, msg
  end

  private

  # Attempt to retrieve classification from the NC service. Returns a
  # Puppet::Node object with the retrieved classification if successfully
  # retrieved, the parsed conflict error response in the case of a
  # classification conflict, otherwise nil if the case of a connection error,
  # timeout, or non-conflict 5xx response.
  def retrieve_classification(node_name, node_facts, requested_environment, service)
    request_path = "#{normalize_prefix(service[:prefix])}/v2/classified/nodes/#{node_name}"
    url = URI(Puppet::Util.uri_encode("https://#{service[:server]}:#{service[:port]}#{request_path}"))
    begin
      client = Puppet.runtime[:http]

      response = client.post(url, node_facts.to_json,
                             headers: {'Content-Type' => 'application/json'},
                             options: {:metric_id => [:classifier, :nodes]})
    rescue Puppet::HTTP::HTTPError, Puppet::SSL::SSLError => e
      Puppet.warning("Could not connect to the Node Manager service at #{url}: #{e.inspect}")
      return nil
    end

    result = JSON.parse(response.body)

    unless response.success?
      if result['kind'] == ClassificationConflict
        explanation = result['msg'].sub(" See the `details` key for all conflicts.", "")
        msg = "Classification of #{node_name} failed due to a classification conflict: #{explanation}"
        return result.merge({'msg' => msg})
      else
        details = result['msg']
        Puppet.warning("Received an unexpected error response from the Node Manager service at #{response.url}: #{response.code} #{response.reason}: #{details}")
        return nil
      end
    end

    result['classes'] = Hash[result['classes'].sort]
    is_agent_specified = (result["environment"] == AgentSpecifiedEnvironment)
    result.delete("environment") if is_agent_specified
    hiera_data = result.delete(KEY_HIERA_DATA)

    node = Puppet::Node.from_data_hash(result)
    adapt_node_with_hiera_data(node, hiera_data)
    node.environment = requested_environment if is_agent_specified

    node.merge({"pe_node_groups" => result['groups']})

    return node
  end

  def config
    @config ||= Puppet::Node::Classifier.load_config
  end

  def services
    config
  end

  def normalize_prefix(prefix)
    prefix.chomp('/')
  end

  def self.merge_defaults(service)
    {
      :server => service["server"] || 'classifier',
      :port => service["port"] || 1262,
      :prefix => service["prefix"] || '',
    }
  end

end
