# This Bolt plan is run over the orchestrator. It will take a list of
# non-infrastructure agent nodes and regenerate the cert of those nodes.
#
# Be careful when using a node_type other than agent, as other considerations
# may need to be made when regenerating a certificate on an infrastructure node,
# such as restarting services.
#
# $agent may be multiple nodes, as long as they are all the same type with the
# same parameters
#
# Only agents installed under the root user are supported at this time
# Steps the plan will take:
# 1. If dns_alt_names is defined as a parameter, verify that puppetserver conf files in
#    /etc/puppetlabs/puppetserver/conf.d have certificate-authority.allow-subject-alt-names
#    set to true. This should be managed to true by default. If dns_alt_names is not passed
#    in as a parameter, make sure it is not set in puppet.conf as well or fail otherwise.
# 2. Disable the agent on the primary and agent nodes with puppet agent --disable.
# 3. Stop the puppet service on the agent node.
# 4. Backup existing certs in /etc/puppetlabs/puppet/ssl or C:/ProgramData/PuppetLabs/puppet/etc/ssl.
#    They are backed up to a directory in the same location with a _bak_<timestamp> suffix.
# 5. If extension_requests or custom_attributes is defined, these are set in in the
#    csr_attributes.yaml file, located at /etc/puppetlabs/puppet or C:/ProgramData/PuppetLabs/puppet/etc.
# 6. Run puppet ssl clean <certname> on the agent node. --localca is specified if
#    the clean_crl parameter is set to true.
# 7. If dns_alt_names is defined, run puppet config set --section main
#    dns_alt_names “<dns_alt_names value>” on the agent node.
# 8. Run puppetserver ca clean --certname=<node> on the primary.
# 9. Run 'puppet ssl submit_request' on the agent node to generate a CSR and submit it to the primary
# 10. If the cert is not autosigned, run puppetserver ca sign --certname <node>
#     on the primary.  Before we do this, we check the CSR with puppetserver ca
#     list --certname <node>.  If dns_alt_names was not specified, we verify
#     that no subject alt names appear in the CSR.
# 11. Run 'puppet ssl download_cert' on the agent node to download signed certificate from the primary
# 12. Run puppet on the agent node.
# 13. If manage_pxp_service is set to true (default), restart the pxp-agent
#     service on the agent node.
# 14. Restore the puppet service to its state before the plan started.
# 15. Re-enable the agent on the primary and agent nodes with puppet agent --enable.
plan enterprise_tasks::agent_cert_regen(
  TargetSpec $agent,
  Optional[TargetSpec] $primary         = 'localhost',
  Optional[Hash] $extension_requests    = undef,
  Optional[Hash] $custom_attributes     = undef,
  Optional[String] $node_type           = 'agent',
  Optional[Boolean] $manage_pxp_service = true,
  Optional[String] $dns_alt_names       = undef,
  Optional[Boolean] $clean_crl          = false,
  Optional[Boolean] $force              = false,
) {
  enterprise_tasks::test_connection([$primary, $agent])
  enterprise_tasks::verify_node($primary, 'primary', $force)
  enterprise_tasks::verify_node($agent, $node_type, $force)

  if $dns_alt_names{
    run_plan(enterprise_tasks::is_subject_alt_names_allowed, primary => $primary, force  => $force)
  }
  $allow_subject_alt_names = $dns_alt_names ? {
    undef   => false,
    default => true
  }

  enterprise_tasks::with_agent_disabled([$primary]) || {
    $errors = get_targets($agent).reduce([]) |$memo, $node| {
      # If the puppet service is running, we need to ensure it gets restarted
      # so it picks up the new cert.
      $status_hash = run_plan(enterprise_tasks::get_service_status, target => $agent,
        service => 'puppet',
      )
      run_task(enterprise_tasks::pe_services, $agent, state => 'stopped') # Stops only 'puppet' since we pass in no role

      $result_or_error = catch_errors() || {
        enterprise_tasks::with_agent_disabled([$node]) || {
          $node_target = get_target($node)
          enterprise_tasks::set_feature($node_target, 'puppet-agent', true)
          $node_facts = run_plan('facts', $node_target, '_catch_errors' => true)
          if $node_facts =~ Error {
            if $node_facts.details()['result_set'].first.value['os']['name'] {
              $osname = $node_facts.details()['result_set'].first.value['os']['name']
            } else {
              enterprise_tasks::message('agent_cert_regen', "${node_facts.kind()}:${node_facts.msg()}")
              fail_plan('agent_cert_regen', "Failed to retrieve os facts from ${node}.")
            }
          } else {
            $osname = $node_facts.first.value['os']['name']
          }

          # Facter plan does not return env_windows_installdir; we must call facter directly to obtain it
          # and use the default bin path if we are still unable to access the fact
          if $osname == 'windows' {
            # On Windows, spaces in the path to executable need to be escaped with a backtick in run_command
            $install_dir = run_command('facter.bat env_windows_installdir', $node_target, '_catch_errors' => true).first['stdout'].strip.split(' ').join('` ')
            if $node_facts =~ Error {
              $puppet_bin = 'C:\Program` Files\Puppet` Labs\Puppet\bin\puppet.bat'
            } else {
                $puppet_bin = "${install_dir}\\bin\\puppet.bat"
            }
          } else {
            $puppet_bin = constants()['puppet_bin']
          }

          # Fail early if puppet.conf has a dns_alt_names setting and agent_cert_regen is called without
          # dns_alt_names parameter. Otherwise CSR will be generated with dns_alt_names and signing will
          # fail later
          $orig_dns_alt_names = run_command("${puppet_bin} config print dns_alt_names --section main", $node).first['stdout'].strip()
          if !empty($orig_dns_alt_names) and !$dns_alt_names {
            enterprise_tasks::message('agent_cert_regen','ERROR: Requires a value for dns_alt_names parameter' )
            enterprise_tasks::message('agent_cert_regen',"dns_alt_names is set in puppet.conf on ${node.host()} but the plan was called without a value for dns_alt_names parameter")
            fail_plan("Failed on ${node.host()}", 'Requires a value for dns_alt_names parameter')
          }
          $backups = run_task(enterprise_tasks::backup_certs, $node).first().value()['backups']
          enterprise_tasks::message('agent_cert_regen',"Certificate backups saved to ${backups}")

          if $extension_requests or $custom_attributes {
            run_task(enterprise_tasks::set_csr_attributes, $node,
              extension_requests => $extension_requests,
              custom_attributes  => $custom_attributes,
            )
          }
          enterprise_tasks::message('agent_cert_regen',"Cleaning SSL directory on ${node.host}...")
          run_task(enterprise_tasks::puppet_ssl_clean, $node,
            certname => $node.host,
            localca  => $clean_crl,
          )

          if $dns_alt_names {
            enterprise_tasks::message('agent_cert_regen', "Setting dns_alt_names in puppet.conf on node ${node.host}")
            run_command("${puppet_bin} config set --section main dns_alt_names \"${dns_alt_names}\"", $node,)
          }

          enterprise_tasks::message('agent_cert_regen',"Cleaning certs for ${node.host}")
          run_task(enterprise_tasks::puppetserver_ca_clean, $primary,
            host    => $node.host,
          )

          # After cleaning the certs,  make sure to use run_command since run_task may fail when downloading the task file
          # to the agent node. When downloading task file to the node, pxp-agent will try to load the certs if it is not
          # already cached and will fail since certs are not available on the agent node at this point.
          enterprise_tasks::message('agent_cert_regen',"Generating CSR on ${node.host} and submitting")
          run_command("${puppet_bin} ssl submit_request", $node,)

          enterprise_tasks::message('agent_cert_regen',"Signing certs for ${node.host}...")
          run_task(enterprise_tasks::puppetserver_ca_sign, $primary,
            host                    => $node.host,
            allow_subject_alt_names => $allow_subject_alt_names,
          )

          enterprise_tasks::message('agent_cert_regen',"Downloading certs for ${node.host}")
          run_command("${puppet_bin} ssl download_cert", $node,)

          # Wait and retry to account for potential SSL lockfile conflicts
          run_task(enterprise_tasks::run_puppet, $node,
            max_timeout => 256,
          )

          if $manage_pxp_service {
            enterprise_tasks::message('agent_cert_regen','Restarting pxp-agent...')
            enterprise_tasks::set_feature($node, 'puppet-agent', true)
            run_task(service, $node,
              action        => 'restart',
              name          => 'pxp-agent',
              _catch_errors => true,
            )
            wait_until_available($node, wait_time => 30)
          }
        }
      }

      if $status_hash[status] == 'running' {
        enterprise_tasks::message('agent_cert_regen', 'Restarting puppet service...')
        run_task(enterprise_tasks::pe_services, $agent, state => 'running')
      }

      if $result_or_error =~ Error {
        enterprise_tasks::message('agent_cert_regen', "ERROR: Failed to regenerate agent certificate on node ${node.host}")
        enterprise_tasks::message('agent_cert_regen', "${result_or_error.kind()}:${result_or_error.msg()}")
        $results = $result_or_error.details()['result_set']
        if $results !~ Undef {
          $results.each |$r| {
            $e = $r.error()
            enterprise_tasks::message('agent_cert_regen', "${e.kind()} ${e.msg()}: ${e.details()}")
          }
        } else {
          enterprise_tasks::message('agent_cert_regen', "${result_or_error.details()}")
        }
        $memo + [$node.host]
      } else { $memo }
    }
    if $errors.length > 0 {
      fail_plan("Error regenerating agent certificates on ${errors.length} node(s).", 'agent_cert_regen/failure', { 'nodes' => $errors })
    }
  }
}
