#!/opt/puppetlabs/puppet/bin/ruby
# frozen_string_literal: true

require_relative '../files/enterprise_task_helper.rb'
require 'json'
require 'open3'

class PuppetserverCASign < EnterpriseTaskHelper

  # Provides information about a particular certificate, given the
  # output of `puppetserver cert list --certname <certname>`
  class CertStatus
    attr_accessor :certname, :alt_names, :signed, :requested

    def initialize(ca_output)
      header, certline = ca_output.split("\n")
      case header
      when 'Signed Certificates:'
        self.signed = true
      when 'Requested Certificates:'
        self.requested = true
      end
      contents = certline.nil? ?
        [] :
        certline.strip.scan(/^[^\s]+|(?<=DNS:)[^"]+/)
      self.certname = contents.first
      self.alt_names = contents[1..-1] || []
    end

    # Eliminate nils
    def signed?; !!signed end
    def requested?; !!requested end # rubocop:disable Layout/EmptyLineBetweenDefs

    def found?
      signed? || requested?
    end
  end

  def task(host:, allow_subject_alt_names: false, signing_timeout: 2, dns_alt_names: nil, **_kwargs)
    # Right now, puppetserver ca does not support passing a flag for allowing
    # SAN or auth extensions for a one-off cert signing. We can detect if a CSR
    # contains alt names in the request by looking at the output of
    # puppetserver ca list. However, auth extensions are not listed here.  So
    # currently, allow_authorized_extensions is an unused argument.
    #
    # Note that if autosigning is enabled, this task will not be able to catch
    # a CSR that contains alt names when allow_subject_alt_names == false

    signed = false
    ca_cmd = '/opt/puppetlabs/server/bin/puppetserver'

    certname_cmd = ['/opt/puppetlabs/bin/puppet', 'config', 'print', 'certname']
    caserver = Open3.capture2e(*certname_cmd)[0].strip

    cert = CertStatus.new('')
    command = []
    output = nil
    status = nil
    list_cmd = [ca_cmd, 'ca', 'list', '--certname', host]
    5.times do
      output, _stderr, status = Open3.capture3(*list_cmd)
      cert = CertStatus.new(output)
      break if cert.found?
      sleep signing_timeout
      signing_timeout *= 2
    end

    if !cert.found?
      raise EnterpriseTaskHelper::Error.new(
        "No CSR detected for host #{host}",
        'puppetlabs.puppetserver-ca-sign/no-csr',
        {
          command: command.join("\n"),
          output: output,
          exitcode: status.exitstatus,
        },
      )
    end

    raise EnterpriseTaskHelper::Error.new("CSR contained the following subject alt names and allow_subject_alt_names == false: #{cert.alt_names.join(', ')}", 'puppetlabs.puppetserver-ca-sign/alt-names-disallowed') if cert.requested? && !cert.alt_names.empty? && !allow_subject_alt_names

    command = [ca_cmd, 'ca', 'sign', '--certname', host]
    output, _stderr, status = Open3.capture3(*command) if cert.requested?
    signed = true if cert.signed? || (status && status.success?)
    raise EnterpriseTaskHelper::Error.new("Could not sign request for host with certname #{host} using caserver #{caserver}", 'puppetlabs.puppetserver-ca-sign/sign-cert-failed', 'output' => output) if !signed
    list_output, _list_stderr, _list_status = Open3.capture3(*list_cmd)
    if !dns_alt_names.nil? && !dns_alt_names.empty?
      dns_alt_names.split(',').each do |name|
        raise EnterpriseTaskHelper::Error.new("Certs for #{host} does not contain the dns_alt_names \'#{name}\'", 'puppetlabs.puppetserver-ca-sign/dns-alt-names-checking-failed', 'output' => list_output) unless list_output.include?("DNS:#{name}")
      end
    end

    result = { signed: signed }
    result.to_json
  end
end

PuppetserverCASign.run if __FILE__ == $PROGRAM_NAME
