# Profile for configuring the pe-webserver to proxy requests for the console service.
#
#
# @param certname [String] The certname the console will use to encrypt network traffic.
# @param dhparam_file [String] The absolute path to the DH Param file to be used with the proxy.
# @param ipv6_only [Boolean] If enabled, the proxy will only allow IPv6 connections.
# @param trapperkeeper_proxy_listen_address [String] The network interface used by the console-ui app
# @param trapperkeeper_proxy_listen_port [Integer] The port for console-ui
# @param ssl_listen_address [String] The network interface used for ssl connections.
# @param ssl_listen_port [Integer] The port used for ssl connections.
# @param browser_ssl_cert [String] Sets the path to the server certificate PEM file used
#        by the web service.
# @param browser_ssl_cert_ttl [Integer] This parameter is for setting the time to live for the console's
#        certificate in days. Defaults to 824 days
# @param Variant[String, Array[String]] ssl_protocols The list of SSL protocols to allow.
# @param browser_ssl_private_key [String] For use with a custom CA, the path to a private
#        key for your public ca certificate.
# @param nginx_gzip [String] Set gzip compression in nginx to on or off
class puppet_enterprise::profile::console::proxy (
  $certname                                 = $facts['clientcert'],
  String $server_name                       = $puppet_enterprise::console_host,
  $dhparam_file                             = '/etc/puppetlabs/nginx/dhparam_puppetproxy.pem',
  Variant[Integer[4,4], Integer[6,6]] $ip_version = $puppet_enterprise::params::ip_version,
  Boolean $ipv6_only                        = $puppet_enterprise::params::ipv6_only,
  $proxy_read_timeout                       = 120,
  $trapperkeeper_proxy_listen_address       = $puppet_enterprise::params::plaintext_address_url_safe,
  $trapperkeeper_proxy_listen_port          = $puppet_enterprise::params::console_services_listen_port,
  $ssl_ciphers                              = $puppet_enterprise::ssl_cipher_suites_browser,
  $ssl_listen_address                       = $puppet_enterprise::params::ssl_address_url_safe,
  $ssl_listen_port                          = $puppet_enterprise::console_port,
  $ssl_prefer_server_ciphers                = 'on',
  Variant[String, Array[String]] $ssl_protocols = $puppet_enterprise::ssl_protocols,
  $browser_ssl_cert_ttl                     = 824,
  $ssl_session_cache                        = 'shared:SSL:50m',
  $ssl_session_timeout                      = '1d',
  $ssl_verify_client                        = 'off',
  $ssl_verify_depth                         = 1,
  Optional[String] $browser_ssl_cert        = undef,
  Optional[String] $browser_ssl_private_key = undef,
  Enum['on','off'] $nginx_gzip              = 'on',
  Puppet_enterprise::Replication_mode $replication_mode = 'none',
  Optional[String] $saml_listen_address     = $puppet_enterprise::params::ssl_address_url_safe,
  Optional[String] $saml_listen_port        = $puppet_enterprise::params::console_services_ssl_listen_port,
) inherits puppet_enterprise {

  # Don't enable or configure the pe-nginx service on replicas
  if $replication_mode == 'replica' {
    class {'pe_nginx':
      ensure => stopped,
      enable => false,
    }
  } else {
    include pe_nginx

    pe_validate_re($ssl_verify_client, '^(on|off|optional|optional_no_ca)$')
    pe_validate_re($ssl_prefer_server_ciphers, '^(on|off)$')
    pe_validate_re($ssl_session_cache, '(^(off|none)$|(builtin:[0-9]+)|(shared:[a-zA-Z]+:[0-9]+[bkmg]?))')
    pe_validate_single_integer($proxy_read_timeout)

    if $browser_ssl_cert != undef and $browser_ssl_private_key == undef {
      fail('browser_ssl_private_key must also be set if setting browser_ssl_cert')
    }
    elsif $browser_ssl_cert == undef and $browser_ssl_private_key != undef {
      fail('browser_ssl_cert must also be set if setting browser_ssl_private_key')
    }

    $_ipv6_only = $ipv6_only ? {
      true => 'on',
      false => 'off'
    }

    $console_cert_dir = $puppet_enterprise::console_services_ssl_dir

    $puppetserver_cmd = "${puppet_enterprise::puppetlabs_bin_dir}/puppetserver"

    exec { 'generate console cert':
      command => "${puppetserver_cmd} ca generate --certname console-cert --ttl ${browser_ssl_cert_ttl}d --subject-alt-names ${puppet_enterprise::console_host}",
      creates => '/etc/puppetlabs/puppet/ssl/certs/console-cert.pem',
      require => Service['pe-puppetserver'],
      before  => Service['pe-nginx'],
    }

    file { "${console_cert_dir}/console-cert.cert.pem":
      ensure    => present,
      source    => "${puppet_enterprise::params::ssl_dir}/certs/console-cert.pem",
      mode      => '0400',
      show_diff => false,
      require   => Exec['generate console cert'],
      notify    => [ Service['pe-console-services'], Service['pe-nginx'] ],
    }

    file { "${console_cert_dir}/console-cert.private_key.pem":
      ensure    => present,
      source    => "${puppet_enterprise::params::ssl_dir}/private_keys/console-cert.pem",
      mode      => '0400',
      show_diff => false,
      require   => Exec['generate console cert'],
      notify    => [ Service['pe-console-services'], Service['pe-nginx'] ],
    }

    $_browser_ssl_cert = $browser_ssl_cert ? {
      undef   => "${console_cert_dir}/${certname}.cert.pem",
      default => $browser_ssl_cert,
    }

    $_browser_ssl_private_key = $browser_ssl_private_key ? {
      undef   => "${console_cert_dir}/${certname}.private_key.pem",
      default => $browser_ssl_private_key,
    }

    $puppetproxy_file = "${puppet_enterprise::nginx_conf_dir}/conf.d/proxy.conf"

    file { $puppetproxy_file:
      ensure  => file,
      owner   => $puppet_enterprise::params::root_user,
      group   => $puppet_enterprise::params::root_group,
      mode    => '0644',
      notify  => Service['pe-nginx'],
      require => Puppet_enterprise::Certs['pe-console-services::server_cert'],
    }

    $default_dhparam_file = '/etc/puppetlabs/nginx/dhparam_puppetproxy.pem'
    file { $default_dhparam_file:
      ensure => file,
      source => 'puppet:///modules/puppet_enterprise/console/dhparam_puppetproxy.pem',
      owner  => $puppet_enterprise::params::root_user,
      group  => $puppet_enterprise::params::root_group,
      mode   => '0644',
      notify => Service['pe-nginx'],
    }

    class { 'puppet_enterprise::profile::console::proxy::nginx_conf' :
      gzip => $nginx_gzip,
    }

    Pe_nginx::Directive {
      directive_ensure => 'present',
      target           => $puppetproxy_file,
      server_context   => $server_name,
    }

    pe_nginx::directive { 'server_name':
      value => $server_name,
    }

    pe_nginx::directive { 'listen':
      value   => "${ssl_listen_address}:${ssl_listen_port} ssl ipv6only=${_ipv6_only}",
    }

    # --BEGIN SSL CONFIG--
    pe_nginx::directive { 'ssl_certificate':
      value => $_browser_ssl_cert,
    }

    pe_nginx::directive { 'ssl_certificate_key':
      value => $_browser_ssl_private_key,
    }

    pe_nginx::directive { 'ssl_crl':
      value => $puppet_enterprise::params::hostcrl,
    }

    pe_nginx::directive { 'ssl_prefer_server_ciphers':
      value => $ssl_prefer_server_ciphers,
    }

    $ssl_protocols_array = type($ssl_protocols) =~ Type[String] ? {
      true => $ssl_protocols.split(' '),
      false => $ssl_protocols,
    }
    if empty($ssl_ciphers) {
      $ciphers = puppet_enterprise::ciphers($ssl_protocols_array.map |$p| { "browser_${p}" }, 'openssl')
    } else {
      $ciphers = type($ssl_ciphers) =~ Type[String] ? {
        true => $ssl_ciphers.split(':'),
        false => $ssl_ciphers,
      }
    }

    # TLSv1.3 OpenSSL ciphers all start with TLS_ and none do pre-1.3,
    # so we use that as a filter here.
    $ciphers_1_3 = $ciphers.filter |$item| { $item =~ /^TLS_/ }
    $ciphers_1_2_or_less = $ciphers - $ciphers_1_3

    if empty($ciphers_1_2_or_less) {
      $ssl_ciphers_ensure = 'absent'
      $ssl_ciphers_value = undef
    } else {
      $ssl_ciphers_ensure = 'present'
      $ssl_ciphers_value = pe_join($ciphers_1_2_or_less, ':')
    }

    pe_nginx::directive { 'ssl_ciphers':
      directive_ensure => $ssl_ciphers_ensure,
      value            => $ssl_ciphers_value,
    }

    if empty($ciphers_1_3) {
      $ssl_conf_command_ensure = 'absent'
      $ssl_conf_command_value = undef
    } else {
      $ssl_conf_command_ensure = 'present'
      $ssl_conf_command_value = "Ciphersuites ${pe_join($ciphers_1_3, ':')}"
    }

    pe_nginx::directive { 'ssl_conf_command':
      directive_ensure => $ssl_conf_command_ensure,
      value            => $ssl_conf_command_value,
    }

    pe_nginx::directive { 'ssl_protocols':
      value => pe_join($ssl_protocols_array, ' '),
    }

    pe_nginx::directive { 'ssl_dhparam':
      value => $dhparam_file,
    }

    pe_nginx::directive { 'ssl_verify_client':
      value => $ssl_verify_client,
    }

    pe_nginx::directive { 'ssl_verify_depth':
      value => $ssl_verify_depth,
    }

    pe_nginx::directive { 'ssl_session_timeout':
      value => $ssl_session_timeout,
    }

    pe_nginx::directive { 'ssl_session_cache':
      value => $ssl_session_cache,
    }
    # --END SSL CONFIG--

    # --BEGIN LOCATION "/" and "/saml" CONFIG--
    pe_nginx::directive { 'default proxy_pass':
      directive_name   => 'proxy_pass',
      value            => "http://${trapperkeeper_proxy_listen_address}:${trapperkeeper_proxy_listen_port}",
      location_context => '/',
    }

    pe_nginx::directive { 'saml proxy_pass':
      directive_name   => 'proxy_pass',
      value            => "https://${saml_listen_address}:${saml_listen_port}",
      location_context => '/saml',
    }

    pe_nginx::directive { 'default proxy_redirect':
      directive_name   => 'proxy_redirect',
      value            => "http://${trapperkeeper_proxy_listen_address}:${trapperkeeper_proxy_listen_port} /",
      location_context => '/',
    }

    pe_nginx::directive { 'saml proxy_redirect':
      directive_name   => 'proxy_redirect',
      value            => "https://${saml_listen_address}:${saml_listen_port} /",
      location_context => '/saml',
    }

    pe_nginx::directive { 'default proxy_read_timeout':
      directive_name   => 'proxy_read_timeout',
      value            => $proxy_read_timeout,
      location_context => '/',
    }

    pe_nginx::directive { 'saml proxy_read_timeout':
      directive_name   => 'proxy_read_timeout',
      value            => $proxy_read_timeout,
      location_context => '/saml',
    }

    pe_nginx::directive { 'default proxy_set_header x-ssl-subject':
      directive_name   => 'proxy_set_header',
      value            => 'X-SSL-Subject $ssl_client_s_dn',
      location_context => '/',
      replace_value    => false,
    }

    pe_nginx::directive { 'saml proxy_set_header x-ssl-subject':
      directive_name   => 'proxy_set_header',
      value            => 'X-SSL-Subject $ssl_client_s_dn',
      location_context => '/saml',
      replace_value    => false,
    }

    pe_nginx::directive { 'default proxy_set_header x-client-dn':
      directive_name   => 'proxy_set_header',
      value            => 'X-Client-DN $ssl_client_s_dn',
      location_context => '/',
      replace_value    => false,
    }

    pe_nginx::directive { 'saml proxy_set_header x-client-dn':
      directive_name   => 'proxy_set_header',
      value            => 'X-Client-DN $ssl_client_s_dn',
      location_context => '/saml',
      replace_value    => false,
    }

    pe_nginx::directive { 'default proxy_set_header x-client-verify':
      directive_name   => 'proxy_set_header',
      value            => 'X-Client-Verify $ssl_client_verify',
      location_context => '/',
      replace_value    => false,
    }

    pe_nginx::directive { 'saml proxy_set_header x-client-verify':
      directive_name   => 'proxy_set_header',
      value            => 'X-Client-Verify $ssl_client_verify',
      location_context => '/saml',
      replace_value    => false,
    }

    pe_nginx::directive { 'default proxy_set_header x-forwarded-for':
      directive_name   => 'proxy_set_header',
      value            => 'X-Forwarded-For $proxy_add_x_forwarded_for',
      location_context => '/',
      replace_value    => false,
    }

    pe_nginx::directive { 'saml proxy_set_header x-forwarded-for':
      directive_name   => 'proxy_set_header',
      value            => 'X-Forwarded-For $proxy_add_x_forwarded_for',
      location_context => '/saml',
      replace_value    => false,
    }

    # used to allow the host/port that appears in SAML requests to have
    # the public name/port
    pe_nginx::directive { 'saml proxy_set_header host':
      directive_name   => 'proxy_set_header',
      value            => 'Host $host',
      location_context => '/saml',
      replace_value    => false,
    }

    # --END LOCATION "/" and SAML CONFIG--

    class { 'puppet_enterprise::profile::console::proxy::http_redirect':
      ssl_listen_port => Integer($ssl_listen_port),
      replication_mode => $replication_mode,
    }
  }
}
