# DO NOT INCLUDE OR DECLARE THIS CLASS. This class is private and intended to
# be declared by puppet_enterprise::repo.
#
# Note: The puppet_enterprise_repo_update tag is used by the upgrade_secondary
# plan to ensure repos are updated on secondary nodes before any package
# resources are applied.
#
# @param pe_ver [String] The PE version string that will be part of the
#   package repository url.
# @param source [String] The server.cert:port or /path/to/packages for either the
#   remote or local repository configuration.
# @param local_config [Boolean] set true for a local configuration.
# @param platform_tag [String] the platform_tag fact.
# @param certname [Optional[String]] Certname of the node where the puppet_enterprise
#   repo is being managed. Should not need to be passed in under most circumstances.
# @param manage [Optional[Boolean]] *Deprecated*
# @param master [Optional[String]] *Deprecated*
class puppet_enterprise::repo::config(
  String $pe_ver,
  String $source,
  Boolean $local_config = false,
  String $platform_tag = $facts['platform_tag'],
  Optional[String] $certname = $facts['clientcert'],
  Optional[Boolean] $manage = undef, # deprecated
  Optional[String] $master = undef, # deprecated
) {
  puppet_enterprise::deprecated_parameter { 'puppet_enterprise::repo::config::manage': }
  puppet_enterprise::deprecated_parameter { 'puppet_enterprise::repo::config::master': }

  # This defaults to the clientcert fact (not so much a real fact, but injected by the agent
  # during fact collection with the value of Puppet[:certname]). In some situations, we are
  # unable to get this fact (e.g. applying this class via a Bolt plan for replica upgrades
  # on RHEL8 for some reason.). At some point, we should determine if we can default to using
  # the trusted fact instead and have it work in all scenarios where this class is applied.
  $_certname = $certname ? {
    undef   => $trusted['certname'],
    default => $certname,
  }
  # On primary, the repo should point to the local directory
  if $local_config {
    if $source !~ Pattern[/^\/.+/] {
      fail("Expected an absolute path for local repository configuration, got '${source}'")
    }
    $base_source = "file://${source}"
    $ssl_options = {}
  } else {
    # Variables for configuring the repos to be able to communicate
    # with the puppetserver using puppets PKI
    $ssl_dir = '/etc/puppetlabs/puppet/ssl'
    $ssl_options = {
      sslcacert     => "${ssl_dir}/certs/ca.pem",
      sslclientcert => "${ssl_dir}/certs/${_certname}.pem",
      sslclientkey  => "${ssl_dir}/private_keys/${_certname}.pem",
    }
    $elements = split($source, ':')
    $_master = $elements[0]
    $_port = $elements[1]
    if $elements !~ Array[String, 2, 2] or $_port !~ Pattern[/^\d+$/]  {
      fail("Expected a remote source server:port for repository configuration, got '${source}'")
    }
    $base_source = "https://${source}/packages"
  }

  $repo_source = "${base_source}/${pe_ver}/puppet_enterprise"

  $gpg_keys = [
    "${base_source}/GPG-KEY-puppet",
    "${base_source}/GPG-KEY-puppet-2025-04-06",
  ]

  $repo_name = 'puppet_enterprise'

  case $platform_tag {
    /^(el|redhatfips)-\d+-x86_64$/: {
      #Remove legacy repo files
      file { '/etc/yum.repos.d/pe_repo.repo':
        ensure => absent,
      }
      file { '/etc/yum.repos.d/pc_repo.repo':
        ensure => absent,
      }

      $yum_proxy_setting = $facts['os']['release']['major'] ? {
        '8'           => '',
        default       => '_none_',
      }

      yumrepo { $repo_name:
        ensure   => present,
        baseurl  => $repo_source,
        descr    => 'Puppet, Inc. PE Packages $releasever - $basearch', # don't want to interpolate those - they are yum macros
        enabled  => true,
        gpgcheck => '1',
        gpgkey   => pe_join($gpg_keys, "\n\t"),
        proxy    => $yum_proxy_setting,
        notify   => Exec['pe_yum_update'],
        tag           => 'puppet_enterprise_repo_update',
        *        => $ssl_options,
      }

      exec { 'pe_yum_update':
        command     => "/usr/bin/yum clean all --disablerepo='*' --enablerepo=${repo_name}",
        logoutput   => 'on_failure',
        refreshonly => true,
        tag         => 'puppet_enterprise_repo_update',
      }
    }
    /^ubuntu-\d+\.\d+-amd64$/: {
      file { '/etc/apt/apt.conf.d/90pe_repo':
        ensure => absent,
      }
      file { '/etc/apt/apt.conf.d/90pc_repo':
        ensure => absent,
      }

      #On primary server repo is pointing to the local directory and no need
      #to configure network URIs
      if $local_config {
        file { "/etc/apt/apt.conf.d/90${repo_name}":
          ensure => absent,
        }
      } else {
        $conf_settings = [
          "Acquire::https::${_master}::Verify-Peer false;",
          "Acquire::http::Proxy::${_master} DIRECT;"
        ]

        # File resource defaults from the master class are getting applied here due to PUP-3692,
        # need to be explicit with owner/group, otherwise they get set to pe-puppet.
        file { "/etc/apt/apt.conf.d/90${repo_name}":
          ensure  => file,
          owner   => '0',
          group   => '0',
          content => pe_join($conf_settings, "\n"),
          notify  => Exec['pe_apt_update'],
          tag     => 'puppet_enterprise_repo_update',
        }
      }

      file { "/etc/apt/sources.list.d/${repo_name}.list":
        ensure  => file,
        owner   => '0',
        group   => '0',
        content => "deb ${repo_source} ./",
        notify  => Exec['pe_apt_update'],
        tag     => 'puppet_enterprise_repo_update',
      }

      exec { 'pe_apt_update':
        command     => "/usr/bin/apt-get update -o Dir::Etc::sourcelist='sources.list.d/${repo_name}.list' -o Dir::Etc::sourceparts='-' -o APT::Get::List-Cleanup='0'",
        logoutput   => 'on_failure',
        refreshonly => true,
        tag         => 'puppet_enterprise_repo_update',
      }
    }
    /^sles-\d+-x86_64$/: {
      # removing erroneously named repo file
      file { '/etc/zypp/repos.d/pe_repo.repo':
        ensure => absent,
      }
      file { '/etc/zypp/repos.d/pc_repo.repo':
        ensure => absent,
      }

      $repo_file = "/etc/zypp/repos.d/${repo_name}.repo"

      # In Puppet Enterprise, agent packages are served by the same server
      # as puppet server, which can be using either a self signed CA, or an external CA.
      # Zypper has issues with validating a self signed CA, so for now disable ssl verification.
      $repo_settings = {
        'name'        => $repo_name,
        'enabled'     => '1',
        'autorefresh' => '0',
        'baseurl'     => "${repo_source}?ssl_verify=no",
        'type'        => 'rpm-md',
      }

      $repo_settings.each |String $setting, String $value| {
        pe_ini_setting { "zypper ${repo_name} ${setting}":
          ensure  => present,
          path    => $repo_file,
          section => $repo_name,
          setting => $setting,
          value   => $value,
          notify  => Exec['pe_zyp_update'],
          tag     => 'puppet_enterprise_repo_update',
        }
      }

      # the repo should be refreshed on any change so that new artifacts
      # will be recognized
      exec { 'pe_zyp_update':
        command     => "/usr/bin/zypper ref ${repo_name}",
        logoutput   => 'on_failure',
        refreshonly => true,
        tag         => 'puppet_enterprise_repo_update',
      }
    }
    default: {
      fail("Unable to generate package repository configuration. Platform described by facts['platform_tag'] '${platform_tag}' is not a known primary platform.")
    }
  }
}
