# Ensures default PE Node Groups are created.
#
# See: https://docs.puppet.com/pe/latest/console_classes_groups_preconfigured_groups.html
#
# @param pe_node_group_environment [String] Allows overriding the environment
#   that the PE Node Groups are set to. Defaults to 'production'. This does
#   not apply to the two environment groups, which are only created and not
#   updated from release to release.
class pe_install::install::classification(
  String $pe_node_group_environment = 'production',
  Boolean $manage_pe_database_pinned_nodes = true,
) {
  include 'puppet_enterprise::profile::console'

  Pe_node_group {
    ensure      => present,
    server      => $::pe_install::console_certname,
    require     => Service['pe-console-services'],
    environment => $pe_node_group_environment,
  }

  $pcp_broker_host = lookup('puppet_enterprise::pcp_broker_host',
    { 'default_value' => $::pe_install::master_certname })
  $base_puppet_enterprise_parameters = {
    'certificate_authority_host'      => $::pe_install::ca_certname,
    'puppet_master_host'              => $::pe_install::master_certname,
    'console_host'                    => $::pe_install::console_certname,
    'puppetdb_host'                   => $::pe_install::puppetdb_certname,
    'database_host'                   => $::pe_install::database_certname,
    'pcp_broker_host'                 => $pcp_broker_host,
  }
  $deprecated_puppet_enterprise_parameters = {}

  # If a user has specified any of the following parameters from the
  # base puppet_enterprise namespace, add it to classifier for ongoing
  # management until MEEP classification work is finished.
  $puppet_enterprise_hiera_params = [
    #Database params
    'activity_database_name',
    'activity_service_migration_db_user',
    'activity_service_regular_db_user',
    'classifier_database_name',
    'classifier_service_migration_db_user',
    'classifier_service_regular_db_user',
    'orchestrator_database_name',
    'orchestrator_service_migration_db_user',
    'orchestrator_service_regular_db_user',
    'rbac_database_name',
    'rbac_service_migration_db_user',
    'rbac_service_regular_db_user',
    'puppetdb_database_name',
    'puppetdb_database_user',
    'database_ssl',
    'database_cert_auth',

    # App overrides
    'console_port',
    'database_port',
    'puppetdb_port',

    # Data analytics collection
    'send_analytics_data',

    #database_host overrides
    'puppetdb_database_host',
    'console_database_host',
    'orchestrator_database_host',
  ]

  $additional_puppet_enterprise_params = get_hiera_overrides('puppet_enterprise', $puppet_enterprise_hiera_params)

  pe_node_group { 'PE Infrastructure':
    parent  => '00000000-0000-4000-8000-000000000000',
    refresh_classes => true,
    classes => {
      'puppet_enterprise' =>
        $base_puppet_enterprise_parameters +
        $additional_puppet_enterprise_params +
        $deprecated_puppet_enterprise_parameters
    },
  }

  pe_node_group { 'PE Certificate Authority':
    parent  => 'PE Infrastructure',
    pinned  => [$::pe_install::ca_certname],
    classes => {
      'puppet_enterprise::profile::certificate_authority' => {},
    }
  }

  $master_profile_hiera_params = [
    'check_for_updates',
    'code_manager_auto_configure',
    'file_sync_enabled',
    'r10k_remote',
    'r10k_private_key',
    'r10k_git_provider',
    'use_legacy_auth_conf',
  ]

  $pe_repo_hiera_params = [
    'base_path',
  ]

  $master_profile_params = {
    # Set a default of 'none' for the replication_mode
    'replication_mode' => 'none'
  # ... but override with the actual value found in the hiera overrides.
  } + get_hiera_overrides('puppet_enterprise::profile::master', $master_profile_hiera_params)
  $pe_repo_params = get_hiera_overrides('pe_repo', $pe_repo_hiera_params)
  $pe_repo_platform = platform_tag_to_pe_repo_class($facts['platform_tag'])
  $default_pe_repo = "pe_repo::platform::${pe_repo_platform}"

  $pe_master_default_classes = {
    'pe_repo'                                          => $pe_repo_params,
    $default_pe_repo                                   => {},
    'pe_repo::platform::windows_x86_64'                => {},
    'pe_repo::platform::windowsfips_x86_64'            => {},
    'puppet_enterprise::profile::master'               => $master_profile_params,
  }
  $deprecated_pe_master_classes = {}
  $pe_master_classes = $pe_master_default_classes + $deprecated_pe_master_classes

  $pe_repo_platform_params = lookup(
    'agent_platforms',
    {'default_value' => []}
  ).reduce({}) |Hash $acc, String $platform_tag| {
    $pe_repo_class = platform_tag_to_pe_repo_class($platform_tag)
    $acc + ["pe_repo::platform::${pe_repo_class}", {}]
  }

  $pe_compiler_rule = ["and", ["=", ["trusted", "extensions", "pp_auth_role"], "pe_compiler"]]
  $pe_master_rule = $::pe_install::enforce_pe_master_rule ? {
    true    => $pe_compiler_rule,
    default => undef,
  }
  pe_node_group { 'PE Master':
    parent  => 'PE Infrastructure',
    pinned  => [$::pe_install::master_certname],
    rule    => $pe_master_rule,
    classes => pe_merge($pe_master_classes, $pe_repo_platform_params)
  }

  pe_node_group { 'PE Compiler':
    parent  => 'PE Master',
    rule    => $pe_compiler_rule,
    classes => {
      'puppet_enterprise::profile::master' => {
        'puppetdb_host' => ['${trusted[\'certname\']}'],
        'puppetdb_port' => [$puppet_enterprise::puppetdb_ports_array[0]],
      },
      'puppet_enterprise::profile::puppetdb' => {},
    },
    require => Pe_node_group['PE Master'],
  }

  pe_node_group { 'PE Orchestrator':
    parent  => 'PE Infrastructure',
    pinned  => [$::pe_install::master_certname],
    classes => {
      'puppet_enterprise::profile::orchestrator' => {},
    }
  }

  pe_node_group { 'PE Console':
    parent  => 'PE Infrastructure',
    pinned  => [$::pe_install::console_certname],
    classes => {
      'puppet_enterprise::profile::console' => {},
      'puppet_enterprise::license' => {},
    }
  }

  pe_node_group { 'PE PuppetDB':
    parent  => 'PE Infrastructure',
    pinned  => pe_flatten([$::pe_install::puppetdb_certname]),
    classes => {
      'puppet_enterprise::profile::puppetdb' => {},
    }
  }

  $pe_database_pinned_nodes = $manage_pe_database_pinned_nodes ? {
    true => pe_delete_undef_values( pe_unique(
                [ $::pe_install::database_certname,
                  $::pe_install::puppetdb_database_host,
                  $::pe_install::console_database_host,
                  $::pe_install::orchestrator_database_host ] ) ),
    false => [],
  }

  pe_node_group { 'PE Database':
    parent  => 'PE Infrastructure',
    pinned  => $pe_database_pinned_nodes,
    classes => {
      'puppet_enterprise::profile::database' => {},
    }
  }

  pe_node_group { 'PE Agent':
    parent  => 'PE Infrastructure',
    rule    => ['and',
      ['~', ['fact', 'aio_agent_version'], '.+'],
    ],
    classes => {
      'puppet_enterprise::profile::agent' => {},
    }
  }
  if ($::pe_install::is_install and !$::pe_install::is_upgrade and !$::pe_install::repairing) {
    $agent_hash = {'pcp_broker_list' => ["${::pe_install::master_certname}:8142"], 'server_list' => ["${::pe_install::master_certname}:8140"], 'primary_uris' => ["${::pe_install::master_certname}:8140"], 'manage_puppet_conf' => true,}
  } else {
    $agent_hash = { }
  }
  pe_node_group { 'PE Infrastructure Agent':
    parent  => 'PE Agent',
    rule    => ['and',
      ['~', ['fact', 'pe_server_version'], '.+'],
    ],
    classes => {
      'puppet_enterprise::profile::agent' => $agent_hash
    },
    require => Pe_node_group['PE Agent'],
  }

  # Create out-of-box environment groups, only on new installs
  if ($::pe_install::is_install and !$::pe_install::is_upgrade and !$::pe_install::repairing) {
    pe_node_group { 'All Environments':
      ensure             => present,
      parent             => 'All Nodes',
      description        => 'Environment group parent and default',
      create_only        => true,
      environment_trumps => true,
      environment        => 'production',
      rule               => ['and', ['~', 'name', '.*']],
      classes            => { },
    }

    pe_node_group { 'Production environment':
      ensure             => present,
      parent             => 'All Environments',
      description        => 'Production nodes',
      create_only        => true,
      environment_trumps => true,
      environment        => 'production',
      rule               => ['and', ['=', ['trusted', 'extensions', 'pp_environment'], 'production']],
      classes            => { },
    }

    pe_node_group { 'Development environment':
      ensure             => present,
      parent             => 'All Environments',
      description        => 'Development nodes',
      create_only        => true,
      environment_trumps => true,
      environment        => 'development',
      rule               => ['and', ['=', ['trusted', 'extensions', 'pp_environment'], 'development']],
      classes            => { },
    }

    pe_node_group { 'Development one-time run exception':
      ensure             => present,
      parent             => 'Development environment',
      description        => "Allow development nodes to request a different puppet environment for a one-time run. May request and use any Puppet environment except for 'production'.",
      create_only        => true,
      environment_trumps => true,
      environment        => 'agent-specified',
      rule               => ['and',
        ['~', ['fact', 'agent_specified_environment'], '.+'],
        ['not', ['=', ['fact', 'agent_specified_environment'], 'production']],
      ],
      classes            => { },
    }
  }

}
