require 'puppet/indirector/face'
require 'puppet/agent'
require 'puppet/error'
require 'puppet/util/pe_node_groups'
require 'puppet_x/puppetlabs/meep/util'
require 'puppet_x/util/service_status'
require 'puppet_x/util/rbac'
require 'puppet_x/util/classifier'
require 'puppet_x/util/orchestrator'
require 'puppet_x/util/code_manager'
require 'puppet_x/util/forget'

Puppet::Face.define(:infrastructure, '1.0.0') do
  extend PuppetX::Puppetlabs::Meep::Util

  action :forget do
    summary _('Remove traces of a provisioned replica node that has been permanently removed.')
    description <<-EOT
      Purge replication slots previously dedicated to the given node
      and remove it from the HA Replica node group.
    EOT

    arguments '<replica_certname>'

    option("--force") do
      summary _("Skip all pre-forget validation.")
      default_to {false}
      description <<-EOT
        Skip all pre-forget validation. Skipping checks will mean that
        prerequisite services may not be configured or running and some
        forget steps may fail or need to be manually run.
      EOT
    end

    option('--token-file <token_path>') do
      summary _('Path to a token file')
      description <<-EOT
        Path to an RBAC token to use for authentication.
      EOT
    end

    when_invoked do |replica_certname, options|
      timestamp = time_stamp
      master_certname = Puppet[:certname].strip

      unless options[:force]
        # Fail fast if no token is found
        rbac_token = PuppetX::Util::RBAC.load_token(options[:token_file])

        if master_certname == replica_certname
          Puppet.err(_("You can only forget a replica") + "; " + _("%{certname} is the primary." % { certname: replica_certname } ))
          exit(1)
        end
      end

      puts _("Dropping replication slots for %{certname}.") % { certname: replica_certname }
      PuppetX::Util::Forget.drop_replication_slots_or_exit(master_certname, replica_certname)

      config = PuppetX::Util::ServiceStatus.load_services_config()
      services = PuppetX::Util::ServiceStatus.validate_and_select_services_from_config(config, 'forget', master_certname)

      puts _("Removing %{certname} from the PE HA Replica group.") % { certname: replica_certname }
      nc_service = services[:classifier]
      nc = Puppet::Util::Pe_node_groups.new(nc_service[:server], nc_service[:port].to_i, "/#{nc_service[:prefix]}")
      all_groups = nc.get_groups()
      ha_replica_group_id = PuppetX::Util::Classifier.find_group_id(all_groups, 'PE HA Replica')
      nc.unpin_nodes_from_group(ha_replica_group_id, [replica_certname])

      puts _("Removing %{certname} from the list of allowed replicas.") % { certname: replica_certname }
      PuppetX::Util::Classification.configure_ha_master_node_group_to_remove_replica(nc, all_groups, replica_certname)
      PuppetX::Util::Classification.disable_replica(nc, all_groups, master_certname, replica_certname)

      puts _('Running puppet on infrastructure nodes.')
      rbac_token ||= PuppetX::Util::RBAC.load_token(options[:token_file])
      log = logfile(action: 'forget', timestamp: timestamp, description: 'first_primary_run')
      PuppetX::Util::Orchestrator.run_puppet(services[:orchestrator], master_certname, {:nodes => [master_certname]}, rbac_token, report_log: log)
      log = logfile(action: 'forget', timestamp: timestamp, description: 'other_compilers_run')
      PuppetX::Util::Orchestrator.run_puppet(services[:orchestrator],
                                             'all other compilers',
                                             {:query => ['from', 'resources',
                                                         ['extract', ['certname'],
                                                          ['and',
                                                           ['=', 'type', 'Class'],
                                                           ['=', 'title', 'Puppet_enterprise::Profile::Master'],
                                                           ['not', ['=', 'certname', master_certname]],
                                                           ['not', ['=', 'certname', replica_certname]],
                                                           ['=', ['node', 'active'], true]]]]},
                                             rbac_token,
                                             :allow_empty => true,
                                             report_log: log)

      puts _("Purging %{certname}") % { certname: replica_certname }
      Puppet::Face[:node, :current].purge(replica_certname)

      puts _("Running puppet on primary")
      log = logfile(action: 'forget', timestamp: timestamp, description: 'second_primary_run')
      PuppetX::Util::Orchestrator.run_puppet(services[:orchestrator], master_certname, {:nodes => [master_certname]}, rbac_token, report_log: log)

      puts _("%{certname} removed as an active replica.") % { certname: replica_certname }
    end
  end
end
