require 'puppet/indirector/face'
require 'puppet_x/puppetlabs/meep/infra/feature_flags'
require 'puppet_x/puppetlabs/meep/infra/plan_executor'
require 'puppet_x/puppetlabs/meep/enable/flags'
require 'puppet_x/puppetlabs/meep/provision/base'
require 'puppet_x/puppetlabs/meep/provision/streaming'

Puppet::Face.define(:infrastructure, '1.0.0') do
  extend PuppetX::Puppetlabs::Meep::Infra::FeatureFlags

  action :provision do
    summary _('Provisions a new secondary infrastructure node based on the primary.')
    description(
      _("Provision a PE node with a specific role.\n") +
      _("Known roles:\n") +
      _("  replica\n") +
      _("  compiler\n")
    )

    arguments "<role> <node_certname>"

    PuppetX::Puppetlabs::Meep::Enable::Flags.add_flags(self)

    option("--token-file PATH") do
      summary _("Path to a token file")
      description <<-EOT
        Path to an RBAC token to use for authentication when provisioning a replica.
      EOT
    end

    option("--force") do
      summary _("Skip all pre-provision validation and node type verification in underlying plans.")
      default_to {false}
      description <<-EOT
      Skip all pre-provision validation and node type verification in the underlying plans when upgrading a
      replica or compiler. Ensure the node you are providing to the command
      is the correct type before using this flag.
      EOT
    end

    option('--[no-]streaming') do
      summary _('Stream initial database replication.')
      default_to {true}
      description <<-EOT
        When provisioning a replica, use pg_basebackup to
        accelerate copying of the databases to the replica.
      EOT
    end

    option('--[no-]enable') do
      summary _('After provisioning, enable the replica.')
      default_to {false}
      description <<-EOT
        After provisioning the replica, enable the replica
        (same as puppet infrastructure enable replica). This
        may only be used with the --streaming flag.
      EOT
    end

    option("--dns-alt-names ALTNAMESLIST") do
      summary _('Compiler option: Set alternative names for the new compiler node.')
      default_to {'puppet'}
      description <<-EOT
        Comma-separated list of any alternative names that agents use to connect to compilers. If not specified, 'puppet' will be used by default.
      EOT
    end

    option('--no-dns-alt-names') do
      summary _('Compiler option: Disable setting of dns-alt-names on compiler')
      description <<-EOT
        Disable setting of dns-alt-names on compiler during provisioning.
      EOT
    end

    option('--use-ssh') do
      summary _('Use ssh for transport rather than relying on the Orchestrator.')
    end

    option('--[no-]host-key-check') do
      summary _('Compiler option: Enables or disables automatic host key checking.')
      description <<-EOT
        Enable or disable automatic host key checking when utilizing SSH transport
        to provision a compiler.
      EOT
    end

    option('--private-key KEY') do
      summary _('Compiler option: Specify the private key to use to access the required nodes.')
      description <<-EOT
        Specify the private key to use when accessing the nodes required to provision
        a new compiler. This key must be valid for the node to be provisioned, as well
        as the primary, existing compilers, and the external PE PostgreSQL node, if it exists.
      EOT
    end

    option('--tmpdir DIRECTORY') do
      summary _('Compiler option: Directory that Bolt will use to upload and execute temporary files on the target. Defaults to $TMPDIR if set or /tmp if not.')
    end

    option('--sudo-password PASSWORD') do
      summary _('Compiler option: Pass your superuser password through to your run command.')
    end

    option('--run-as USER') do
      summary _('Compiler option: Run your command as the specified user.')
    end

    option('--password PASSWORD') do
      summary _('Compiler option: Pass your password through to your run command.')
    end

    option('--[no-]tty') do
      summary _('Compiler option: Request a pseudo TTY on nodes that support it.')
    end

    when_invoked do |role, certname, options|
      known_roles = ['replica', 'compiler']
      if !known_roles.include?(role)
        raise PuppetX::Util::InfrastructureUnknownRoleError.new(role, known_roles) # rubocop:disable GetText/DecorateFunctionMessage
      end

      case role
      when 'replica'
        provision_replica(certname, options)
      when 'compiler'
        provision_compiler(certname, options)
      end
    end
  end

  def run_provision_compiler_plan(compiler_certname, options)
    params = {
      'compiler' => compiler_certname,
      'force' => options[:force].to_s,
    }
    params['dns_alt_names'] = options[:dns_alt_names] if !options.include?(:no_dns_alt_names)

    engine = options[:use_ssh] ? 'bolt' : 'orchestrator'

    status, result = PuppetX::Puppetlabs::Meep::Infra::PlanExecutor.run(
      'enterprise_tasks::provision_compiler',
      params: params,
      options: options,
      engine: engine
    )

    if status.exitstatus != 0
      Puppet.err(_("Could not provision compiler. See /var/log/puppetlabs/installer/#{engine}_info.log for more details."))
      Puppet.err(result)
      exit status.exitstatus
    end

    status
  end

  # Run puppet on the primary using bolt to connect to localhost. Since
  # the task is only run on the localhost node, ssh is not involved.
  # Only use this when we expect the puppet run to restart the Orchestrator.
  # An Orchestrator restart will break any other orchestrated run of puppet
  # on the primary, be it a command/deploy, task or plan_run...
  def run_puppet_on_master_localhost
    bolt_action = 'task run enterprise_tasks::run_puppet -t localhost'
    status, _output = PuppetX::Util::Bolt.run_bolt(bolt_action, {})
    if status.exitstatus != 0
      exit status.exitstatus
    end
  end

  def provision_compiler(compiler_certname, options)
    puts _("Provisioning compiler on #{compiler_certname}. This may take several minutes.")
    run_provision_compiler_plan(compiler_certname, options)
    run_puppet_on_master_localhost
    puts _("Provisioning complete.")
  end

  def provision_replica(replica_certname, options)
    impl =  case options[:streaming]
    when true
      PuppetX::Puppetlabs::Meep::Provision::Streaming.new(replica_certname, options)
    else
      PuppetX::Puppetlabs::Meep::Provision::Base.new(replica_certname, options)
    end

    impl.run
  end
end
