require 'puppet/util/colors'
require 'puppet_x/util/bolt'
require 'puppet_x/util/orchestrator'
require 'puppet_x/util/rbac'
require 'puppet_x/util/service_status'
require 'puppet_x/puppetlabs/meep/util'

module PuppetX
module Puppetlabs
module Meep
module Infra

  # We have two engines for running plans, the pe-installer's Bolt,
  # and the Orchestrator.
  #
  # Preferentially we use the Orchestrator. But not every plan can function
  # over the Orchestrator (enterprise_tasks::rebuild_ca,
  # enterprise_tasks::primary_cert_regen, for example) and the user might need
  # to fallback to ssh for other reasons (Orchestrator down, no RBAC, etc.)
  #
  # The PuppetX::Util::Orchestrator.run_plan() method can be used directly
  # when the code in question knows that it can or must always use the
  # Orchestrator.
  #
  # This PlanExecutor class is here for other cases where a puppet-infra
  # action might need either engine.
  class PlanExecutor
    include Puppet::Util::Colors
    include PuppetX::Puppetlabs::Meep::Util

    def self.run(plan, params:, options: {}, replacements: {}, engine:)
      executor = new
      executor.run(plan, params: params, options: options, replacements: replacements, engine: engine)
    end

    # Run a plan.
    #
    # To run a plan with the Orchestrator:
    #
    #   run(
    #     'some::plan',
    #     params: { 'arg1' => 'foo' },
    #     options: { token_file: '/path/to/rbac_token' }, # optional example
    #       # the default token_file path is ~/.puppetlabs/token (defined
    #       # in PuppetX::Util::RBAC)
    #     engine: 'orchestrator'
    #   )
    #
    # To run a plan with Bolt:
    #
    #   run(
    #     'some::plan',
    #     params: [ 'arg1=foo' ],
    #     options: { host_key_check: true }, # optional example
    #     replacements: { 'some::plan' => 'some_infra_action' }, # optional example
    #     engine: 'bolt'
    #   )
    #
    # @param plan [String] the actual plan name.
    # @param params [Array<String>,Hash] these are the parameters to be passed
    # to the plan. If given an array of strings, we're expecting strings
    # as would be set on the command line (ex: ['param1=foo', 'param2=bar']).
    # @param options [Hash] in general, this is the hash of options which will
    # be passed to PuppetX::Util::Bolt to generate command line options for
    # bolt execution. However, there is one option, :token_file, which can
    # be extracted for the orchestrator to lookup the rbac_token.
    # @param replacements [Hash] a hash of strings to replace in bolt execution
    # output; used by `puppet-infra run` to masquerade plan names. Not currently
    # used when running via the Orchestrator.
    # @param engine [String] determines which engine to run the plan with.
    # Expects either 'orchestrator or 'bolt'.
    def run(plan, params:, options: {}, replacements: {}, engine:)
      case engine
      when 'orchestrator'
        log_name = '/var/log/puppetlabs/installer/orchestrator_info.log'
        log = File.new(log_name, 'a')
        status, result = _run_orchestrator(plan, params: params, options: options, log: log)
        log.close
        return status, result
      else
        _run_bolt(plan, params: params, options: options, replacements: replacements)
      end
    end

    attr_writer :_orchestrator_config, :_rbac_token, :_nodes_config

    # @api private
    # @return [Array<Symbol>] list of services required to run plans over the
    # Orchestrator on the primary.
    def self._plan_executor_services()
      PuppetX::Util::ServiceStatus.required_services.reject do |s|
        [:classifier, :code_manager, :file_sync_storage, :file_sync_client].include?(s)
      end
    end

    def self.primary_certname
      Puppet[:certname]
    end

    # Load and validate the services configuration, raise an error for missing
    # or non-running orchestrator, puppetserver or pcp-broker
    # ({PlanExecutor._plan_executor_services}).
    #
    # @api private
    # @return [Hash] orchestrator service config
    # @raise [PuppetX::Util::ServicesStatus] if missing a service, or service not
    # in a ready state.
    def self._load_and_validate_orchestrator_config(services_conf_path = PuppetX::Util::ServiceStatus.config_path)

      # Load the arrays of known PE infrastructure services and nodes from the
      # services.conf file.
      services_config = PuppetX::Util::ServiceStatus.load_services_config(services_conf_path)
      nodes_config = PuppetX::Util::ServiceStatus.load_nodes_config(services_conf_path)

      # Validate that +primary_certname+ has the services we need for plan execution.
      services, failed_services, errors = PuppetX::Util::ServiceStatus.validate_required_services(services_config, primary_certname, _plan_executor_services)

      if !errors.empty?
        verb = failed_services.count > 1 ? 'are' : 'is'
        error_message = [
          _("Puppet cannot run a plan with the Orchestrator until %{failed} services #{verb} enabled.") % { failed: failed_services.join(", ") }
        ] << errors
        raise(PuppetX::Util::ServiceStatusError, error_message.join("\n")) # rubocop:disable GetText/DecorateFunctionMessage
      else
        # Validate that the required services are in fact running and ready.
        # First, hit their status endpoints
        status = ['master', 'orchestrator'].map do |service|
          PuppetX::Util::ServiceStatus.services_for(primary_certname, service, services_config, 5, nodes_config: nodes_config)
        end.flatten
        # Parse status and raise errors if necessary
        if !PuppetX::Util::ServiceStatus.all_services_running?(status)
          error_message = PuppetX::Util::ServiceStatus.services_not_running(status).map do |e|
            _("%{display_name} is not running") % { display_name: e[:display_name] }
          end
          raise(PuppetX::Util::ServiceStatusError, error_message.join("\n")) # rubocop:disable GetText/DecorateFunctionMessage

        end
      end

      # If we are here, everything expected is running
      {
        orchestrator_service: services[:orchestrator],
        nodes_config: nodes_config,
      }
    end

    # @api private
    def cache_services_conf
      service_and_nodes = self.class._load_and_validate_orchestrator_config
      self._orchestrator_config = service_and_nodes[:orchestrator_service]
      self._nodes_config = service_and_nodes[:nodes_config]
    end

    # @api private
    def _orchestrator_config
      unless @_orchestrator_config
        cache_services_conf
      end
      @_orchestrator_config
    end

    # api private
    def _nodes_config
      unless @_nodes_config
        cache_services_conf
      end
      @_nodes_config
    end

    # @api private
    def _rbac_token(options)
      unless @_rbac_token
        self._rbac_token = PuppetX::Util::RBAC.load_token(options[:token_file])
      end
      @_rbac_token
    end

    # This is emulating a Process::Status for _run_orchestrator to return
    # an 'exitstatus' with, so that caller doesn't have to care whether this was
    # PTY/Bolt or Orchestrator operating the wheel.
    class Status
      attr_accessor :exitstatus
      def initialize(state)
        self.exitstatus = case state
        when 'success' then 0
        else 1
        end
      end
    end

    # The Orchestrator does not have localhost in its inventory.
    # Most of our plans that perform actions on the primary implicitly (often in
    # addition to actions on other remote nodes) have 'master' or 'primary' defaulted to
    # localhost to sidestep requiring the primary to be able to ssh to itself.
    #
    # To work arond this when running the plans over the orchestrator,
    # we lookup the plan metadata, and if it has a parameter 'master' or 'primary' defaulting
    # to localhost, we replace it with the primary's certname, assuming it
    # wasn't passed in the parameters already.
    #
    # @api private
    # @param plan_metadata [Hash] metadata hash as returned by
    # PuppetX::Util::Orchestrator.get_plan_metadata()
    # @param parameters [Hash] the original set of parameters.
    # @param options [Hash] options passed to the PlanExecutor.
    # @return [Hash] the same parameters Hash given, but possibly with
    # master/primary parameters added pointing to {PlanExecutor.primary_certname}, if
    # metadata indicated it would otherwise default to localhost.
    def _munge_master_primary(plan_metadata, parameters, options)
      if options[:skip_localhost_rewrite]
        return parameters
      end
      # Don't munge if a user set the parameter rather than default 'localhost'
      skip = []
      skip << 'master' if parameters.include?('master')
      skip << 'primary' if parameters.include?('primary')

      ['master','primary'].each do |param|
        next if skip.include?(param)
        all_parameters_metadata = plan_metadata['parameters'] || {}
        param_metadata = all_parameters_metadata[param] || {}
        default_value = param_metadata['default_value']

        if default_value =~ /'?localhost'?/
          parameters[param] = self.class.primary_certname
        end
      end
      parameters
    end

    # Plan parameters may be coming from the commandline, in which
    # case they will all be Strings, but we are transporting a JSON
    # blob to the Orchestrator, and some plans have Numeric or Boolean
    # parameters. If these are sent as strings, they fail the type
    # check when the orchestrator executes the plan.
    #
    # This method checks the given plan_metadata and casts basic
    # Numeric/Boolean types to avoid this.
    #
    # @api private
    # @param plan_metadata [Hash] metadata hash as returned by
    # PuppetX::Util::Orchestrator.get_plan_metadata()
    # @param parameters [Hash] the original set of parameters.
    # @param options [Hash] options passed to the PlanExecutor.
    # @return [Hash] the same parameters Hash given, but possibly with
    # cast Numeric/Boolean parameters.
    def _type_parameters(plan_metadata, parameters, _options)
      parameters_metadata = plan_metadata['parameters'] || {}
      parameters.keys.each do |param|
        param_meta = parameters_metadata[param] || {}
        type = param_meta['type']
        case type
        when /(?:Optional\[)?Boolean(?:\])?/
          value = case parameters[param]
                  when 'true' then true
                  when 'false' then false
                  else
                    plan = plan_metadata['name']
                    message = "The %{plan} parameter %{parameter_name} expected type Boolean, got '%{parameter_value}'. Plan metadata: %{plan_metadata}" % { plan: plan, parameter_name: param, parameter_value: parameters[param], plan_metadata: plan_metadata }
                    raise(PuppetX::Util::OrchestratorError, message) # rubocop:disable GetText/DecorateFunctionMessage
                  end
          parameters[param] = value
        when /(?:Optional\[)?Integer(?:\])?/
          parameters[param] = parameters[param].to_i
        when /(?:Optional\[)?Float(?:\])?/
          parameters[param] = parameters[param].to_f
        else
          # skip
        end
      end
      parameters
    end

    # @api private
    def _ensure_parameter_hash(params)
      case params
      when Array
        params.each_with_object({}) do |p, hash|
          k, v = p.split('=')
          hash[k] = v
          begin
            unless v.nil?
              parsed = JSON.parse(v, quirks_mode: true)
              hash[k] = ["true", "false"].include?(v.downcase) ? v : parsed
            end
          rescue JSON::ParserError
          end
        end
      else
        params
      end
    end

    # Extracts the variable target parameters from the given param set
    # and returns their values as a flat array of the target certnames.
    #
    # Used to get the list of target node certnames from our various plans that
    # have different parameters for this (compiler, replica, target, etc.)
    #
    # @api private
    # @param params [Hash] hash of plan parameters.
    # @return [Array] of certnames.
    def _extract_target_params(params)
      # this varies by plan...
      target_param_keys = [
        'target',
        'targets',
        'compiler',
        'compilers',
        'replica',
        'replicas',
      ]
      params_to_check = params.select { |k,_v| target_param_keys.include?(k) }
      params_to_check.values.flatten
    end

    # Check that PE Infrastructure targets in the given plan are connected
    # to the primary pcp-broker (that the nodes are part of a correctly
    # configured PE Infrastructure Agents node group that has the primary set
    # for it's pxp-agent configuration). In a large or extra-large
    # installation, the PE Agent is configuring pxp-agent to the load balancer,
    # and this will break many plans during upgrade or provisioning if PE
    # Infrastructure Agent is not configuring pxp-agent to the primary.
    #
    # Warn if they are not, but do not raise an error.
    #
    # @api private
    # @param params [Hash] the hash of parameters for the plan.
    # @param infra_nodes [Array] the certnames of the installation's PE infra
    # agent nodes. (compilers, replica, primary)
    # @param rbac_token [String] the RBAC token needed to connect to the Orchestrator.
    # @param log [File] our log file.
    # @return [Boolean] true if targets are connected to the primary pcp-broker.
    def _check_target_pcp_brokers(params:, infra_nodes:, rbac_token:, log:)
      targets = _extract_target_params(params)
      infra_targets = targets & infra_nodes
      misconfigured_targets = []

      if !infra_targets.empty?
        primary_cert = self.class.primary_certname
        log.puts(_('Checking for pcp connectivity from %{primary_cert} pcp-broker to: %{infra_targets}') % { primary_cert: primary_cert, infra_targets: infra_targets })
        misconfigured_targets = PuppetX::Util::Orchestrator.get_misconfigured_pcp_broker_connections(_orchestrator_config, rbac_token, infra_targets, primary_cert)

        if !misconfigured_targets.empty?
          messages = misconfigured_targets.map do |t|
              _("Not connected to primary's pcp-broker: %{target}") % { target: t }
          end
          warning = _(
            <<~EOS
              Some target nodes that are PE Infrastructure nodes are not connected
              to the primary's pcp-broker at %{primary_broker}. This may be an
              indication of a misconfigured PE Infrastructure Agent node group for
              a large or extra-large installation in a load balanced environment.
              Please see the documentation for compiler configuration.

              Target nodes checked: %{target_nodes}
            EOS
          ) % { target_nodes: infra_targets, primary_broker: primary_cert }

          puts(colorize(:red, warning))
          messages.each do |m|
            puts(colorize(:red, "  #{m}"))
            log.puts("  #{m}")
          end
        end
      end

      misconfigured_targets.empty?
    end

    # Prepare our inputs and then pass on to the Orchestrator lib.
    #
    # @api private
    # @param plan [String] the name of the plan to run.
    # @param params [Hash,Array<String>] need a Hash, but may be given an array
    # of Strings, if passed commandline parameters.
    # @param options [Hash] commandline options, of which token_file is relevant.
    # @param log [File] our log file.
    def _run_orchestrator(plan, params:, options:, log:)
      parameters = _ensure_parameter_hash(params)
      rbac_token = _rbac_token(options)
      plan_metadata = PuppetX::Util::Orchestrator.get_plan_metadata(
        _orchestrator_config,
        plan: plan,
        rbac_token: rbac_token
      )
      parameters = _type_parameters(plan_metadata, parameters, options)
      parameters = _munge_master_primary(plan_metadata, parameters, options)

      infra_nodes = _nodes_config.map { |n| n[:certname] }
      _check_target_pcp_brokers(params: parameters, infra_nodes: infra_nodes, rbac_token: rbac_token, log: log)

      description = options[:description] || "Execution of #{plan} on behalf of the puppet-infrastructure command."
      env = options[:pe_environment] || 'production'
      log.puts "-------------------------------"
      log.puts "Running plan #{plan}"
      state = PuppetX::Util::Orchestrator.run_plan(
        _orchestrator_config,
        plan: plan,
        params: parameters,
        description: description,
        rbac_token: rbac_token,
        environment: env,
        log: log
      )
      # We return 0 for successful runs, 1 otherwise.
      _output = ''
      log.puts "Finished plan #{plan}"
      log.puts "-------------------------------"
      [Status.new(state), _output]
    end

    # Prepare params before passing to bolt
    #
    # @api private
    # @param params [Array<String>,Hash] need an Array of Strings, but may be given a Hash.
    # @return [String] the same parameters given, but in the string form as in
    # "param=value  hash_param='{"key1":"value1"}' array_param='["val1","val2"]'"
    def _prep_bolt_params(params)
      parameters = _ensure_parameter_hash(params)
      args = parameters.map do |k,v|
        if v.is_a?(Hash)
          "#{k}='#{v.to_json}'"
        elsif v.is_a?(Array)
          "#{k}='#{v}'"
        else
          "#{k}=#{v}"
        end
      end
      args.join(' ')
    end
    # @api private
    # @param plan [String] name of the plan to run.
    # @param params [Array<String>,Hash] need an Array of Strings, but may be given a Hash.
    # @param options [Hash] commandline options to pass on to bolt.
    # @param replacements [Hash] replacement strings for the puppet-infra run
    # comamnd -> plan translations in the output.
    def _run_bolt(plan, params:, options:, replacements:)
      args_str = _prep_bolt_params(params)
      opts_str = PuppetX::Util::Bolt.options_to_string(options)
      bolt_action = "plan run #{plan} #{args_str} #{opts_str}"
      PuppetX::Util::Bolt.run_bolt(bolt_action.strip, replacements)
    end
  end

end
end
end
end

