require 'puppet/agent'
require 'time'
module PuppetX
module Puppetlabs
module Meep
  # Utility methods.
  module Util
    include Puppet::Agent::Disabler
    include Puppet::Agent::Locker
    # Expects dot separated version strings such as "2017.3.0"
    #
    # Short circuits returning nil if either argument is nil or an empty string.
    #
    # @return -1, 0 or 1 based on <=> comparison of the Gem::Version
    #   where -1 indicates a < b
    #          0 indicates a = b
    #          1 indicates a > b
    def version_cmp(a, b)
      return nil if a.nil? || b.nil? || a.empty? || b.empty?
      v1 = Gem::Version.create(a)
      v2 = Gem::Version.create(b)
      v1 <=> v2
    end

    # This is similar to what we do in the enterprise_tasks with_agent_disabled function,
    # but only works on the local node this is running from.
    # Note: the enable method of Puppet::Agent::Disabler interferes with
    # puppet infra enable, so it is recreated in the ensure block.
    def with_agent_disabled(message, timeout=300, &block)
      already_disabled = disabled?
      disable(message) unless already_disabled
      start = Time.now
      if lockfile.locked?
        Puppet.debug(_("Puppet run in progress. Waiting up to #{timeout} seconds for lock to clear."))
        while lockfile.locked? && Time.now - start < timeout
          sleep(1)
        end
        raise Puppet::LockError.new(_("Agent lock did not clear in #{timeout} seconds.")) if lockfile.locked?
      end
      yield
    ensure
      unless already_disabled
        Puppet.notice _("Enabling Puppet.")
        disable_lockfile.unlock
      end
    end

    def time_stamp(fmt='%Y-%m-%dT%H.%M.%S%z')
      Time.now.strftime(fmt)
    end

    def self.default_logfile_path
      '/var/log/puppetlabs/installer'
    end

    def logfile(action: nil, timestamp: nil, description: nil, path: PuppetX::Puppetlabs::Meep::Util.default_logfile_path, ext: 'log')
      file = "#{path}/"
      file += "#{action}_" if action
      file += timestamp ? "#{timestamp}" : "#{time_stamp}"
      file += "_#{description}" if description
      file += ".#{ext}" if ext
      file
    end
  end
end
end
end
