require 'puppet/util/colors'
require 'hocon'
require 'puppet_x/util/classification_error'

module PuppetX
  module Util
    class Classifier

      # This should probably just go into pe_node_group for idempotency use cases.
      def self.find_group(all_groups, group_name)
        all_groups.find { |g| g['name'] == group_name }
      end

      def self.find_group_id(all_groups, group_name)
        group = self.find_group(all_groups, group_name)
        if group
          group['id']
        else
          nil
        end
      end

      def self.create_ha_master_node_group(nc, all_groups, pe_master_group_id)
        return if self.find_group_id(all_groups, 'PE HA Master')

        begin
          nc.create_group(
            { :name => 'PE HA Master',
              :parent => pe_master_group_id,
              :classes => {
                'puppet_enterprise::profile::master' => {
                  :file_sync_enabled => true,
                  :replication_mode => 'source',
                  :provisioned_replicas => [],
                },
                'puppet_enterprise::profile::console' => {
                  :replication_mode => 'source'
                },
                'puppet_enterprise::profile::database' => {
                  :replication_mode => 'source',
                  :replica_hostnames => []
                },
                'puppet_enterprise::profile::puppetdb' => {
                  :sync_allowlist => []
                }
              }
          })
        rescue
          # rubocop:disable GetText/DecorateFunctionMessage
          raise PuppetX::Util::CreateNodeGroupError.new('PE HA Master', pe_master_group_id)
        end
      end

      def self.create_ha_replica_node_group(nc, all_groups, pe_infra_group_id)
        return if self.find_group(all_groups, 'PE HA Replica')

        begin
          nc.create_group(
            { :name => 'PE HA Replica',
              :parent => pe_infra_group_id,
              :classes => {
                'puppet_enterprise::profile::primary_master_replica' => {},
              }
          })
        rescue
          # rubocop:disable GetText/DecorateFunctionMessage
          raise PuppetX::Util::CreateNodeGroupError.new('PE HA Replica', pe_infra_group_id)
        end
      end

      def self.get_groups(nc)
        begin
          nc.get_groups()
        rescue
          raise PuppetX::Util::GetNodeGroupsError.new()
        end
      end

      def self.pin_node_to_group(nc, group_name, group_id, certname)
        begin
          nc.pin_nodes_to_group(group_id, [certname])
        rescue
          raise PuppetX::Util::PinNodeError.new(group_name, group_id, certname)
        end
      end

      def self.unpin_node_from_group(nc, group_name, group_id, certname)
        begin
          nc.unpin_nodes_from_group(group_id, [certname])
        rescue
          raise PuppetX::Util::UnpinNodeError.new(group_name, group_id, certname)
        end
      end

      def self.nodes_pinned_to_group(all_groups, group_name)
        group = find_group(all_groups, group_name)
        return [] unless group
        rule = group["rule"]
        if rule[0] != "or"
          []
        else
          rule.select { |clause| clause[0] == "=" and clause[1] == "name" }
            .map { |clause| clause[2] }
        end
      end

      def self.node_pinned_to_group?(all_groups, group_name, node_name)
        nodes_pinned_to_group(all_groups, group_name).include? node_name
      end

      def self.node_is_pinned_as_replica?(all_groups, node_name)
        node_pinned_to_group?(all_groups, 'PE HA Replica', node_name)
      end
    end
  end
end
