require 'json'
require 'puppet_x/util/orchestrator/report'
require 'puppet_x/util/orchestrator/runner'

module PuppetX
  module Util
    class Orchestrator

      # Handles Orchestrator:
      #
      # /orchestrator/v1/command/deploy
      #
      # which kicks off a Puppet run. Results are available through the Job class.
      class PuppetRunner < Runner
        include PuppetX::Util::Orchestrator::Report

        def initialize(service_config, rbac_token)
          super
          @command_type = 'puppet'
        end

        def start_command(scope:, allow_empty: false, **_kwargs)
          url = "#{service_config[:url]}/v1/command/deploy"
          payload = {:scope => scope, :environment => '', :enforce_environment => false}
          uri = URI(url)
          response = orch_connection.post(uri, payload.to_json, headers: orch_request_headers)

          check_orch_response(response, 202)

          body = JSON.parse(response.body)

          # Yes, this is right. The numerical id of a job comes back as its name,
          # whereas the url is called 'id' in the response.
          id = body['job']['name']
          job_for(id)
        end

        # @raise [PuppetX::Util::OrchestratorJobFailedError] with a list of console links to
        # the failed puppet reports.
        def _handle_job_failure(job)
          details = job.failed_node_details
          report_urls = details.map { |d| d['report-url'] }
          raise(PuppetX::Util::OrchestratorJobFailedError.new(_('Puppet failed on one or more nodes.'), report_urls))
        end

        def _save_log(job, args)
          save_report_log(args[:display_scope], job, args[:report_log]) if !args[:report_log].nil?
        end
      end
    end
  end
end
