require 'json'
require 'puppet_x/util/orchestrator/runner'
module PuppetX
  module Util
    class Orchestrator

      # Handles Orchestrator:
      #
      # /orchestrator/v1/command/task
      #
      # Results are primarily available through the Job class.
      class TaskRunner < Runner
        def initialize(service_config, rbac_token)
          super
          @command_type = 'task'
        end

        # Start a task on a remote node via orchestrator.
        # @param scope [Hash] See https://puppet.com/docs/pe/2019.2/orchestrator_api_commands_endpoint.html#scope
        # @param task [String] Task name
        # @param params [Hash] Parameter names and values for the given task
        # @param environment [String] Environment that contains the task, default to 'production'
        # @return [Hash] The job ID and job URL
        def start_command(scope:, task:, params:, environment: 'production', **_kwargs)
          url = "#{service_config[:url]}/v1/command/task"
          payload = {:environment => environment, :scope => scope, :task => task, :params => params}
          uri = URI(url)
          response = orch_connection.post(uri, payload.to_json, headers: orch_request_headers)
          check_orch_response(response, 202)
          body = JSON.parse(response.body)
          id = body['job']['name']
          job_for(id)
        end

        # In the case of a task, the result contains output we may want to
        # return to the user.
        def _get_job_result(job)
          job.node_items
        end

        def _handle_job_failure(job)
          errors = job.task_errors
          errors.each do |node, error|
            Puppet.debug("Task failed on #{node}: #{error}")
          end
          first_error = errors.first[1]
          raise PuppetX::Util::OrchestratorTaskFailedError.new(first_error['msg'], first_error['kind'], first_error['details']) #rubocop:disable GetText/DecorateFunctionMessage
        end
      end
    end
  end
end
