# This profile sets up a PE Plan Executor service.
#
# @param enable [Boolean] Whether or not the plan executor should be available.
# @param orchestrator_url [String] The orchestrator's URL, including the port number.
# @param environments_codedir [Optional[String]] The directory where filesync deploys to and orchestrator looks
#                                                for code when using environments
# @param basemodulepath [Optional[String]] The location where the plan executor will expect to load base modules
# @param environmentpath [Optional[String]] The location where the plan executor will expect to load environments
# @param enable_file_sync_client [Optional[Boolean]] Set to true to enable the file-sync client for the
#                                                    orchestrator. It is _highly_ discouraged to use plans
#                                                    without file-sync enabled
# @param enable_file_sync_locking [Optional[Boolean]] DEPRECATED: This parameter is deprecated and no longer used
# @param versioned_deploys [Optional[Boolean]] enable versioned deployments for environment code. Enabling this
#                                              setting disables the Orchestrator's locking mechanism during code
#                                              deploys
# @param file_sync_data_dir [Optional[String]] The directory for the orchestrator file-sync client to store data
# @param projects_codedir [Optional[String]] The directory where filesync deploys to and orchestrator looks for
#                                            code when using projects
# @param file_sync_poll_interval [Integer] The interval, in seconds, at which a file sync client should poll
#                                          the storage server


class puppet_enterprise::profile::plan_executor(
  Boolean $enable = true,
  String $orchestrator_url = $puppet_enterprise::orchestrator_url,
  Optional[String] $environments_codedir = undef,
  Optional[String] $basemodulepath = undef,
  Optional[String] $environmentpath = undef,
  Boolean $enable_file_sync_client = $puppet_enterprise::profile::master::code_manager_auto_configure,
  Optional[Boolean] $enable_file_sync_locking = undef,
  Boolean $versioned_deploys = false,
  String $file_sync_data_dir = '/opt/puppetlabs/server/data/orchestration-services/data-dir',
  String $projects_codedir = '/opt/puppetlabs/server/data/orchestration-services/projects',
  Array $builtin_content_dir = $puppet_enterprise::params::builtin_bolt_content_dir,
  Integer $file_sync_poll_interval = 5,
  Optional[Boolean] $file_sync_enable_forceful_sync = undef,
  Integer $file_sync_stream_file_threshold = 1024,
  String $gem_path = '/opt/puppetlabs/server/apps/bolt-server/lib/ruby',
  String $gem_home = '/opt/puppetlabs/server/apps/bolt-server/lib/ruby'
) inherits puppet_enterprise::params {
  $container = 'orchestration-services'

  puppet_enterprise::deprecated_parameter { 'puppet_enterprise::profile::plan_executor::file_sync_enable_forceful_sync': }
  puppet_enterprise::deprecated_parameter { 'puppet_enterprise::profile::plan_executor::enable_file_sync_locking': }

  #########################
  # Retire extinct ruby service
  #
  # Previously, there was an option between two types of plan executor: one written in jruby and
  # one written in ruby. The ruby option is extinct so we need to ensure there's no leftover service
  # or service files
  $ruby_service_name = 'pe-plan-executor'
  service { $ruby_service_name :
    ensure => stopped,
    enable => false,
  }

  $_ruby_service_file = {
    ensure  => absent,
    require => Service[$ruby_service_name],
  }

  file { "/etc/logrotate.d/${ruby_service_name}":
    ensure  => absent,
  }

  $_sysvinit = ($facts['os']['family'] == 'RedHat' and Integer($facts['os']['release']['major']) <= 6) or
    ($facts['os']['family'] == 'Suse' and Integer($facts['os']['release']['major']) <= 11)

  if $_sysvinit {
    file { "/etc/rc.d/init.d/${ruby_service_name}":
      ensure  => absent,
    }

    file { "/etc/sysconfig/${ruby_service_name}":
      ensure  => absent,
    }
  } else {
    $_systemd_service_file = $facts['os']['family'] ? {
      'Debian' => "/lib/systemd/system/${ruby_service_name}.service",
      default  => "/usr/lib/systemd/system/${ruby_service_name}.service",
    }

    file { $_systemd_service_file:
      ensure  => absent,
    }
  }

  ###################################
  # Hocon and bootstrap configuration
  # settings

  Pe_hocon_setting {
    ensure => present,
    path   => "/etc/puppetlabs/${container}/conf.d/orchestrator.conf",
    notify => Service["pe-${container}"]
  }

  if $enable {
    pe_hocon_setting { 'orchestrator.url':
      value => "${orchestrator_url}/orchestrator/v1",
    }

    pe_hocon_setting { 'orchestrator.plan-executor-enable':
      value   => 'jruby',
    }

    pe_hocon_setting { 'orchestrator.jruby.gem-home':
      value  => $gem_home,
    }

    pe_hocon_setting { 'orchestrator.jruby.gem-path':
      value  => $gem_path,
    }

    pe_hocon_setting { 'orchestrator.jruby.ruby-load-path':
      type   => 'array',
      value  => [
        '/opt/puppetlabs/server/apps/bolt-server/lib/ruby',
      ],
    }

    # The following three settings are _intentionally obfuscated_ from the user. These
    # puppet vars can be found in the Console UI with a little digging, but we intentionally
    # do not want them to look like default settings that are meant to be used. So the following
    # code ensures that if these settings are set in puppet code they show up in the orchestrator
    # configuration, but if they are not set these settings aren't shown at all in the conf files.
    #
    # All default values are designed to work correctly together to allow bolt content to
    # be found. These settings are made available in the rare case a user cannot use
    # code manager/file sync and they wish to point orchestrator at a specific location for their
    # code.
    #
    # USING THESE SETTINGS TO AVOID CODE MANAGER/FILE SYNC IS HIGHLY DISCOURAGED. It should work
    # but we need to ensure that users who ignore code manager/file sync understand it could be
    # potentially damaging to their infrastructure if they change code while running a plan.
    #
    # ---------------------------------------------------------------------
    if $environments_codedir {
      $codedir_ensure = present
    } else {
      $codedir_ensure = absent
    }

    pe_hocon_setting { 'orchestrator.plan_executor.codedir':
      ensure  => $codedir_ensure,
      setting => 'orchestrator.plan_executor.codedir',
      value   => $environments_codedir,
    }

    if $basemodulepath {
      $basemodulepath_ensure = present
    } else {
      $basemodulepath_ensure = absent
    }

    pe_hocon_setting { 'orchestrator.plan_executor.basemodulepath':
      ensure  => $basemodulepath_ensure,
      setting => 'orchestrator.plan_executor.basemodulepath',
      value   => $basemodulepath,
    }

    if $environmentpath {
      $environmentpath_ensure = present
    } else {
      $environmentpath_ensure = absent
    }

    pe_hocon_setting { 'orchestrator.plan_executor.environmentpath':
      ensure  => $environmentpath_ensure,
      setting => 'orchestrator.plan_executor.environmentpath',
      value   => $environmentpath,
    }
    # ----------------------------------------------------------------------- End obfuscated settings

    pe_hocon_setting { 'orchestrator.plan_executor.builtin-content-dir':
      setting => 'orchestrator.plan_executor.builtin-content-dir',
      type    => 'array',
      value   => $builtin_content_dir,
    }

    pe_hocon_setting { 'orchestrator.plan_executor.projects-dir':
      setting => 'orchestrator.plan_executor.projects-dir',
      value   => $projects_codedir,
    }

    pe_hocon_setting { 'orchestrator.enable-file-sync-client':
      setting => 'orchestrator.enable-file-sync-client',
      value => $enable_file_sync_client,
    }


    $confdir = "/etc/puppetlabs/${container}/conf.d"
    $client_certname = $puppet_enterprise::puppet_master_host
    $master_host = $puppet_enterprise::puppet_master_host
    $master_port = $puppet_enterprise::puppet_master_port

    if $enable_file_sync_client {
      file { "${confdir}/file-sync.conf":
        ensure => present,
        owner  => "pe-${container}",
        group  => "pe-${container}",
        mode   => '0640',
      }

      pe_hocon_setting { "${container}.web-router-service.file-sync-web-service":
        path    => "${confdir}/web-routes.conf",
        setting => 'web-router-service."puppetlabs.enterprise.services.file-sync-web-service.file-sync-web-service/file-sync-web-service"',
        value   => {
          'route' => '/file-sync',
          'server' => 'orchestrator',
        }
      }

      pe_hocon_setting { "${container}.file-sync.data-dir":
        path    => "${confdir}/file-sync.conf",
        setting => 'file-sync.data-dir',
        value   => $file_sync_data_dir,
      }

      pe_hocon_setting { "${container}.file-sync.client.client-identifier":
        path    => "${confdir}/file-sync.conf",
        setting => 'file-sync.client.client-identifier',
        value   => "${client_certname}-orchestrator",
      }

      pe_hocon_setting { "${container}.file-sync.client.poll-interval":
        path    => "${confdir}/file-sync.conf",
        setting => 'file-sync.client.poll-interval',
        value   => $file_sync_poll_interval,
      }

      pe_hocon_setting { "${container}.file-sync.client.server-api-url":
        path    => "${confdir}/file-sync.conf",
        setting => 'file-sync.client.server-api-url',
        value   => "https://${master_host}:${master_port}/file-sync/v1",
      }

      pe_hocon_setting { "${container}.file-sync.client.server-repo-url":
        path    => "${confdir}/file-sync.conf",
        setting => 'file-sync.client.server-repo-url',
        value   => "https://${master_host}:${master_port}/file-sync-git",
      }

      pe_hocon_setting { "${container}.file-sync.client.ssl-ca-cert":
        path    => "${confdir}/file-sync.conf",
        setting => 'file-sync.client.ssl-ca-cert',
        value   => $puppet_enterprise::params::localcacert,
      }

      pe_hocon_setting { "${container}.file-sync.client.ssl-cert":
        path    => "${confdir}/file-sync.conf",
        setting => 'file-sync.client.ssl-cert',
        value   => "/etc/puppetlabs/${container}/ssl/${client_certname}.cert.pem",
      }

      pe_hocon_setting { "${container}.file-sync.client.ssl-key":
        path    => "${confdir}/file-sync.conf",
        setting => 'file-sync.client.ssl-key',
        value   => "/etc/puppetlabs/${container}/ssl/${client_certname}.private_key.pem",
      }

      # Setting is deprecated and no longer respected
      pe_hocon_setting { "${container}.file-sync.client.enable-forceful-sync":
        ensure  => absent,
        path    => "${confdir}/file-sync.conf",
        setting => 'file-sync.client.enable-forceful-sync',
      }

      pe_hocon_setting { "${container}.file-sync.client.stream-file-threshold":
        path    => "${confdir}/file-sync.conf",
        setting => 'file-sync.client.stream-file-threshold',
        value   => $file_sync_stream_file_threshold,
      }

      if $environments_codedir {
        $_fs_livedir = $environments_codedir
      } else {
        $_fs_livedir = '/opt/puppetlabs/server/data/orchestration-services/code'
      }

      pe_hocon_setting { "${container}.file-sync.repos.puppet-code.live-dir":
        path    => "${confdir}/file-sync.conf",
        setting => 'file-sync.repos.puppet-code.live-dir',
        value   => $_fs_livedir,
      }

      # Enabling versioned deploys will:
      # 1. enable the versioned dirs setting in filesync
      # 2. disable locking in orchestrator
      #
      # If versioned deploys is false (the default):
      # 1. the versioned dirs setting should be absent
      # 2. enable locking in orchestrator
      if $versioned_deploys {
        pe_hocon_setting { "${container}.file-sync.repos.puppet-code.versioned-dirs":
          path    => "${confdir}/file-sync.conf",
          setting => 'file-sync.repos.puppet-code.versioned-dirs',
          value   => true,
        }
        pe_hocon_setting { 'orchestrator.enable-file-sync-locking':
          setting => 'orchestrator.enable-file-sync-locking',
          value => false,
        }
      } else {
        pe_hocon_setting { "${container}.file-sync.repos.puppet-code.versioned-dirs":
          path    => "${confdir}/file-sync.conf",
          setting => 'file-sync.repos.puppet-code.versioned-dirs',
          ensure  => absent,
        }
        pe_hocon_setting { 'orchestrator.enable-file-sync-locking':
          setting => 'orchestrator.enable-file-sync-locking',
          value => true,
        }
      }

      pe_hocon_setting { "${container}.file-sync.repos.projects.versioned-dirs":
        path    => "${confdir}/file-sync.conf",
        setting => 'file-sync.repos.projects.versioned-dirs',
        value   => true,
      }

      pe_hocon_setting { "${container}.file-sync.repos.projects.use-external-sha":
        path    => "${confdir}/file-sync.conf",
        setting => 'file-sync.repos.projects.use-external-sha',
        value   => true,
      }

      pe_hocon_setting { "${container}.file-sync.repos.projects.only-submodules":
        ensure  => absent,
        path    => "${confdir}/file-sync.conf",
        setting => 'file-sync.repos.projects.only-submodules',
      }

      pe_hocon_setting { "${container}.file-sync.repos.projects.live-dir":
        path    => "${confdir}/file-sync.conf",
        setting => 'file-sync.repos.projects.live-dir',
        value   => $projects_codedir,
      }

      pe_hocon_setting { "${container}.file-sync.repos.projects.live-dir-mode":
        path    => "${confdir}/file-sync.conf",
        setting => 'file-sync.repos.projects.live-dir-mode',
        value   => 'all',
      }

      puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:file-sync web-service" :
        container => $container,
        namespace => 'puppetlabs.enterprise.services.file-sync-web-service.file-sync-web-service',
        service   => 'file-sync-web-service',
      }

      puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:file-sync client-service" :
        container => $container,
        namespace => 'puppetlabs.enterprise.services.file-sync-client.file-sync-client-service',
        service   => 'file-sync-client-service',
      }
    } else {
      file { "${confdir}/file-sync.conf":
        ensure => absent,
      }

      pe_hocon_setting { "${container}.web-router-service.file-sync-web-service":
        ensure  => absent,
        path    => "${confdir}/web-routes.conf",
        setting => 'web-router-service."puppetlabs.enterprise.services.file-sync-web-service.file-sync-web-service/file-sync-web-service"',
      }
    }
  } else {
    # Make sure the plan executor is disabled
    pe_hocon_setting { 'orchestrator.plan-executor-enable':
      value  => false
    }
  }
}
