# This class can be used to setup and install PuppetDB on a node
#
# This class should not be called directly, but rather is used by the profiles.
# For more information, see the [README.md](./README.md)
#
# @param cert_whitelist_path * Deprecated * Use cert_allowlist_path instead.
# @param cert_allowlist_path The file where the puppetdb certificate whitelist file will
#        be written. Note: this is a required parameter, but to maintain functionality
#        with users using the deprecated cert_whitelist_path parameter, it is set to undef
#        by default.
# @param database_host The hostname of the database that PuppetDB will be running on
# @param certname Name of a certificate Postgres will use for encrypting network traffic
# @param confdir The path to PuppetDB's confdir
# @param database_name The name of the PuppetDB Database
# @param database_password The password of the user
# @param database_port The port that the database is running on
# @param database_user The user logging into the database
# @param gc_interval The interval, in minutes, at which garbage collection should occur
# @param node_purge_ttl The amount of time that must elapse before a deactivated node is
#        purged from PuppetDB
# @param node_ttl The amount of time that must elapse before a node is deactivated from
#        PuppetDB
# @param report_ttl The amount of time that must elapse before a report is deleted
# @param listen_address The address which the database is listening on for plain text
#        connections
# @param listen_port The port which PuppetDB Listens on for plain text connections
# @param ssl_listen_address The address which the database is listening on for SSL
#        connections
# @param ssl_listen_port The port which PuppetDB Listens on for SSL connections
# @param localcacert The path to the local CA certificate
# @param database_properties A url encoded string of JDBC options. This will replace the
#        default database property string which enables SSL connections.
# @param command_processing_threads The number of threads processing the command queue.
# @param java_args A hash containing Java options that puppetdb will run with
# @param concurrent_writes The number of threads allowed to write to disk at any one time.
# @param auto_configure_sync Set this to true to automatically create a puppetdb sync.ini
#        configured to sync with all other configured PuppetDB nodes.
# @param read_maximum_pool_size this value will determine the maximum number
#        of actual connections to the database backend for read connections
# @param write_maximum_pool_size This value will determine the maximum number
#        of actual connections to the database backend for write connections
# @param read_database_name The name of the read PuppetDB Database
# @param read_database_password The password of the user for the read PuppetDB database
# @param read_database_port The port that the read database is running on
# @param read_database_user The user logging into the read database
# @param read_database_host The hostname of the read database that PuppetDB will be running on
# @param read_database_properties A url encoded string of JDBC options. This will replace the
#        default read database property string which enables SSL connections.
# @param rbac_url The URL of the RBAC service.
# @param syncing [Boolean] Toggles the generation of the sync.ini configuration. Defaults
#        to true. If false, sync.ini is removed and the instance won't synchronize with
#        sync_peers.
# @param flight_recorder_enabled [Boolean] Enable/disable Java Flight Recorder for all Java services
# @param flight_recorder_settings [String] Settings value for JFR. The 'default' setting is low overhead, 'profile' gives more data.
# @param flight_recorder_stackdepth [Integer] Number of stack frames to capture. Default is 128 to account for Clojure/JRuby's deeply nested function calls.
# @param flight_recorder_maxage [String] Maximum amount of time to keep recorded data for each service. Can be specified by s (seconds), m (minutes), h (hours), or d (days) suffixes.
# @param flight_recorder_maxsize [String] Maximum size of recorded data files for each service. Can be specified with k (kilobytes), m (megabytes) and g (gigabytes) suffixes.
# @param flight_recorder_repository [String] Directory to save JFR chunks while recording.
# @param flight_recorder_filename [String] File to save completed recording.
class puppet_enterprise::puppetdb(
  Optional[Puppet_enterprise::Unixpath]  $cert_whitelist_path = undef,
  Puppet_enterprise::Unixpath     $cert_allowlist_path        = $cert_whitelist_path,
  String                          $database_host,
  String                          $certname                   = $facts['clientcert'],
  Puppet_enterprise::Unixpath     $confdir                    = $puppet_enterprise::params::puppetdb_confdir,
  String                          $database_name              = $puppet_enterprise::params::puppetdb_database_name,
  Optional[String]                $database_password          = undef,
  Optional[String]                $migrator_password          = undef,
  Puppet_enterprise::Port         $database_port              = $puppet_enterprise::database_port,
  String                          $database_user              = $puppet_enterprise::params::puppetdb_database_user,
  String                          $database_read_user         = $puppet_enterprise::params::puppetdb_database_read_user,
  String                          $migrator_user              = $puppet_enterprise::params::puppetdb_migrator_user,
  Variant[Pattern[/^[0-9]+$/],Integer] $gc_interval           = $puppet_enterprise::params::puppetdb_gc_interval,
  Optional[String]                $node_purge_ttl             = undef,
  Puppet_enterprise::Puppetdb_ttl $node_ttl                   = $puppet_enterprise::params::puppetdb_node_ttl,
  Puppet_enterprise::Puppetdb_ttl $report_ttl                 = $puppet_enterprise::params::puppetdb_report_ttl,
  Variant[Puppet_enterprise::Ip, Puppet_enterprise::Localhost] $listen_address = $puppet_enterprise::params::plaintext_address,
  Puppet_enterprise::Port         $listen_port                = $puppet_enterprise::params::puppetdb_listen_port,
  Puppet_enterprise::Ip           $ssl_listen_address         = $puppet_enterprise::params::ssl_address,
  Puppet_enterprise::Port         $ssl_listen_port            = $puppet_enterprise::params::puppetdb_ssl_listen_port,
  Puppet_enterprise::Unixpath     $localcacert                = $puppet_enterprise::params::localcacert,
  String                          $database_properties        = '',
  Integer                         $command_processing_threads = puppet_enterprise::calculate_puppetdb_command_processing_threads(),
  Hash                            $java_args                  = $puppet_enterprise::params::puppetdb_java_args,
  Optional[Integer]               $read_maximum_pool_size     = puppet_enterprise::calculate_puppetdb_read_maximum_pool_size(
                                                                  $command_processing_threads),
  Optional[Integer]               $write_maximum_pool_size    = $command_processing_threads * 2,
  Optional[Integer]               $concurrent_writes          = pe_clamp(1, $command_processing_threads, 4),
  Optional[Boolean]               $enable_gc_logging          = undef,
  Integer                         $service_stop_retries       = 60,
  Integer                         $start_timeout              = 900,
  $auto_configure_sync                                        = undef, # deprecated
  Optional[
    Array[
      Struct[{
        host => String,
        port => Integer,
        sync_interval_minutes => Integer}]]] $sync_peers      = undef,
  Boolean                         $syncing                    = true,
  String                          $read_database_host         = $database_host,
  String                          $read_database_name         = $database_name,
  Optional[String]                $read_database_password     = $database_password,
  Integer                         $read_database_port         = Integer($database_port),
  String                          $read_database_user         = $database_read_user,
  String                          $read_database_properties   = $database_properties,
  Optional[String]                $rbac_url                   = undef,
  Boolean                         $migrate                    = $puppet_enterprise::params::puppetdb_migrate,
  Boolean $flight_recorder_enabled                            = $puppet_enterprise::flight_recorder_enabled,
  String $flight_recorder_settings                            = $puppet_enterprise::flight_recorder_settings,
  Integer $flight_recorder_stackdepth                         = $puppet_enterprise::flight_recorder_stackdepth,
  String $flight_recorder_maxage                              = $puppet_enterprise::flight_recorder_maxage,
  String $flight_recorder_maxsize                             = $puppet_enterprise::flight_recorder_maxsize,
  String $flight_recorder_repository                          = '/var/log/puppetlabs/puppetdb/jfr',
  String $flight_recorder_filename                            = '/var/log/puppetlabs/puppetdb/last_recording.jfr',
) inherits puppet_enterprise::params {

  puppet_enterprise::deprecated_parameter{ 'puppet_enterprise::puppetdb::cert_whitelist_path': replacement => 'puppet_enterprise::puppetdb::cert_allowlist_path' }

  include puppet_enterprise::packages
  $container = 'puppetdb'
  Package <| tag == 'pe-puppetdb-packages' |>

  #database.ini
  class { 'puppet_enterprise::puppetdb::database_ini':
    confdir             => $confdir,
    database_host       => $database_host,
    database_name       => $database_name,
    database_password   => $database_password,
    migrator_password   => $migrator_password,
    database_port       => $database_port,
    database_user       => $database_user,
    migrator_user       => $migrator_user,
    database_properties => $database_properties,
    gc_interval         => $gc_interval,
    node_purge_ttl      => $node_purge_ttl,
    node_ttl            => $node_ttl,
    report_ttl          => $report_ttl,
    write_maximum_pool_size => $write_maximum_pool_size,
    migrate             => $migrate,
  }

  #read-database.ini
  puppet_enterprise::puppetdb::shared_database_settings { 'read-database' :
    confdir             => $confdir,
    database_host       => $read_database_host,
    database_name       => $read_database_name,
    database_password   => $read_database_password,
    database_port       => $read_database_port,
    database_user       => $read_database_user,
    database_properties => $read_database_properties,
    maximum_pool_size   => $read_maximum_pool_size,
  }

  class { 'puppet_enterprise::puppetdb::jetty_ini':
    cert_allowlist_path => $cert_allowlist_path,
    confdir             => $confdir,
    listen_address      => $listen_address,
    listen_port         => $listen_port,
    ssl_listen_address  => $ssl_listen_address,
    ssl_listen_port     => $ssl_listen_port,
    localcacert         => $localcacert,
  }

  class { 'puppet_enterprise::puppetdb::rbac_consumer_conf':
    certname    => $certname,
    confdir     => $confdir,
    localcacert => $localcacert,
    rbac_url    => $rbac_url,
  }

  if($syncing and $sync_peers) {
    class { 'puppet_enterprise::puppetdb::sync_ini':
      peers   => $sync_peers,
      confdir => $confdir,
    }
  } else {
    file { "${confdir}/sync.ini":
      ensure  => absent,
      require => Package['pe-puppetdb'],
      notify  => Service['pe-puppetdb'],
    }
  }

  class { 'puppet_enterprise::puppetdb::config_ini' :
    confdir                    => $confdir,
    command_processing_threads => $command_processing_threads,
    concurrent_writes          => $concurrent_writes,
  }

  class { 'puppet_enterprise::puppetdb::service': }

  file { '/var/log/puppetlabs/puppetdb':
    ensure  => directory,
    owner   => 'pe-puppetdb',
    group   => 'pe-puppetdb',
    mode    => '0640',
    require => Package['pe-puppetdb']
  }

  file { '/var/log/puppetlabs/puppetdb/puppetdb.log':
    ensure  => present,
    owner   => 'pe-puppetdb',
    group   => 'pe-puppetdb',
    mode    => '0640',
    require => Package['pe-puppetdb']
  }

  if $flight_recorder_enabled {
    $_jfr_java_args = puppet_enterprise::jfr_java_args(
      $flight_recorder_repository,
      $flight_recorder_filename,
      $flight_recorder_settings,
      $flight_recorder_stackdepth,
      $flight_recorder_maxage,
      $flight_recorder_maxsize
    )
  } else {
    $_jfr_java_args = {}
  }
  
  $_java_args = pe_merge($java_args, $_jfr_java_args)
  puppet_enterprise::trapperkeeper::java_args { $container :
    java_args => $_java_args,
    require => Package['pe-puppetdb'],
    enable_gc_logging => $enable_gc_logging,
  }

  puppet_enterprise::trapperkeeper::init_defaults { $container :
    service_stop_retries => $service_stop_retries,
    start_timeout        => $start_timeout,
  }
}
