define puppet_enterprise::trapperkeeper::java_args (
  Hash   $java_args,
  String $java_version = $puppet_enterprise::params::java_version,
  String $container   = $title,
  String $initconfdir = $puppet_enterprise::params::defaults_dir,
  Boolean $enable_gc_logging = true,
  String $jdk_security_policy = $puppet_enterprise::params::jdk_security_policy,
  Integer $jdk_ephemeral_dh_key_size = $puppet_enterprise::jdk_ephemeral_dh_key_size,
  Variant[Integer[4,4], Integer[6,6]] $ip_version = $puppet_enterprise::params::ip_version,
){

  $pe_container = "pe-${container}"
  $initconf = "${initconfdir}/${pe_container}"
  $dh_key_size_args = {'Djdk.tls.ephemeralDHKeySize=' => String($jdk_ephemeral_dh_key_size)}

  # The default G1 garbage collector in Java 11 currently causes problems (PE-28206),
  # so we override to use the parallel GC unless a user already explicitly sets the GC
  # to be used.
  $user_gc_present = $java_args and $java_args.any |$arg| { $arg[0] =~ /XX:\+Use.*GC/ }
  if $user_gc_present {
    $pargc_arg = {}
  } else {
    $pargc_arg = { 'XX:+UseParallelGC' => '' }
  }

  if $enable_gc_logging {
    $java_gc_log_file = "/var/log/puppetlabs/${container}/${container}_gc.log"

    if $java_version == '8' {
      $gc_logging_args = { 'XX:+PrintGCDetails' => '',
                'XX:+PrintGCDateStamps'    => '',
                'Xloggc:' => $java_gc_log_file,
                'XX:+UseGCLogFileRotation' => '',
                'XX:NumberOfGCLogFiles='   => '16',
                'XX:GCLogFileSize='        => '64m', }
    } else {
      $gc_logging_args = { "Xlog:gc*:file=${java_gc_log_file}" => ':time,uptime,level,tags:filecount=16,filesize=16m', }
    }
  } else {
    $gc_logging_args = {}
  }

  if $ip_version == 6 {
    $ipv6_args = { 'Djava.net.preferIPv6Addresses=' => 'true' }
  } else {
    $ipv6_args = {}
  }
  if $facts['fips_enabled'] {
    if $java_args['Djava.security.egd'] {
      warning("Attempting to override '-Djava.security.egd' in FIPS mode")
    }

    # the == at the end of this property name ensures that the value is written
    # as -Djava.security.properties==/opt/puppetlabs/share/jdk8-fips-security
    # the == means that this is the *only* security policy loaded by the JVM
    $_java_args = $java_args + $ipv6_args + $pargc_arg + $gc_logging_args + $dh_key_size_args + {
      'Djava.security.properties==' => $jdk_security_policy
    }
  } else {
    $_java_args = $java_args + $ipv6_args + $pargc_arg + $gc_logging_args + $dh_key_size_args
  }

  if !pe_empty($_java_args) {
    $_java_args_override = pe_join(pe_wrap_join_keys_to_values($_java_args, '-', '', ''), ' ')

    pe_ini_setting { "${pe_container}_java_args_override" :
      path              => $initconf,
      key_val_separator => '=',
      section           => '',
      setting           => 'JAVA_ARGS',
      value             => "\"${_java_args_override}\"",
      require           => Package[$pe_container],
      notify            => Exec["${pe_container} service full restart"],
    }
  }
}
