define puppet_enterprise::trapperkeeper::rbac (
  $certname,
  $container                                            = $title,
  $database_host                                        = 'localhost',
  $database_name                                        = $puppet_enterprise::params::rbac_database_name,
  $database_password                                    = undef,
  $database_port                                        = $puppet_enterprise::database_port,
  $database_properties                                  = '',
  $database_user                                        = $puppet_enterprise::rbac_service_regular_db_user,
  $database_migration_user                              = $puppet_enterprise::rbac_service_migration_db_user,
  $localcacert                                          = $puppet_enterprise::params::localcacert,
  Optional[String] $ds_trust_chain                      = undef,
  Optional[Integer] $failed_attempts_lockout            = undef,
  Optional[Integer] $account_expiry_check_minutes       = undef,
  Optional[Integer] $account_expiry_days                = undef,
  $group                                                = "pe-${title}",
  Optional[Integer] $password_reset_expiration          = undef,
  Optional[Integer] $session_timeout                    = undef,
  String            $token_auth_lifetime,
  Optional[String] $token_maximum_lifetime              = '10y',
  $user                                                 = "pe-${title}",
  # host and port used for the publicly available SAML endpoints metadata url formation
  String $saml_host                                     = $puppet_enterprise::console_host,
  Integer $saml_port                                    = $puppet_enterprise::params::console_ssl_listen_port,
  # number of days until expiration for the creation of the saml cert
  Integer $saml_ssl_cert_ttl                            = 824,
  Puppet_enterprise::Replication_mode $replication_mode = 'none',
  Integer $number_of_previous_passwords                 = $puppet_enterprise::number_of_previous_passwords,
  Integer $special_characters_required                  = $puppet_enterprise::special_characters_required,
  Integer $lowercase_letters_required                   = $puppet_enterprise::lowercase_letters_required,
  Integer $password_minimum_length                      = $puppet_enterprise::password_minimum_length,
  Integer $login_minimum_length                         = $puppet_enterprise::login_minimum_length,
  Boolean $username_substring_match                     = $puppet_enterprise::username_substring_match,
  Integer $substring_character_limit                    = $puppet_enterprise::substring_character_limit,
  Integer $letters_required                             = $puppet_enterprise::letters_required,
  Integer $numbers_required                             = $puppet_enterprise::numbers_required,
  Integer $uppercase_letters_required                   = $puppet_enterprise::uppercase_letters_required,
  Integer $password_hash_output_size                    = $puppet_enterprise::password_hash_output_size,
  Integer $password_algorithm_parallelism               = $puppet_enterprise::password_algorithm_parallelism,
  Integer $password_algorithm_memory_in_kb              = $puppet_enterprise::password_algorithm_memory_in_kb,
  Integer $number_of_iterations                         = $puppet_enterprise::number_of_iterations,
  Integer $password_salt_size_bytes                     = $puppet_enterprise::password_salt_size_bytes,
  String  $password_algorithm                           = $puppet_enterprise::password_algorithm,
  Integer $ldap_sync_period_seconds                     = $puppet_enterprise::ldap_sync_period_seconds,
) {

  $cert_dir = "${puppet_enterprise::server_data_dir}/${container}/certs"
  $ssl_key = "${cert_dir}/${certname}.private_key.pem"
  $ssl_cert =  "${cert_dir}/${certname}.cert.pem"
  $saml_private_key = "${cert_dir}/saml-cert.private_key.pem"
  $saml_cert =  "${cert_dir}/saml-cert.cert.pem"
  $confdir = "/etc/puppetlabs/${container}/conf.d"

  File {
    owner => $user,
    group => $group,
    mode  => '0640',
  }

  file { "${confdir}/secrets":
    ensure => directory,
    owner  => $user,
    group  => $group,
    mode   => '0640',
  }

  Pe_hocon_setting {
    ensure  => present,
    notify  => Service["pe-${container}"],
  }

  # Don't enable or configure the saml settings on replicas
  if $replication_mode != 'replica' {
    $puppetserver_cmd = "${puppet_enterprise::puppetlabs_bin_dir}/puppetserver"

    exec { 'generate saml cert':
      command => "${puppetserver_cmd} ca generate --certname saml-cert --ttl ${saml_ssl_cert_ttl}d --subject-alt-names ${saml_host}",
      creates => '/etc/puppetlabs/puppet/ssl/certs/saml-cert.pem',
      require => Service['pe-puppetserver'],
      before  => Service["pe-${container}"],
    }

    file { $saml_cert:
      ensure    => present,
      source    => "${puppet_enterprise::params::ssl_dir}/certs/saml-cert.pem",
      mode      => '0400',
      show_diff => false,
      require   => Exec['generate saml cert'],
      notify    => [ Service["pe-${container}"] ],
    }

    file { $saml_private_key:
      ensure    => present,
      source    => "${puppet_enterprise::params::ssl_dir}/private_keys/saml-cert.pem",
      mode      => '0400',
      show_diff => false,
      require   => Exec['generate saml cert'],
      notify    => [ Service["pe-${container}"] ],
    }

    # add entries for the SAML cert configuration
    pe_hocon_setting { "${container}.rbac.saml-key":
      path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
      setting => 'rbac.saml-key',
      value   => $saml_private_key,
    }
    pe_hocon_setting { "${container}.rbac.saml-cert":
      path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
      setting => 'rbac.saml-cert',
      value   => $saml_cert,
    }
  }

  $cert_allowlist_path = "/etc/puppetlabs/${container}/rbac-certificate-allowlist"
  # Uses
  #   $ssl_key
  #   $ssl_cert
  #   $cert_allowlist_path
  file { "/etc/puppetlabs/${container}/conf.d/rbac.conf":
    ensure => present,
  }
  pe_hocon_setting { "${container}.rbac.certificate-allowlist":
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.certificate-allowlist',
    value   => $cert_allowlist_path,
  }
  pe_hocon_setting { "${container}.rbac.token-private-key":
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.token-private-key',
    value   => $ssl_key,
  }
  pe_hocon_setting { "${container}.rbac.token-public-key":
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.token-public-key',
    value   => $ssl_cert,
  }

  # Configure the SSL settings to enable SSL when communicating with puppet server
  pe_hocon_setting { "${container}.rbac.ssl-key":
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.ssl-key',
    value   => $ssl_key,
  }
  pe_hocon_setting { "${container}.rbac.ssl-cert":
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.ssl-cert',
    value   => $ssl_cert,
  }
  pe_hocon_setting { "${container}.rbac.ssl-ca-cert":
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.ssl-ca-cert',
    value   => $localcacert,
  }

  pe_hocon_setting {"${container}.rbac.token-maximum-lifetime":
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.token-maximum-lifetime',
    value   => $token_maximum_lifetime,
  }

  if $password_reset_expiration {
    $password_reset_expiration_ensure = present
  } else {
    $password_reset_expiration_ensure = absent
  }

  pe_hocon_setting { "${container}.rbac.password-reset-expiration":
    ensure  => $password_reset_expiration_ensure,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.password-reset-expiration',
    value   => $password_reset_expiration,
  }

  pe_hocon_setting { "${container}.rbac.session-timeout":
    ensure  => absent,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.session-timeout',
  }

  if $token_auth_lifetime and ! pe_empty($token_auth_lifetime) {
    pe_validate_re($token_auth_lifetime, '^[0-9]+[smhdy]?$', '$token_auth_lifetime must either be an integer or a string of digits optionally followed by "s", "m", "h", "d", or "y".')
    $token_auth_lifetime_ensure = present
  } else {
    $token_auth_lifetime_ensure = absent
  }

  pe_hocon_setting { "${container}.rbac.token-auth-lifetime":
    ensure  => $token_auth_lifetime_ensure,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.token-auth-lifetime',
    value   => $token_auth_lifetime,
  }

  if $ds_trust_chain and !pe_empty($ds_trust_chain) {
    $ds_trust_chain_ensure = present
  } else {
    $ds_trust_chain_ensure = absent
  }

  pe_hocon_setting { "${container}.rbac.ds-trust-chain":
    ensure  => $ds_trust_chain_ensure,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.ds-trust-chain',
    value   => $ds_trust_chain,
  }

  if $failed_attempts_lockout {
    $failed_attempts_lockout_ensure = present
  } else {
    $failed_attempts_lockout_ensure = absent
  }

  pe_hocon_setting { "${container}.rbac.failed-attempts-lockout":
    ensure  => $failed_attempts_lockout_ensure,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.failed-attempts-lockout',
    value   => $failed_attempts_lockout,
  }

  if $account_expiry_check_minutes {
    $account_expiry_check_minutes_ensure = present
  } else {
    $account_expiry_check_minutes_ensure = absent
  }

  pe_hocon_setting { "${container}.rbac.account-expiry-check-minutes":
    ensure  => $account_expiry_check_minutes_ensure,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.account-expiry-check-minutes',
    value   => $account_expiry_check_minutes,
  }

  if $account_expiry_days {
    $account_expiry_days_ensure = present
  } else {
    $account_expiry_days_ensure = absent
  }

  pe_hocon_setting { "${container}.rbac.account-expiry-days":
    ensure  => $account_expiry_days_ensure,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.account-expiry-days',
    value   => $account_expiry_days,
  }

  pe_hocon_setting { "${container}.rbac.saml-host":
    ensure  => present,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.saml-host',
    value   => $saml_host,
  }

  pe_hocon_setting { "${container}.rbac.saml-port":
    ensure  => present,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.saml-port',
    value   => $saml_port,
  }

  pe_hocon_setting {"${container}.rbac.local-user-policy.number-of-previous-passwords":
    ensure  => present,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.local-user-policy.number-of-previous-passwords',
    value   => $number_of_previous_passwords,
  }

  pe_hocon_setting {"${container}.rbac.local-user-policy.special-characters-required":
    ensure  => present,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.local-user-policy.special-characters-required',
    value   => $special_characters_required,
  }

  pe_hocon_setting {"${container}.rbac.local-user-policy.lowercase-letters-required":
    ensure  => present,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.local-user-policy.lowercase-letters-required',
    value   => $lowercase_letters_required,
  }

  pe_hocon_setting {"${container}.rbac.local-user-policy.login-minimum-length":
    ensure  => present,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.local-user-policy.login-minimum-length',
    value   => $login_minimum_length,
  }

  pe_hocon_setting {"${container}.rbac.local-user-policy.password-minimum-length":
    ensure  => present,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.local-user-policy.password-minimum-length',
    value   => $password_minimum_length,
  }

  pe_hocon_setting {"${container}.rbac.local-user-policy.username-substring-match":
    ensure  => present,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.local-user-policy.username-substring-match',
    value   => $username_substring_match,
  }

  pe_hocon_setting {"${container}.rbac.local-user-policy.substring-character-limit":
    ensure  => present,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.local-user-policy.substring-character-limit',
    value   => $substring_character_limit,
  }

  pe_hocon_setting {"${container}.rbac.local-user-policy.letters-required":
    ensure  => present,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.local-user-policy.letters-required',
    value   => $letters_required,
  }

  pe_hocon_setting {"${container}.rbac.local-user-policy.numbers-required":
    ensure  => present,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.local-user-policy.numbers-required',
    value   => $numbers_required,
  }

  pe_hocon_setting {"${container}.rbac.local-user-policy.uppercase-letters-required":
    ensure  => present,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.local-user-policy.uppercase-letters-required',
    value   => $uppercase_letters_required,
  }

  pe_hocon_setting {"${container}.rbac.password-configuration.output-size":
    ensure  => present,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.password-configuration.output-size',
    value   => $password_hash_output_size,
  }

  pe_hocon_setting {"${container}.rbac.password-configuration.parallelism":
    ensure  => present,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.password-configuration.parallelism',
    value   => $password_algorithm_parallelism,
  }

  pe_hocon_setting {"${container}.rbac.password-configuration.memory-in-kb":
    ensure  => present,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.password-configuration.memory-in-kb',
    value   => $password_algorithm_memory_in_kb,
  }

  pe_hocon_setting {"${container}.rbac.password-configuration.number-of-iterations":
    ensure  => present,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.password-configuration.number-of-iterations',
    value   => $number_of_iterations,
  }

  pe_hocon_setting {"${container}.rbac.password-configuration.size-of-salt":
    ensure  => present,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.password-configuration.size-of-salt',
    value   => $password_salt_size_bytes,
  }

  pe_hocon_setting {"${container}.rbac.password-configuration.algorithm":
    ensure  => present,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.password-configuration.algorithm',
    value   => $password_algorithm,
  }

  pe_hocon_setting { "${container}.rbac.keypath":
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.keypath',
    value   => "${confdir}/secrets",
  }

  pe_hocon_setting {"${container}.rbac.ldap-sync-period-seconds":
    ensure  => present,
    path    => "/etc/puppetlabs/${container}/conf.d/rbac.conf",
    setting => 'rbac.ldap-sync-period-seconds',
    value   => $ldap_sync_period_seconds,
  }

  puppet_enterprise::trapperkeeper::database_settings { 'rbac' :
    container           => $container,
    database_host       => $database_host,
    database_name       => $database_name,
    database_password   => $database_password,
    database_port       => Integer($database_port),
    database_properties => $database_properties,
    database_user       => $database_user,
    migration_user      => $database_migration_user,
    migration_password  => $database_password,
    group               => $group,
    user                => $user,
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:rbac rbac-service" :
    container => $container,
    namespace => 'puppetlabs.rbac.services.rbac',
    service   => 'rbac-service',
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:rbac rbac-storage-service" :
    container => $container,
    namespace => 'puppetlabs.rbac.services.storage.permissioned',
    service   => 'rbac-storage-service',
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:rbac rbac-http-api-service" :
    container => $container,
    namespace => 'puppetlabs.rbac.services.http.api',
    service   => 'rbac-http-api-service',
  }

    puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:rbac rbac-http-saml-service" :
    container => $container,
    namespace => 'puppetlabs.rbac.services.http.saml',
    service   => 'rbac-http-saml-service',
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:rbac activity-reporting-service" :
    container => $container,
    namespace => 'puppetlabs.activity.services',
    service   => 'activity-reporting-service',
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:rbac jetty9-service" :
    container => $container,
    namespace => 'puppetlabs.trapperkeeper.services.webserver.jetty9-service',
    service   => 'jetty9-service',
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:rbac audit-service" :
    container => $container,
    namespace => 'puppetlabs.rbac.services.audit',
    service   => 'audit-service',
  }
}
