# Configure PE for lockless deploys, run puppet to configure,
# deploy the new lockless codedir, and clean up the old codedir
#
# @param primary [TargetSpec] The certname of the primary.
#   NB: the default of "localhost" is only valid when running this plan via
#   bolt, when running via the Orchestrator the certname must be provided.
# @param enable [Boolean] True if this plan should enable lockless
#   deploys, False if it should disable lockless deploys
# @param old_code_directory [Enum['delete', 'save']] Whether or not
#   the old codedir should be deleted or saved. Saved codedirs will be moved
#   to "${codedir}_backup" and should be removed at ones earliest convienence.
plan enterprise_tasks::toggle_lockless_deploys(
  TargetSpec $primary                        = 'localhost',
  Optional[Boolean] $enable                  = true,
  Enum['delete', 'save'] $old_code_directory = 'save',
) {
  $constants = constants()
  $plan_name = 'enable_lockless_deploys'

  $old_codedir = $enable ? {
    true    => '/etc/puppetlabs/code',
    default => '/etc/puppetlabs/puppetserver/code',
  }

  $new_codedir = $enable ? {
    true    => '/etc/puppetlabs/puppetserver/code',
    default => '/etc/puppetlabs/code',
  }

  if $enable {
    enterprise_tasks::message(
      $plan_name,
      'Enabling Lockless Deploys will consume more disk space than before.
      Please review https://puppet.com/docs/pe/latest/lockless-code-deploys.html#system-reqs-lockless-deploys
      and confirm all compilers have enough space on the filesystem where
      `/opt/puppetlabs/server/apps/puppetserver` resides.'
    )
  }

  $primary_target = get_targets($primary)[0]
  enterprise_tasks::test_connection($primary_target)

  $update_group_result = run_task('enterprise_tasks::update_node_group', $primary_target,
    'group_name'       => 'PE Master',
    'class_parameters' => {
      'puppet_enterprise::profile::master' => {
        'versioned_deploys' => $enable,
      },
    },
  ).first()
  $master_group = $update_group_result.value()['group']

  run_task(enterprise_tasks::add_modify_conf_keys,
    $primary_target,
    file    => $constants['pe_conf'],
    hash    => { 'puppet_enterprise::profile::master::versioned_deploys' => $enable },
  )

  $new_compilers = enterprise_tasks::get_nodes_with_profile('pe_compiler')
  $legacy_compilers = enterprise_tasks::get_nodes_with_profile('master')
  $all_compilers = ($new_compilers + $legacy_compilers + $primary).unique
  $non_primary_compilers = $all_compilers - $primary
  $all_fs_agents = get_targets($all_compilers)
  $non_primary_fs_agents = get_targets($non_primary_compilers)

  enterprise_tasks::with_agent_disabled($primary_target) |$agent| {
    enterprise_tasks::message(
      $plan_name,
      "Running puppet on the primary: ${agent} to apply puppet_enterprise::profile::master::versioned_deploys: ${enable}"
    )
    # the puppet run request can fail if the orchestrator is rebooted. Retry the task every 10 seconds for up to 2 minutes
    ctrl::do_until({ 'interval' => 10, 'limit' => 12 }) || {
      run_task('enterprise_tasks::run_puppet', $agent, '_catch_errors' => true).ok()
    }
    enterprise_tasks::message(
      $plan_name,
      "Running puppet on the primary: ${agent} a second time to modify configuration based on puppetdb_query"
    )
    run_task('enterprise_tasks::run_puppet', $agent)
  }

  enterprise_tasks::with_agent_disabled($non_primary_fs_agents) |$agent| {
    enterprise_tasks::message(
      $plan_name,
      "Running puppet on non-primary compiler(s): ${agent} to apply puppet_enterprise::profile::master::versioned_deploys: ${enable}"
    )
    run_task('enterprise_tasks::run_puppet', $agent)
    enterprise_tasks::message(
      $plan_name,
      "Running puppet on non-primary compiler(s): ${agent} a second time to modify configuration based on puppetdb_query"
    )
    run_task('enterprise_tasks::run_puppet', $agent)
  }

  enterprise_tasks::with_agent_disabled($all_fs_agents) |$agent| {
    $deploy_type = $enable ? {
      true  => 'lockless',
      false => 'locking',
    }

    enterprise_tasks::message(
      $plan_name,
      "Committing all previously deployed environments to trigger a ${deploy_type} deploy"
    )

    $environments_results = run_command(
      "find ${old_codedir}/environments -mindepth 1 -maxdepth 1 -type d -print",
      $primary_target,
      _catch_errors => true
    )

    # If there's a failure in setting up lockless deploys we do not
    # want to potentially delete the old codedir as it may be helpful
    # for recovery if the issue(s) escalate.
    if $environments_results.ok {
      $environment_paths = $environments_results.first.value['stdout']
      $lines = split($environment_paths, "\n")
      $environment_names = map($lines) |$line| {
        split($line, '/')[-1]
      }

      run_task(
        'enterprise_tasks::file_sync_commit',
        $primary_target,
        'filesync_host'     => $primary_target.host,
        'environment_names' => $environment_names
      )

      ctrl::sleep(5)

      enterprise_tasks::message(
        $plan_name,
        'Polling the primary for up to 20 minutes until environments have been deployed.'
      )

      $success = run_task('enterprise_tasks::wait_for_env_deploy', $primary_target, 'enable_lockless' => $enable, _catch_errors => true).ok

      if ! $success {
        $msg = "Could not validate environments have been deployed on ${primary_target}. Please manually deploy code before agents check in."
        enterprise_tasks::message($plan_name, $msg)
        fail_plan($msg)
      }
    } else {
      $msg = "Could not find environments to deploy on ${primary_target}. Please manually deploy code before agents check in."
      enterprise_tasks::message($plan_name, $msg)

      fail_plan($msg)
    }

    if $old_code_directory == 'delete' {
      run_command("rm -rf ${old_codedir}", $agent)
    } else {
      $new_old_codedir = "${old_codedir}_backup"
      run_command("rm -rf ${new_old_codedir}", $agent)
      run_command("mv -f ${old_codedir} ${new_old_codedir}", $agent)
      enterprise_tasks::message(
        $plan_name,
        "The old codedir on ${agent} has been archived at `${new_old_codedir}`. Please remove when convenient."
      )
    }

    run_task('enterprise_tasks::run_puppet', $agent)
  }

  $enable_prefix = $enable ? {
    true  => 'en',
    false => 'dis',
  }

  enterprise_tasks::message(
    $plan_name,
    "Successfully ${enable_prefix}abled lockless deploys"
  )
}
