#!/opt/puppetlabs/puppet/bin/ruby
# frozen_string_literal: true
require_relative "../../ruby_task_helper/files/task_helper.rb"
require "open3"

# This task takes an orchestrator job id, finds the process group ID associated with that job and kills the entire group.
# It acts on the files that describe a running or previously run job in pxp-agent's spool directory:
# exitcode  metadata  pid  stderr  stdout
class KillPxpAgentJobProcess < TaskHelper
  PXP_AGENT_SPOOL_PATH = "/opt/puppetlabs/pxp-agent/spool"

  def task(job_id:, **kwargs)
    job_spool_path = File.join(PXP_AGENT_SPOOL_PATH, job_id)
    validate_pxp_agent_job_dir_exists(job_id, job_spool_path)
    validate_job_did_not_exit(job_id, job_spool_path)

    # Job exists and did not exit, proceed
    pid = Integer(File.read(File.join(job_spool_path, "pid")))

    # The pid in the file associated with the orchestrator job is pxp-agent's
    # execution wrapper process, therefore we need the child of that process.
    child_pid = get_child_process(pid)

    # Find the process group id of child pid
    child_pgid = Process.getpgid(child_pid)

    # Kill -9 all processes under process group id by using negative pgid
    { kill_status: Process.kill(9, -child_pgid) }
  end

  def get_child_process(pid)
    # Using pgrep as there is no standard Ruby equivalent
    stdout, stderr, status = Open3.capture3('pgrep', '-P', pid.to_s)
    if status.success?
      validate_pgrep_output(stdout)

      Integer(stdout)
    else
      raise TaskHelper::Error.new("Could not determine PID associated with that job.",
                                  "kill-task/failed-to-determine-pid",
                                  { stderr: stderr })
    end
  end

  def validate_pxp_agent_job_dir_exists(job_id, job_spool_path)
    # If an orchestrator job is run by pxp agent, it should create a directory
    # under the spool path with the name of the job id, e.g. /opt/puppetlabs/pxp-agent/spool/1
    if !File.directory?(File.join(job_spool_path))
      raise TaskHelper::Error.new("Job id #{job_id} does not exist.",
                                  "kill-task/job-id-does-not-exist")
    end
  end

  def validate_job_did_not_exit(job_id, job_spool_path)
    job_exitcode_path = File.join(job_spool_path, "exitcode")
    if File.file?(job_exitcode_path)
      raise TaskHelper::Error.new("Cannot kill process, "\
                                  "job #{job_id} already exited with exit code #{Integer(File.read(job_exitcode_path))}.",
                                  "kill-task/job-already-exited")
    end
  end

  def validate_pgrep_output(stdout)
    if stdout.empty?
      raise TaskHelper::Error.new("Process associated with that job is not running.",
                                  "kill-task/process-not-running")
    elsif stdout.lines.count != 1
      raise TaskHelper::Error.new("Cannot kill process, "\
                                  "did not find the process group id.",
                                  "kill-task/pgid-not-found")
    end
  end
end

if __FILE__ == $PROGRAM_NAME
  KillPxpAgentJobProcess.run
end
