# @summary Sets up bulk_pluginsync artifacts on the primary
#
# This class sets up cron jobs to make sure a tarball and
# a zipfile of plugins are created and placed in Jetty's
# package directory so that frictionless agent install can
# download and extract them before installing puppet-agent.
#
# @example
#   include pe_repo::bulk_pluginsync
class pe_repo::bulk_pluginsync(
  $compile_master_pool_address = pe_pick($pe_repo::compile_master_pool_address,
                                         $pe_repo::master)
){

  include puppet_enterprise::params
  $cache_dir = '/opt/puppetlabs/puppet/cache'
  $jetty_packages_directory = '/opt/puppetlabs/server/data/packages/public'

  $tarball_location = "${jetty_packages_directory}/bulk_pluginsync.tar.gz"
  $tar_command = "tar --directory ${cache_dir} -cf - lib | gzip -n > ${tarball_location}; chmod 644 ${tarball_location}; if ls -Z ${tarball_location} | grep -q unconfined_u; then chcon -u system_u ${tarball_location}; fi"

  $zip_location = "${jetty_packages_directory}/bulk_pluginsync.zip"
  $zip_command = "/opt/puppetlabs/server/bin/GenerateZip -z ${zip_location} -s ${cache_dir}/lib; chmod 644 ${zip_location}; if ls -Z ${zip_location} | grep -q unconfined_u; then chcon -u system_u ${zip_location}; fi"

  $random_minute = fqdn_rand(60)

  cron { 'create tar.gz of pluginsync cache':
    command => $tar_command,
    hour    => '*',
    minute  => $random_minute,
  }

  cron { 'create zip of pluginsync cache':
    command => "sleep 60; ${zip_command}",
    hour    => '*',
    minute  => $random_minute,
  }

  exec { 'create tar.gz of pluginsync cache - onetime':
    path    => $facts['path'],
    command => $tar_command,
    creates => $tarball_location,
  }

  exec { 'create zip of pluginsync cache - onetime':
    path    => $facts['path'],
    command => $zip_command,
    creates => $zip_location,
    require => Package[$puppet_enterprise::params::java_package_name],
  }

  # The require here is to ensure the files are actually created
  # so that the default ensure => file from pe_repo init doesn't
  # create a blank file
  file { [ $tarball_location, $zip_location ] :
    mode    => '0644',
    require => [Exec['create tar.gz of pluginsync cache - onetime'], Exec['create zip of pluginsync cache - onetime']],
  }
}
