# DESCRIPTION
# Internal class for Puppet Enterprise to configure package repositories
#
# USAGE
# This class should be include'd by any puppet_enterprise class which requires
# the package repository to be configured. That's likely to be any class that
# ensures packages present. Example usage:
#
#     include puppet_enterprise::repo
#
# Note: The puppet_enterprise_repo_update tag is used by the upgrade_secondary
# plan to ensure repos are updated on secondary nodes before any package
# resources are applied.
#
# @param pe_ver [String] The PE version string that will be part of the
#   package repository url.
# @param manage [Boolean] Whether or not to manage the repository
#   configuration. Initial installation overrides this. This parameter
#   should not be managed by anything other than the installer.
# @param certname [Optional[String]] Certname of the node where the puppet_enterprise
#   repo is being managed. Should not need to be passed in under most circumstances.
# @param installing [Optional[Boolean]] *Deprecated* had the inverse function
#   of manage, but with a more confusing name.
class puppet_enterprise::repo (
  String  $pe_ver,
  Boolean $manage = true,
  Optional[String] $certname = $facts['clientcert'],
  Optional[Boolean] $installing = undef
) {
  puppet_enterprise::deprecated_parameter { 'puppet_enterprise::repo::installing': }

  # This defaults to the clientcert fact (not so much a real fact, but injected by the agent
  # during fact collection with the value of Puppet[:certname]). In some situations, we are
  # unable to get this fact (e.g. applying this class via a Bolt plan for replica upgrades
  # on RHEL8 for some reason.). At some point, we should determine if we can default to using
  # the trusted fact instead and have it work in all scenarios where this class is applied.
  $_certname = $certname ? {
    undef   => $trusted['certname'],
    default => $certname,
  }
  if $manage {
    include puppet_enterprise
    if ($_certname == $::puppet_enterprise::puppet_master_host) {
      $source = "${::puppet_enterprise::packages_dir}/public"
      $local_config = true
    } else {
      $source = "${::puppet_enterprise::puppet_master_host}:${puppet_enterprise::puppet_master_port}"
      $local_config = false
    }
    class { 'puppet_enterprise::repo::config':
      pe_ver       => $pe_ver,
      tag          => 'puppet_enterprise_repo_update',
      source       => $source,
      local_config => $local_config,
      certname     => $_certname,
    }
    contain puppet_enterprise::repo::config
  }
}
