# Given a path to the directory of an untarred PE tarball, setup the
# /opt/puppetlabs/server/data/packages repository structure for this version of
# PE.
#
# NOTE: This class is not currently included in pe_install::prepare. It can
# be run separately to prep the local primary package repository when given
# the tarball location and bootstrap.metadata. This class replaces a chunk of
# prep work that the puppet-enterprise-installer shim used to perform before
# invoking `puppet-infra upgrade`.
#
# NOTE: It does not generate the package manager's repository configuration.
#
# (See {pe_install::bootstrap} for parameter docs.)
class pe_install::bootstrap::package_repository(
  Pe_install::Absolute_path $pe_tarball_dir,
  Pe_install::Pe_build $pe_build_version,
  String $platform_tag,
  Pe_install::Agent_version $aio_agent_version,
  Pe_install::Agent_build $aio_install_version,
) {
  include 'pe_install::params'

  $public_pkg_dir = "${pe_install::params::server_packages_dir}/public"
  $versioned_pkg_dir = "${public_pkg_dir}/${pe_build_version}"
  $master_pkg_dir = "${versioned_pkg_dir}/${platform_tag}-${aio_agent_version}"
  $pe_pkg_link = "${versioned_pkg_dir}/puppet_enterprise"

  $installer_pkg_path = "${pe_tarball_dir}/packages"
  # Public keys for import into rpm
  $gpg_key = 'GPG-KEY-puppet'
  # A gpg keyring with the same public keys for dropping into /etc/apt/trusted.gpg.d/
  # TODO: This has to be added to the tarball
  # `gpg --no-default-keyring --keyring puppet-enterprise-keyring.gpg --import GPG-KEY-puppet*`
  # will create this file, as an example...
  $gpg_keyring = 'puppet-enterprise-keyring.gpg'

  $file_opts = {
    owner => 'root',
    group => 'root',
    mode  => '0644',
  }

  # For now, assume the bootstrap classes are applied separately, and don't worry
  # about potential file resource conflicts for base directories.
  $package_dirs = [
    $pe_install::params::server_data_dir,
    $pe_install::params::server_packages_dir,
    $public_pkg_dir,
    $versioned_pkg_dir,
  ]
  file { $package_dirs:
    ensure => 'directory',
    *      => $file_opts,
  }

  # Copy Puppet's public GPG keys
  file { "${public_pkg_dir}/${gpg_key}":
    ensure => 'file',
    source => "puppet:///modules/pe_repo/${gpg_key}",
    *      => $file_opts,
  }
  file { "${public_pkg_dir}/${gpg_keyring}":
    ensure => 'file',
    source => "puppet:///modules/pe_repo/${gpg_keyring}",
    *      => $file_opts,
  }


  # Copy the PE package version list
  file { "${versioned_pkg_dir}/pe-package-versions.json":
    ensure => 'file',
    source => "${installer_pkg_path}/${platform_tag}-package-versions.json",
    *      => $file_opts,
  }

  # Copy the primary platform packages and repository metadata
  file { $master_pkg_dir:
    ensure  => 'directory',
    source  => "${installer_pkg_path}/${platform_tag}",
    recurse => true,
    *       => $file_opts,
  }
  file { $pe_pkg_link:
    ensure => 'link',
    target => $master_pkg_dir,
  }

  # Copy the included windows packages
  ['windows', 'windowsfips'].each |$winplatform| {
    # Directory we are building in /opt/puppetlabs/server/data/packages
    $win_pkg_dir = "${versioned_pkg_dir}/${winplatform}-${aio_agent_version}"

    # Directory and msi we are copying from out of the tarball/packages
    $installer_win_pkg_path = "${installer_pkg_path}/${winplatform}-x86_64"
    $installer_win_msi = "${installer_win_pkg_path}/puppet-agent-${aio_install_version}-x64.msi"

    file { $win_pkg_dir:
      ensure => 'directory',
      *      => $file_opts,
    }
    file { "${win_pkg_dir}/puppet-agent-x64.msi":
      ensure => 'file',
      source => $installer_win_msi,
      *      => $file_opts,
    }
  }

  # Add keys
  case $facts['os']['family'] {
    'RedHat','SLES','Suse': {
      [
        [$gpg_key, '9e61ef26'],
      ].each |$pair| {
        $key_name = $pair[0]
        $id = $pair[1]
        $path_to_key = "${public_pkg_dir}/${key_name}"
        exec { "Importing ${key_name} key to rpm":
          command => "rpm --import ${path_to_key}",
          path    => ['/usr/bin', '/usr/sbin'],
          unless  => "rpm -qi gpg-pubkey-${id}",
          require => File[$path_to_key],
        }
      }
    }
    'Debian': {
      $path_to_keyring = "${public_pkg_dir}/${gpg_keyring}"
      file { "/etc/apt/trusted.gpg.d/${gpg_keyring}":
        ensure  => 'file',
        source  => $path_to_keyring,
        require => File[$path_to_keyring],
        *       => $file_opts,
      }
    }
    default: {
      fail("Unable to process gpg release keys for unknown primary platform: ${facts['os']}.")
    }
  }
}
