require 'puppet/agent'
require 'puppet/error'
require 'puppet/util/pe_node_groups'
require 'puppet_x/puppetlabs/meep/enable/interview'
require 'puppet_x/puppetlabs/meep/util'
require 'puppet_x/util/service_status'
require 'puppet_x/util/classification'
require 'puppet_x/util/infrastructure_error'
require 'puppet_x/util/orchestrator'

module PuppetX
module Puppetlabs
module Meep
module Enable

  class Action
    include PuppetX::Puppetlabs::Meep::Util

    # The certificate name for the replica we are provisioning
    attr_accessor :replica_certname

    # The options hash from the enable (or provision with --enable) action
    attr_accessor :options

    # The primary's certificate name.
    attr_accessor :master_certname

    # The array of PE services from /etc/puppetlabs/client/services.conf
    attr_accessor :config

    # The array of PE node roles from /etc/puppetlabs/client/services.conf
    attr_accessor :nodes_config

    # Hash of PE service information for service found running on the
    # +replica_certname+ host.
    attr_accessor :services

    # The RBAC token for accessing PE services.
    attr_accessor :rbac_token

    # The hash containing infrastructure information taken either
    # from flags or from the interview
    attr_accessor :infra_conf

    # A hash returned from the classifier containing all node groups
    attr_accessor :all_groups

    # Common information for constructing log file paths.
    attr_accessor :log_context

    def initialize(replica_certname, log_context, options, config: nil, nodes_config: nil, rbac_token: nil, services: nil, all_groups: nil, infra_conf: nil, classifier_object: nil)
      self.replica_certname = replica_certname
      self.options = options
      self.master_certname = Puppet.settings[:certname].strip

      self.config = config
      self.nodes_config = nodes_config
      self.services = services
      self.all_groups = all_groups
      self.rbac_token = rbac_token
      self.infra_conf = infra_conf
      self.log_context = log_context
      @classifier = classifier_object
    end

    def prep_config
      if options[:print_config]
        required_services = PuppetX::Util::ServiceStatus.replica_services().select {|service, _| service == :classifier}
      else
        required_services = PuppetX::Util::ServiceStatus.primary_services()
      end
      self.config ||= PuppetX::Util::ServiceStatus.load_services_config()
      self.nodes_config ||= PuppetX::Util::ServiceStatus.load_nodes_config()
      self.services ||= PuppetX::Util::ServiceStatus.validate_and_select_services_from_config(config, 'enable', master_certname, required_services)
      self.all_groups ||= classifier.get_groups
      self.rbac_token ||= PuppetX::Util::RBAC.load_token(options[:token_file])
    end

    # @return A Puppet::Util::Pe_node_groups instance initialized with :classifier
    # service information from the +services+ hash. Used to interact with
    # the classifier service API.
    def classifier
      unless @classifier
        nc_service = services[:classifier]
        @classifier = Puppet::Util::Pe_node_groups.new(nc_service[:server], nc_service[:port].to_i, "/#{nc_service[:prefix]}")
      end
      return @classifier
    end

    def print_config
      Puppet.notice(
        _("Current configuration:\n") +
        PuppetX::Util::Classification.print_config(all_groups)
      )
      exit(0)
    end

    def validate
      unless options[:force]
        unless PuppetX::Util::ServiceStatus.node_role(nodes_config, replica_certname) == 'primary_master_replica'
          Puppet.err(_("The node %{replica_certname} does not appear to be provisioned as a replica.") % { replica_certname: replica_certname } + " " + _("Make sure that it has been provisioned with the 'puppet infrastructure provision replica' command and run 'puppet agent -t' on the primary to update service configuration."))
          exit(1)
        end
        unless PuppetX::Util::Orchestrator.node_in_orch_inventory(services[:orchestrator], rbac_token, replica_certname)
          Puppet.err(_("The node %{replica_certname} is not connected to the primary via PCP.") % { replica_certname: replica_certname } + " " + _("Make sure that orchestration services are enabled on the primary and that the PXP agent is running on %{replica_certname}") % { replica_certname: replica_certname })
          exit(1)
        end
      end
    end

    def interview
      self.infra_conf ||= PuppetX::Puppetlabs::Meep::Enable::Interview.enable_interview(replica_certname, master_certname, all_groups, options)
    end

    def classify
      PuppetX::Util::Classification.enable_replica(classifier, all_groups, master_certname, replica_certname, infra_conf, !options[:skip_agent_config])
      Puppet.notice(_("Updated classification"))
    end

    def run_puppet_on(certname, description:, nodes: {:nodes => [certname]}, allow_empty: false)
      task_log_context = log_context.step("#{description}_#{certname}")
      display_scope = nodes.include?(:nodes) ? nil : certname
      PuppetX::Puppetlabs::Meep::Provision::Base.invoke_task(
        'enterprise_tasks::run_puppet',
        orch_config: services[:orchestrator],
        rbac_token: rbac_token,
        display_scope: display_scope,
        nodes: nodes,
        params: { 'max_timeout' => PuppetX::Puppetlabs::Meep::Provision::MAX_RUN_PUPPET_BACKGROUND_TIMEOUT_SECS },
        log_context: task_log_context,
        allow_empty: allow_empty
      )
    end

    # Enforce configuration changes on the primary and replica
    # by running Puppet through the Orchestrator.
    def run_puppet
      run_puppet_on(master_certname, description: 'enable.primary_run')
      run_puppet_on(replica_certname, description: 'enable.replica_run')
      run_puppet_on('all-other-compilers',
        description: 'enable.other_compilers_run',
        allow_empty: true,
        nodes: { :query =>
          ['from', 'resources',
            ['extract', ['certname'],
              ['and',
                ['=', 'type', 'Class'],
                ['=', 'title', 'Puppet_enterprise::Profile::Master'],
                ['not', ['=', 'certname', master_certname]],
                ['not', ['=', 'certname', replica_certname]],
                ['=', ['node', 'active'], true]
              ]
            ]
          ]
        })
    end

    def closing_text
      Puppet.info("#{replica_certname} has been enabled for your Puppet Enterprise infrastructure.")
      unless options[:skip_agent_config]
        Puppet.notice(<<~EOT)

          Agent configuration is now managed with Puppet, and will be updated
          the next time each agent runs to enable replica failover.

          If you wish to immediately run Puppet on all your agents, you can do so
          with this command:

              puppet job run --no-enforce-environment --query 'nodes {deactivated is null and expired is null}'
        EOT
      end
    end

    def setup
      prep_config
      print_config if options[:print_config]
      validate
      interview
    end

    def run(skip_setup: false)
      # The provision action will already pass in the pieces prep_config sets up,
      # validate isn't needed since we just finished provisioning, and the
      # interview would be skipped since infra_conf is passed in
      setup unless skip_setup
      classify
      if skip_setup
        # We are being called on behalf of provision, which has already
        # disabled the agent.
        run_puppet
      else
        PuppetX::Puppetlabs::Meep::Provision::Base.with_agent_disabled_on(
          [master_certname, replica_certname],
          orch_config: services[:orchestrator],
          rbac_token: rbac_token,
          log_context: log_context
        ) do
          run_puppet
        end
      end
      closing_text
    end
  end

end
end
end
end
