require 'puppet_x/util/classifier'
require 'puppet_x/util/classification_error'
require 'hocon'
require 'set'

module PuppetX
  module Util
    class Classification

      # Get a Pe_node_groups instance using the node classifier service
      # (internal)
      def self.get_node_classifier
        nc_service = PuppetX::Util::ServiceStatus.get_service_on_primary('classifier')
        Puppet::Util::Pe_node_groups.new(nc_service[:server], nc_service[:port].to_i, "/#{nc_service[:prefix]}")
      end

      # Find a group with the given name in all_groups; raise an error if it doesn't exist.
      # (internal)
      def self.find_and_check_node_group(all_groups, group_name, required_classes: [])
        group = PuppetX::Util::Classifier.find_group(all_groups, group_name)
        raise PuppetX::Util::NodeGroupNotFoundError.new(group_name) unless group

        required_classes.each do |c|
          raise PuppetX::Util::MissingClassError.new(group_name, c) unless group['classes'][c]
        end

        group
      end

      # Find the id of the group with the given name in all_groups; raise an error if it doesn't exist.
      # (internal)
      def self.find_and_check_node_group_id(all_groups, group_name)
        group_id = PuppetX::Util::Classifier.find_group_id(all_groups, group_name)
        raise PuppetX::Util::NodeGroupNotFoundError.new(group_name) unless group_id
        group_id
      end

      # (internal)
      def self.configure_ha_master_node_group_to_allow_replica(nc, all_groups, replica_certname)
        master_group = find_and_check_node_group(all_groups, 'PE HA Master',
                                                 required_classes: ['puppet_enterprise::profile::database',
                                                                    'puppet_enterprise::profile::puppetdb'])

        replica_hostnames = master_group['classes']['puppet_enterprise::profile::database']['replica_hostnames']
        sync_allowlist = master_group['classes']['puppet_enterprise::profile::puppetdb']['sync_allowlist']

        replica_hostnames << replica_certname
        sync_allowlist << replica_certname

        update_pe_ha_master_group(nc, master_group, replica_hostnames.uniq.sort, sync_allowlist.uniq.sort)
      end

      # (internal)
      def self.configure_ha_master_node_group_to_remove_replica(nc, all_groups, replica_certname)
        pe_ha_master_group = find_and_check_node_group(all_groups, 'PE HA Master',
                                                       required_classes: ['puppet_enterprise::profile::database',
                                                                          'puppet_enterprise::profile::puppetdb'])

        replica_hostnames = pe_ha_master_group['classes']['puppet_enterprise::profile::database']['replica_hostnames']
        sync_allowlist = pe_ha_master_group['classes']['puppet_enterprise::profile::puppetdb']['sync_allowlist']

        replica_hostnames.delete(replica_certname)
        sync_allowlist.delete(replica_certname)

        update_pe_ha_master_group(nc, pe_ha_master_group, replica_hostnames, sync_allowlist)
      end

      # (internal)
      def self.update_pe_ha_master_group(nc, master_group, replica_hostnames, sync_allowlist)
        nc.update_group(
            { :id => master_group['id'],
              :classes => {
                  'puppet_enterprise::profile::master' => {
                      :provisioned_replicas => replica_hostnames
                  },

                  'puppet_enterprise::profile::database' => {
                      :replica_hostnames => replica_hostnames
                  },

                  'puppet_enterprise::profile::puppetdb' => {
                      :sync_allowlist => sync_allowlist
                  }
              }})
      end

      # (internal)
      def self.update_ha_enabled_replicas(nc, pe_infra_group, ha_enabled_replicas)
        begin
          nc.update_group(
            { :id => pe_infra_group['id'],
              :classes => {
                'puppet_enterprise' => {
                  :ha_enabled_replicas => ha_enabled_replicas,
                },
              },
            }
          )
        rescue
          # Workaround FM-6292
          # rubocop:disable GetText/DecorateFunctionMessage
          raise PuppetX::Util::UpdateClassificationParametersError.new(
                  'PE Infrastructure',
                  pe_infra_group['id'],
                  'puppet_enterprise',
                  ['ha_enabled_replicas'])
          # rubocop:enable
        end
      end

      # (internal)
      def self.update_pe_agent_group(nc, pe_agent_group, puppet_server_list, pcp_broker_list, primary_uris)
        begin
          nc.update_group(
            { :id => pe_agent_group['id'],
              :classes => {
                'puppet_enterprise::profile::agent' => {
                    :manage_puppet_conf => true,
                    :server_list        => puppet_server_list,
                    :pcp_broker_ws_uris => nil,
                    :pcp_broker_list    => pcp_broker_list,
                    :primary_uris       => primary_uris,
                },
              },
            }
          )
        rescue
          # Workaround FM-6292
          # rubocop:disable GetText/DecorateFunctionMessage
          raise PuppetX::Util::UpdateClassificationParametersError.new(
                  'PE Agent',
                  pe_agent_group['id'],
                  'puppet_enterprise::profile::agent',
                  ['manage_puppet_conf', 'server_list', 'pcp_broker_list'])
          # rubocop:enable
        end
      end

      def self.update_pe_infra_agent_group(nc, pe_infra_agent_group, puppet_server_list, pcp_broker_list, primary_uris)
        nc.update_group(
          { :id => pe_infra_agent_group['id'],
            :classes => {
              'puppet_enterprise::profile::agent' => {
                :manage_puppet_conf => true,
                :server_list        => puppet_server_list,
                :pcp_broker_ws_uris => nil,
                :pcp_broker_list    => pcp_broker_list,
                :primary_uris       => primary_uris,
              }
            }
          })
      end

      # (internal)
      def self.update_master_profile(nc, pe_master_group, puppetdb_servers, puppetdb_ports, classifier_servers, classifier_ports)
        begin
          nc.update_group(
              { :id => pe_master_group['id'],
                :classes => {
                    'puppet_enterprise::profile::master' => {
                        'puppetdb_host' => puppetdb_servers,
                        'puppetdb_port' => puppetdb_ports,
                        'classifier_host' => classifier_servers,
                        'classifier_port' => classifier_ports,
                        'classifier_client_certname' => classifier_servers,
                    }
                }
              })
        rescue
          # Workaround FM-6292
          # rubocop:disable GetText/DecorateFunctionMessage
          raise PuppetX::Util::UpdateClassificationParametersError.new(
                  'PE Master',
                  pe_master_group['id'],
                  'puppet_enterprise::profile::master',
                  ['puppetdb_host', 'puppetdb_port',
                  'classifier_host', 'classifier_port', 'classifier_client_certname'])
          # rubocop:enable
        end
      end

      # (internal)
      def self.update_pins_for_replica_during_promotion(nc, all_groups, replica_certname, primary_master_group_ids)
        pe_ha_replica_group_id = find_and_check_node_group_id(all_groups, 'PE HA Replica')
        begin
          nc.unpin_nodes_from_group(pe_ha_replica_group_id, [replica_certname])
        rescue
          raise PuppetX::Util::UnpinNodeError.new(
                  'PE HA Replica',
                  pe_ha_replica_group_id,
                  replica_certname)
        end

        primary_master_group_ids.each do |group_id|
          begin
            nc.pin_nodes_to_group(group_id, [replica_certname])
          rescue
            raise PuppetX::Util::PinNodeError.new(
                    all_groups.find { |g| g['id'] == group_id }['name'],
                    group_id,
                    replica_certname)
          end
        end
      end

      # (internal)
      def self.update_puppet_enterprise_classification_for_promotion(nc, pe_infra_group, ha_enabled_replicas, replica_certname)
        begin
          nc.update_group(
            { :id => pe_infra_group['id'],
              :classes => {
                'puppet_enterprise' => {
                  'puppet_master_host' => replica_certname,
                  'puppetdb_host' => replica_certname,
                  'console_host' => replica_certname,
                  'database_host' => replica_certname,
                  'pcp_broker_host' => replica_certname,
                  'certificate_authority_host' => replica_certname,
                  'ha_enabled_replicas' => ha_enabled_replicas,
                },
              },
            }
          )
        rescue
          # Workaround FM-6292
          # rubocop:disable GetText/DecorateFunctionMessage
          raise PuppetX::Util::UpdateClassificationParametersError.new(
                  'PE Infrastructure',
                  pe_infra_group['id'],
                  'puppet_enterprise',
                  ['puppet_master_host', 'puppetdb_host', 'console_host', 'database_host',
                   'pcp_broker_host', 'certificate_authority_host', 'ha_enabled_replicas'])
          # rubocop:enable
        end
      end

      def self.find_primary_master(all_groups)
        pe_infra_group = find_and_check_node_group(all_groups, 'PE Infrastructure',
                                                   required_classes: ['puppet_enterprise'])
        pe_infra_group['classes']['puppet_enterprise']['puppet_master_host']
      end

      def self.find_primary_master_group_ids(nc, all_groups, master_certname)
        master_classification = nil

        begin
          master_classification = nc.get_classification(master_certname)
        rescue
          raise PuppetX::Util::GetClassificationError.new(master_certname)
        end

        master_group_ids = master_classification.empty? ? [] : master_classification['groups']
        default_master_group_ids = ['PE Master',
                                    'PE HA Master',
                                    'PE Console',
                                    'PE Orchestrator',
                                    'PE PuppetDB',
                                    'PE Database',
                                    'PE Certificate Authority'].map { |name| PuppetX::Util::Classifier.find_group_id(all_groups, name) }
        master_group_ids & default_master_group_ids
      end

      def self.get_config(all_groups)
        pe_agent_group = PuppetX::Util::Classifier.find_group(all_groups, 'PE Agent')
        pe_master_group = PuppetX::Util::Classifier.find_group(all_groups, 'PE Master')
        pe_infra_agent_group = PuppetX::Util::Classifier.find_group(all_groups, 'PE Infrastructure Agent')
        agent_list = get_server_list(pe_agent_group)
        infra_agent_list =  get_server_list(pe_infra_agent_group)
        pcp_broker_list =  get_pcp_brokers(pe_agent_group) || []
        infra_pcp_broker_list = get_pcp_brokers(pe_infra_agent_group) || []
        primary_uris = get_primary_uris(pe_agent_group) || []
        infra_primary_uris = get_primary_uris(pe_infra_agent_group) || []

        { agent_list: agent_list,
          infra_agent_list: infra_agent_list,
          pcp_broker_list: pcp_broker_list,
          infra_pcp_broker_list: infra_pcp_broker_list,
          primary_uris_list: primary_uris,
          infra_primary_uris_list: infra_primary_uris,
          puppetdb_list: get_termini(pe_master_group, 'puppetdb'),
          classifier_list: get_termini(pe_master_group, 'classifier'),
          agent_port: get_server_port(agent_list) || 8140,
          infra_agent_port: get_server_port(infra_agent_list) || get_server_port(agent_list) || 8140,
          pcp_broker_port: get_pcp_broker_port(pcp_broker_list) || 8142,
          infra_pcp_broker_port: get_pcp_broker_port(infra_pcp_broker_list) || 8142,
          primary_uris_port: get_server_port(primary_uris) || 8140,
          infra_primary_uris_port: get_server_port(infra_primary_uris) || 8140,
          puppetdb_port: get_termini_port(pe_master_group, 'puppetdb') || 8081,
          classifier_port: get_termini_port(pe_master_group, 'classifier') || 4433,
        }

      end

      def self.setting_output(list_setting)
        if list_setting.empty?
          "not configured for HA"
        else
          "#{list_setting.join(', ')}"
        end
      end

      def self.config_output(conf, configure_agent=true)
        output = []
        output << "  agent-server-urls       #{setting_output(conf[:agent_list])}" if configure_agent
        output << "  infra-agent-server-urls #{setting_output(conf[:infra_agent_list])}"
        output << "  pcp-brokers             #{setting_output(conf[:pcp_broker_list])}" if configure_agent
        output << "  infra-pcp-brokers       #{setting_output(conf[:infra_pcp_broker_list])}"
        output << "  primary-uris            #{setting_output(conf[:primary_uris_list])}" if configure_agent
        output << "  infra-primary-uris      #{setting_output(conf[:infra_primary_uris_list])}"
        output << "  puppetdb-termini        #{setting_output(conf[:puppetdb_list])}"
        output << "  classifier-termini      #{setting_output(conf[:classifier_list])}"

        output.join("\n")
      end

      def self.have_config_to_print?(conf)
        not (conf[:agent_list].empty? and
             conf[:infra_agent_list].empty? and
             conf[:pcp_broker_list].empty? and
             conf[:infra_pcp_broker_list].empty? and
             conf[:primary_uris_list].empty? and
             conf[:infra_primary_uris_list].empty? and
             conf[:puppetdb_list].empty? and
             conf[:classifier_list].empty?)
      end

      def self.print_config(all_groups, configure_agent=true)
        conf = get_config(all_groups)
        config_output(conf, configure_agent)
      end

      def self.suggest_enable_config(conf, master_certname, replica_certname, configure_agent)
        suggest = lambda do |prefix|
          list = conf["#{prefix}_list".to_sym].dup
          port = conf["#{prefix}_port".to_sym]
          list.push("#{master_certname}:#{port}") if list.empty?
          list.push("#{replica_certname}:#{port}") unless list.include?("#{replica_certname}:#{port}")
          list
        end

        settings = ['agent', 'infra_agent', 'pcp_broker', 'infra_pcp_broker', 'primary_uris', 'infra_primary_uris', 'puppetdb', 'classifier']

        if not configure_agent then
          settings -= ['agent', 'pcp_broker', 'primary_uris']
        end

        suggestions = Hash.new
        settings.each do |setting|
          suggestions["#{setting}_list".to_sym] = suggest.call(setting)
        end
        suggestions
      end

      def self.suggest_promote_config(conf, topo, master_certname, replica_certname, configure_agent)
        master_regex = Regexp.new("^#{master_certname}:?\\d*$", Regexp::IGNORECASE)
        suggest = lambda do |prefix|
          list = conf["#{prefix}_list".to_sym].dup
          port = conf["#{prefix}_port".to_sym]
          list = list.reject {|s| s =~ master_regex }
          if topo == :a && list.none? {|s| s == "#{replica_certname}:#{port}" }
            list.push("#{replica_certname}:#{port}")
          end
          list
        end

        settings = ['agent', 'infra_agent', 'pcp_broker', 'infra_pcp_broker', 'primary_uris', 'infra_primary_uris', 'puppetdb', 'classifier']

        if not configure_agent then
          settings -= ['agent', 'pcp_broker', 'primary_uris']
        end

        suggestions = Hash.new
        settings.each do |setting|
          suggestions["#{setting}_list".to_sym] = suggest.call(setting)
        end
        suggestions
      end

      def self.split_hosts_and_ports(server_list)
        servers_and_ports = server_list.map { |pair| pair.split(/:/) }
        servers = servers_and_ports.map(&:first)
        ports = servers_and_ports.map(&:last).map(&:to_i)
        [servers, ports]
      end

      # TODO: HA tech debt. This stinks. We should update the classification
      # functions to work with suggestions. It might also make sense for
      # suggestions/conf to get a class rather then just be free-form maps.
      def self.infra_options_from_conf(conf)
        puppetdb_hosts, puppetdb_ports = split_hosts_and_ports(conf[:puppetdb_list])
        classifier_hosts, classifier_ports = split_hosts_and_ports(conf[:classifier_list])

        {
          agent_server_urls: conf[:agent_list],
          infra_agent_server_urls: conf[:infra_agent_list],
          pcp_broker_list: conf[:pcp_broker_list],
          infra_pcp_broker_list: conf[:infra_pcp_broker_list],
          primary_uris: conf[:primary_uris_list],
          infra_primary_uris: conf[:infra_primary_uris_list],
          :puppetdb_hosts => puppetdb_hosts,
          :puppetdb_ports => puppetdb_ports,
          :classifier_hosts => classifier_hosts,
          :classifier_ports => classifier_ports,
        }
      end

      def self.get_server_port(server_list)
        return nil unless server_list
        return nil if server_list.empty?
        first_server = server_list.first
        return nil unless first_server.respond_to?(:split)
        parts = first_server.split(/:/)
        return nil unless parts.length == 2
        return parts[1].to_i
      end

      def self.get_server_list(group)
        Array(group['classes']['puppet_enterprise::profile::agent']['server_list'])
      end

      def self.get_pcp_brokers(pe_agent_group)
        # TODO: for now pcp_broker_ws_uris are authoritative.
        # we should prefer broker_list once they are deprecated.
        if broker_uris = pe_agent_group['classes']['puppet_enterprise::profile::agent']['pcp_broker_ws_uris']
          Array(broker_uris).map do |uri|
            matcher = uri.match(%r{^wss\://([^\:]+\:\d+)/pcp/$})
            unless matcher
              Puppet.err(_("Error parsing pcp_broker_ws_uris at %{uri}, uri not of the scheme %{scheme}") % { uri: uri, scheme: "wss://<host>:<port>/pcp/" })
            end
            matcher ? matcher.captures.first : nil
          end.compact
        else
          pe_agent_group['classes']['puppet_enterprise::profile::agent']['pcp_broker_list']
        end
      end

      def self.get_pcp_broker_port(pcp_broker_list)
        return nil unless pcp_broker_list
        first_uri = pcp_broker_list.first
        return nil unless first_uri.respond_to?(:match)
        port_matcher = first_uri.match(%r{^[^\:]+\:(\d+)$})
        return nil unless (port_matcher && port_matcher.captures.length > 1)
        port_matcher.captures[0].to_i
      end

      def self.get_primary_uris(pe_agent_group)
        # Pretty sure this always has to be https:// if the prefix is supplied, but allowing for
        # http:// just in case.
        primary_uris = pe_agent_group['classes']['puppet_enterprise::profile::agent']['primary_uris']
        primary_uris = primary_uris.map { |uri| uri.sub(/^https?:\/\//,'') } if primary_uris
        primary_uris
      end

      def self.get_termini(pe_master_group, service)
        master_profile = pe_master_group['classes']['puppet_enterprise::profile::master']
        servers = Array(master_profile["#{service}_host"])
        ports = Array(master_profile["#{service}_port"])

        servers.zip(ports).map do |server, port|
          "#{server}:#{port}" if server && port
        end
      end

      def self.get_termini_port(pe_master_group, service)
        master_profile = pe_master_group['classes']['puppet_enterprise::profile::master']
        Array(master_profile["#{service}_port"]).first
      end

      def self.update_ha_master_sync_peers(nc, all_groups, master_certname, puppetdb_servers, puppetdb_ports)
        puppetdb_servers ||= []
        puppetdb_ports ||= []
        pe_ha_master_group = PuppetX::Util::Classifier.find_group(all_groups, 'PE HA Master')

        query = <<~EOT
          resources[parameters.puppetdb_sync_interval_minutes] {
            certname = "#{master_certname}" and
            type = "Class" and
            title = "Puppet_enterprise"
          }
        EOT
        begin
          # The require goes here so we don't try loading it during install. 
          # Since this should only ever run on the primary that has PuppetDB installed, 
          # we know this library will be present.
          require 'puppet/util/puppetdb'
          result = Puppet::Util::Puppetdb.query_puppetdb(query)
          puppetdb_sync_interval_minutes = result.nil? || result.empty? ? 5 : result.first['parameters.puppetdb_sync_interval_minutes']
        rescue Exception => e
          Puppet.warning(_('Error attempting to query PuppetDB to find the current value of puppet_enterprise::puppetdb_sync_interval_minutes. Defaulting to 5 minutes.'))
          Puppet.debug(e.full_message)
          puppetdb_sync_interval_minutes = 5
        end

        nc.update_group(
          {
            :id => pe_ha_master_group['id'],
            :classes => {
              'puppet_enterprise::profile::puppetdb' => {
                :sync_peers => puppetdb_servers.zip(puppetdb_ports)
                                 .select { |server, _| server != master_certname }
                                 .map do |replica, port|
                  { 'host' => replica,
                    'port' => port,
                    'sync_interval_minutes' => puppetdb_sync_interval_minutes, }
                end
              }}
          })
      end

      def self.promote_replica(nc, all_groups, replica_certname, primary_master_group_ids, infra_conf, master_certname, configure_agent)
        infra_options = infra_options_from_conf(infra_conf)
        pe_infra_group = find_and_check_node_group(all_groups, 'PE Infrastructure', required_classes: ['puppet_enterprise'])
        pe_agent_group = find_and_check_node_group(all_groups, 'PE Agent')
        pe_infra_agent_group = find_and_check_node_group(all_groups, 'PE Infrastructure Agent')
        pe_master_group = find_and_check_node_group(all_groups, 'PE Master')

        ha_enabled_replicas = pe_infra_group['classes']['puppet_enterprise']['ha_enabled_replicas'] || []
        ha_enabled_replicas.delete(replica_certname)
        configure_ha_master_node_group_to_remove_replica(nc, all_groups, replica_certname)
        update_pins_for_replica_during_promotion(nc, all_groups, replica_certname, primary_master_group_ids)

        update_puppet_enterprise_classification_for_promotion(nc, pe_infra_group, ha_enabled_replicas, replica_certname)

        if configure_agent
          update_pe_agent_group(nc, pe_agent_group, infra_options[:agent_server_urls], infra_options[:pcp_broker_list], infra_options[:primary_uris])
        end
        update_pe_infra_agent_group(nc, pe_infra_agent_group, infra_options[:infra_agent_server_urls], infra_options[:infra_pcp_broker_list], infra_options[:infra_primary_uris])

        update_master_profile(nc, pe_master_group,
                              infra_options[:puppetdb_hosts],
                              infra_options[:puppetdb_ports],
                              infra_options[:classifier_hosts],
                              infra_options[:classifier_ports])
        update_ha_master_sync_peers(nc, all_groups, replica_certname,
                                    infra_options[:puppetdb_hosts],
                                    infra_options[:puppetdb_ports])
      end

      def self.enable_replica(nc, all_groups, master_certname, replica_certname, infra_conf, configure_agent)
        infra_options = infra_options_from_conf(infra_conf)
        pe_infra_group = find_and_check_node_group(all_groups, 'PE Infrastructure', required_classes: ['puppet_enterprise'])
        pe_agent_group = find_and_check_node_group(all_groups, 'PE Agent')
        pe_infra_agent_group = find_and_check_node_group(all_groups, 'PE Infrastructure Agent')
        pe_master_group = find_and_check_node_group(all_groups, 'PE Master')

        ha_enabled_replicas = pe_infra_group['classes']['puppet_enterprise']['ha_enabled_replicas'] || []
        ha_enabled_replicas.push(replica_certname) unless ha_enabled_replicas.include?(replica_certname)
        update_ha_enabled_replicas(nc, pe_infra_group, ha_enabled_replicas)

        if configure_agent
          update_pe_agent_group(nc, pe_agent_group, infra_options[:agent_server_urls], infra_options[:pcp_broker_list], infra_options[:primary_uris])
        end
        update_pe_infra_agent_group(nc, pe_infra_agent_group, infra_options[:infra_agent_server_urls], infra_options[:infra_pcp_broker_list], infra_options[:infra_primary_uris])

        update_master_profile(nc, pe_master_group,
                              infra_options[:puppetdb_hosts],
                              infra_options[:puppetdb_ports],
                              infra_options[:classifier_hosts],
                              infra_options[:classifier_ports])
        update_ha_master_sync_peers(nc, all_groups, master_certname,
                                    infra_options[:puppetdb_hosts],
                                    infra_options[:puppetdb_ports])
      end

      # Given a server_list (with or without ports), remove entries that
      # correspond to the given server_name. Returns a copy.
      def self.remove_server(server_list, server_name)
        return nil if server_list.nil?
        server_list.delete_if { |server| server.start_with? "#{server_name}:" }
        server_list.delete server_name
        server_list
      end

      # Returns true if at least one replica is enabled.
      def self.replication_enabled?(all_groups)
        pe_infra_group = find_and_check_node_group(all_groups, 'PE Infrastructure', required_classes: ['puppet_enterprise'])
        ha_enabled_replicas = pe_infra_group['classes']['puppet_enterprise']['ha_enabled_replicas'] || []
        ha_enabled_replicas.count >= 1
      end

      def self.disable_replica(nc, all_groups, master_certname, replica_certname)
        # It's not safe to continue if no replicas are enabled.
        # We could end up overwriting the agent server list with [], which would break puppet.
        return nil unless replication_enabled? all_groups

        pe_infra_group = find_and_check_node_group(all_groups, 'PE Infrastructure', required_classes: ['puppet_enterprise'])
        pe_infra_agent_group = find_and_check_node_group(all_groups, 'PE Infrastructure Agent')
        pe_agent_group = find_and_check_node_group(all_groups, 'PE Agent')
        pe_master_group = find_and_check_node_group(all_groups, 'PE Master', required_classes: ['puppet_enterprise::profile::master'])

        # TODO: Why don't we just get_config here?
        master_profile = pe_master_group['classes']['puppet_enterprise::profile::master']
        puppetdb_port = Array(master_profile['puppetdb_port']).first || 8081
        classifier_port = Array(master_profile['classifier_port']).first || 4433
        puppetdb_hosts = Array(master_profile['puppetdb_host'])
        classifier_hosts = Array(master_profile['classifier_host'])

        ha_enabled_replicas = pe_infra_group['classes']['puppet_enterprise']['ha_enabled_replicas']
        infra_agent_server_urls = pe_infra_agent_group['classes']['puppet_enterprise::profile::agent']['server_list']
        agent_server_urls = pe_agent_group['classes']['puppet_enterprise::profile::agent']['server_list']
        pcp_broker_list = get_pcp_brokers(pe_agent_group)
        infra_pcp_broker_list = get_pcp_brokers(pe_infra_agent_group)
        primary_uris = get_primary_uris(pe_agent_group)
        infra_primary_uris = get_primary_uris(pe_infra_agent_group)

        ha_enabled_replicas = remove_server(ha_enabled_replicas, replica_certname)
        infra_agent_server_urls = remove_server(infra_agent_server_urls, replica_certname)
        agent_server_urls = remove_server(agent_server_urls, replica_certname)
        puppetdb_hosts = remove_server(puppetdb_hosts, replica_certname)
        puppetdb_ports = puppetdb_hosts.map { |_| puppetdb_port }
        classifier_hosts = remove_server(classifier_hosts, replica_certname)
        classifier_ports = classifier_hosts.map { |_| classifier_port }
        pcp_broker_list = remove_server(pcp_broker_list, replica_certname)
        infra_pcp_broker_list = remove_server(infra_pcp_broker_list, replica_certname)
        primary_uris = remove_server(primary_uris, replica_certname)
        infra_primary_uris = remove_server(infra_primary_uris, replica_certname)

        update_ha_enabled_replicas(nc, pe_infra_group, ha_enabled_replicas)
        update_pe_agent_group(nc, pe_agent_group, agent_server_urls, pcp_broker_list, primary_uris)
        update_pe_infra_agent_group(nc, pe_infra_agent_group, infra_agent_server_urls, infra_pcp_broker_list, infra_primary_uris)
        update_master_profile(nc, pe_master_group, puppetdb_hosts, puppetdb_ports, classifier_hosts, classifier_ports)
        update_ha_master_sync_peers(nc, all_groups, master_certname, puppetdb_hosts, puppetdb_ports)
      end

      def self.provision_replica(nc, master_certname, replica_certname)
        all_groups = PuppetX::Util::Classifier.get_groups(nc)

        pe_infra_group_id = find_and_check_node_group_id(all_groups, 'PE Infrastructure')
        pe_master_group_id = find_and_check_node_group_id(all_groups, 'PE Master')

        PuppetX::Util::Classifier.create_ha_master_node_group(nc, all_groups, pe_master_group_id)
        PuppetX::Util::Classifier.create_ha_replica_node_group(nc, all_groups, pe_infra_group_id)

        all_groups = PuppetX::Util::Classifier.get_groups(nc)

        ha_master_group_id = find_and_check_node_group_id(all_groups, 'PE HA Master')
        ha_replica_group_id = find_and_check_node_group_id(all_groups, 'PE HA Replica')

        PuppetX::Util::Classifier.pin_node_to_group(nc, 'PE HA Master', ha_master_group_id, master_certname)
        PuppetX::Util::Classifier.pin_node_to_group(nc, 'PE HA Replica', ha_replica_group_id, replica_certname)

        configure_ha_master_node_group_to_allow_replica(nc, all_groups, replica_certname)
      end

      def self.get_ha_master_sync_allowlist(all_groups)
        pe_ha_master_group = PuppetX::Util::Classifier.find_group(all_groups, 'PE HA Master')
        pe_ha_master_group['classes']['puppet_enterprise::profile::puppetdb']['sync_allowlist']
      end

      def self.get_ha_master_sync_whitelist(all_groups)
        pe_ha_master_group = PuppetX::Util::Classifier.find_group(all_groups, 'PE HA Master')
        pe_ha_master_group['classes']['puppet_enterprise::profile::puppetdb']['sync_whitelist']
      end

      def self.update_ha_master_sync_allowlist(nc, all_groups, sync_allowlist)
        pe_ha_master_group = PuppetX::Util::Classifier.find_group(all_groups, 'PE HA Master')
        nc.update_group(
          {
            :id => pe_ha_master_group['id'],
            :classes => {
              'puppet_enterprise::profile::puppetdb' => {
                :sync_allowlist => sync_allowlist
              }
            }
          }
        )
      end

      def self.update_ha_master_sync_whitelist(nc, all_groups, sync_whitelist)
        pe_ha_master_group = PuppetX::Util::Classifier.find_group(all_groups, 'PE HA Master')
        nc.update_group(
          {
            :id => pe_ha_master_group['id'],
            :classes => {
              'puppet_enterprise::profile::puppetdb' => {
                :sync_whitelist => sync_whitelist
              }
            }
          }
        )
      end

      def self.get_ha_master_sync_peers(all_groups)
        pe_ha_master_group = PuppetX::Util::Classifier.find_group(all_groups, 'PE HA Master')
        sync_peers = pe_ha_master_group['classes']['puppet_enterprise::profile::puppetdb']['sync_peers']
        return nil if sync_peers.nil?
        sync_peers.map { |peer| [peer['host'], peer['port']] }
      end

      def self.get_all_sync_peers(all_groups, recipient)
        pe_master_group = PuppetX::Util::Classifier.find_group(all_groups, 'PE Master')
        puppetdb_servers = pe_master_group['classes']['puppet_enterprise::profile::master']['puppetdb_host']
        # If this is not defined, then we have no enabled replicas and no sync peers
        return nil if puppetdb_servers.nil?
        puppetdb_ports = pe_master_group['classes']['puppet_enterprise::profile::master']['puppetdb_port']
        puppetdb_servers.zip(puppetdb_ports).reject { |server, _| server == recipient }
      end

      # Finds if a particular rule is included in the rules of the node group.
      # Does not determine how it is included (e.g. ANDed or ORed with other rules),
      # just if it exists at all. The rule should be an array like
      # ['=', ['trusted, 'extensions', 'pp_auth_role'], 'pe_compiler']
      # Rules should be the full set of rules for the node group.
      def self.rule_exists?(rules, rule)
        return true if rules == rule
        if rules.is_a?(Array)
          return !rules.find { |r| rule_exists?(r, rule) }.nil?
        else
          return false
        end
      end
    end
  end
end
