# frozen_string_literal: true

# Sleeps for specified number of seconds.
Puppet::Functions.create_function(:'ctrl::sleep') do
  # @param period Time to sleep (in seconds)
  # @example Sleep for 5 seconds
  #   ctrl::sleep(5)
  dispatch :sleeper do
    required_param 'Numeric', :period
    return_type 'Undef'
  end

  def sleeper(period)
    # Send Analytics Report
    Puppet.lookup(:bolt_executor) {}&.report_function_call(self.class.name)
    # The reference to the active thread needs to be created *before* the
    # call to sleep (and any subsequent changes like interrupts that the
    # sleep makes on the thread). Otherwise if you try to pull the active
    # thread after the sleep has completed JRuby appears to give you a
    # reference to a thread not connected to what just happened.
    #
    # I have suspicions this is due to https://github.com/jruby/jruby/blob/163d0aa13f9299925e754d8b0b07581e8bdac832/core/src/main/java/org/jruby/javasupport/util/ObjectProxyCache.java#L13L31
    # but I'm not 100% positive about that. Regardless, as long as the
    # thread reference is created before the sleep everything appears to
    # work well, so I'm not digging any more.
    #
    #                             - Sean McDonald 10/12/2022
    java_thread = nil
    if defined?(JRuby)
      java_thread = JRuby.reference(Thread.current).native_thread
    end

    sleep(period)

    if java_thread&.interrupted?
      raise "sleep interrupted, stopping"
    end

    nil
  end
end
