# frozen_string_literal: true

require 'bolt/error'

# THIS IS A PE SPECIFIC IMPLEMENTATION OF RUN_SCRIPT: THIS IMPLEMENTATION WILL OVERRIDE THE ONE
# THAT COMES WITH A BOLT INSTALLATION
#
# Open-source bolt implementation:
# https://github.com/puppetlabs/bolt/blob/master/bolt-modules/boltlib/lib/puppet/functions/run_script.rb
#
# run_script for plans-in-PE will only run scripts within modules
# (for security reasons). This is handled by Orchestrator's executor,
# which will fetch the script from puppetserver. This PE implementation
# will send the `script` parameter raw to `executor.run_script` rather
# than fully qualifying it.
Puppet::Functions.create_function(:run_script, Puppet::Functions::InternalFunction) do
  dispatch :run_script do
    scope_param
    param 'String[1]', :script
    param 'Boltlib::TargetSpec', :targets
    optional_param 'Hash[String[1], Any]', :options
    return_type 'ResultSet'
  end

  dispatch :run_script_with_description do
    scope_param
    param 'String[1]', :script
    param 'Boltlib::TargetSpec', :targets
    param 'String', :description
    optional_param 'Hash[String[1], Any]', :options
    return_type 'ResultSet'
  end

  def run_script(scope, script, targets, options = {})
    run_script_with_description(scope, script, targets, nil, options)
  end

  def run_script_with_description(scope, script, targets, description = nil, options = {})
    unless Puppet[:tasks]
      raise Puppet::ParseErrorWithIssue
        .from_issue_and_stack(Bolt::PAL::Issues::PLAN_OPERATION_NOT_SUPPORTED_WHEN_COMPILING, action: 'run_script')
    end

    arguments = options['arguments'] || []
    options = options.select { |opt| opt.start_with?('_') }.map { |k, v| [k.sub(/^_/, '').to_sym, v] }.to_h
    options[:description] = description if description
    executor = Puppet.lookup(:bolt_executor)
    inventory = Puppet.lookup(:bolt_inventory)

    executor.report_function_call(self.class.name)

    targets = inventory.get_targets(targets)
    if targets.empty?
      call_function('debug', "Simulating execution of '#{script}' - no targets given - no action taken")
      r = Bolt::ResultSet.new([])
    else
      r = executor.run_script(targets, script, arguments, options, Puppet::Pops::PuppetStack.top_of_stack)
    end

    if !r.ok && !options[:catch_errors]
      raise Bolt::RunFailure.new(r, 'run_script', script)
    end
    r
  end
end
