# This class manages the steps required to prep a node to the point
# where a pe_install::install catalog could proceed with package
# installation and configuration.
#
# It handles setting up the local /opt/puppetlabs/server/data/packages/public
# package repository for this version/platform of PE, moving packages from the
# tarball into that repository, copying and creating a few key metadata and
# configuration files and installing the puppet-agent package.
#
# These steps were formally all handled by the puppet-enterprise-installer
# shim script before it run `puppet-infra configure` to complete installation,
# since they were required before you had a working agent that could execute
# the infrastructure face.
#
# Since the pe-installer package contains all the needed tools and libraries
# now, it can run everything once installed.
#
# It may be possibly to include what these classes do within pe_install,
# but at the moment they are intended to be applied separately as
# pe_install::bootstrap, before pe_install is applied. At a minimum,
# there are a few File resource conflicts. Also, this class requires a local
# $tarball_dir to bootstrap from, whereas pe_install is self-sufficient once
# the bootstrap step is completed, and does not need a reference to a tarball.
#
# = Parameters
#
# @param $effective_date The date the PE install was released; used for
# determining whether a numerical later update version is actually more current
# (security patch wise) than a current numerically earlier z release, for
# example. So is 2019.5.0 actually more recent than 2018.1.15?
#
# @param $pe_build_version The (possibly dev) build of PE being installed; in a
# dev build, this will be longer than the three number release build.
#
# @param $manage_permissions If true, we're performing a fresh install and can
# set permissions for files. For an upgrade, that shouldn't be touched. The
# discrepancy comes because this class is applied before puppet-agent is installed,
# and it's the puppet-agent package that provides the puppet user that would own
# many of these. However, if we don't manage permissions at all for initial installs,
# then a restrictive umask may leave some files unreadable if a later catalog does
# not enforce their permissions state.
#
# @param $pe_tarball_dir The directory of the untarred PE tarball we
# are copying files from.
#
# @param $platform_tag Identifies platform-version-arch ('el-7-x86_64', for example)
# @param $aio_agent_version The puppet agent version (like 6.0.3 or 6.0.3.157)
# @param $aio_install_version In dev builds this may include an additional git string
# (so compared with aio_agent_version, it might be '6.0.3.157.g12345678')
#
# @param $pe_conf Absolute path to the pe.conf file to be copied to
# /etc/puppetlabs/enterprise/conf.d. If not given, assumed to be in place already.
class pe_install::bootstrap(
  Pe_install::Absolute_path $pe_tarball_dir,
  Pe_install::Pe_build $pe_build_version,
  String $platform_tag,
  Pe_install::Agent_version $aio_agent_version,
  Pe_install::Agent_build $aio_install_version,
  Pe_install::Effective_date $effective_date,
  Boolean $manage_permissions = false,
  Optional[Pe_install::Absolute_path] $pe_conf = undef,
) {
  include 'pe_install::params'

  file { $pe_install::params::server_dir:
    ensure => 'directory',
    owner  => 'root',
    group  => 'root',
    mode   => '0644',
  }

  class { 'pe_install::bootstrap::package_repository':
    pe_tarball_dir      => $pe_tarball_dir,
    pe_build_version    => $pe_build_version,
    platform_tag        => $platform_tag,
    aio_agent_version   => $aio_agent_version,
    aio_install_version => $aio_install_version,
  }
  -> class { 'pe_install::bootstrap::install_tarball_files':
    pe_tarball_dir     => $pe_tarball_dir,
    pe_build_version   => $pe_build_version,
    effective_date     => $effective_date,
    manage_permissions => $manage_permissions,
  }
  -> class {'puppet_enterprise::repo::config':
    pe_ver       => $pe_build_version,
    platform_tag => $platform_tag,
    local_config => true,
    source       => "${pe_install::params::server_packages_dir}/public",
  }
  -> class {'pe_install::bootstrap::agent':
    aio_install_version => $aio_install_version,
  }

  contain 'pe_install::bootstrap::package_repository'
  contain 'pe_install::bootstrap::install_tarball_files'
  contain 'puppet_enterprise::repo::config'
  contain 'pe_install::bootstrap::agent'

  if $pe_conf =~ NotUndef {
    file { "${pe_install::params::enterprise_etc_dir}/conf.d/pe.conf":
      ensure  => 'file',
      source  => $pe_conf,
      owner   => 'root',
      group   => 'root',
      mode    => '0600',
      require => Class['Pe_install::Bootstrap::Agent'],
    }
  }
}
