# Upgrade the managed PE Postgresql instance from an old version to a new version.
#
# @param old_postgres_version [String] The major.minor version string for the
#   current intallation of postgresql that we are upgrading from. ('9.4' for example)
# @param new_postgres_version [String] The major.minor version string of the
#   version of postgresql that we are upgrading to. ('9.6' for example)
# @param pg_upgrade_timeout [Integer] the number of seconds to wait for
#   pg_upgrade to complete (the same timeout is also used to wait for vacuumdb).
#   Defaults to 0, which does not impose a timeout
# @param require_pglogical [Boolean] true if pe-postgresql-pglogical should be a package
#   dependency for the upgrade.  This is the case for the main PE Postgresql instance,
#   but is not present in a Razor postgres instance, for example.
# @param replica_migration [Boolean] true if we are attempting to migrate a replica database
#   rather than the source database.
class pe_install::upgrade::postgres(
  String $old_postgres_version,
  String $new_postgres_version,
  Integer $pg_upgrade_timeout = 0,
  Boolean $require_pglogical = true,
  Boolean $replica_migration = false,
) {
  include 'pe_install::params'

  $postgres_data_root = "${pe_install::params::server_data_dir}/postgresql"
  $postgres_app_root = "${pe_install::params::server_dir}/apps/postgresql"

  $new_data_dir = "${postgres_data_root}/${new_postgres_version}/data"
  $new_app_dir  = "${postgres_app_root}/${new_postgres_version}"
  $new_bin_dir  = "${new_app_dir}/bin"
  $new_lib_dir  = "${new_app_dir}/lib"

  $old_data_dir  = "${postgres_data_root}/${old_postgres_version}/data"
  $old_app_dir   = "${postgres_app_root}/${old_postgres_version}"
  $old_bin_dir   = "${old_app_dir}/bin"
  $old_lib_dir   = "${old_app_dir}/lib"

  $pe_postgresql_info = pe_pick($facts['pe_postgresql_info'], {})

  $base_package_dependencies = [
    Package['postgresql-client'],
    Package['postgresql-server'],
    Package['postgresql-contrib'],
  ]
  $package_dependencies = $require_pglogical ? {
    true  => $base_package_dependencies + Package[$puppet_enterprise::params::postgresql_pglogical_package_name],
    false => $base_package_dependencies,
  }

  #  1. Drop pg_repack
  #     
  #     We need to do this due to potential pg_repack changes on the new
  #     postgres version. On the PG14 upgrade, pg_upgrade would fail due to 
  #     a missing function. This must happen before we shut down the old
  #     postgres service. When upgrading a replica, this is performed by
  #     the enterprise_tasks::upgrade_and_migrate_replica plan instead.

  if !$replica_migration {
    $databases = [
      $puppet_enterprise::classifier_database_name,
      $puppet_enterprise::rbac_database_name,
      $puppet_enterprise::activity_database_name,
      $puppet_enterprise::orchestrator_database_name,
      $puppet_enterprise::puppetdb_database_name,
      $puppet_enterprise::inventory_database_name,
    ]

    $databases.map |$database_name| {
      puppet_enterprise::psql { "Drop pg_repack on ${database_name} prior to migration":
        db      => $database_name,
        command => 'DROP EXTENSION IF EXISTS "pg_repack" CASCADE',
        before  => [Package['postgresql-server'], Pe_install::Stop_service['Stop pe-postgresql service']],
      }
    }
  }

  #  2. Copy the original postgresql.conf

  if $pe_postgresql_info['installed_server_version'] == $old_postgres_version {
    $copy_command = "cp -f ${old_data_dir}/postgresql.conf ${new_data_dir}/postgresql.conf"
    $owner_command = "chown pe-postgres pe-postgres ${new_data_dir}/postgresql.conf"
    $permissions_command = "chmod 600 ${new_data_dir}/postgresql.conf"

    exec { 'copy postgresql.conf from old install location to new install location':
      command   => "${copy_command}; ${owner_command}; ${permissions_command}",
      path      => $facts['path'],
      logoutput => true,
      loglevel  => 'info',
      before    => [Class['pe_postgresql::server::config'], Pe_install::Stop_service['Stop pe-postgresql before pg_upgrade'] ],
      require   => [Package['postgresql-server'],Class['pe_postgresql::server::initdb']],
    }
  }

  #  3. Stop service pe-postgresql
  #
  #     Before upgrading the pe-postgresql-server package. On some
  #     systems, the new pe-postgres-server package's service script cannot stop
  #     the old service. Although we stop all services before upgrading primary infrastructure
  #     nodes, if this class is being included elsewhere (Razor, for example) that will not
  #     have happened, and we need to be explicit about this shutdown.
  $disable_service = ($facts['os']['name'] != 'Ubuntu') or ($facts['os']['release']['major'] !~ /18.04/)
  pe_install::stop_service { 'Ensure old pe-postgresql server stopped before upgrading package':
    service          => 'pe-postgresql',
    before_reference => [Package['postgresql-server']],
    disable_service  => $disable_service,
  }
  #     And before calling pg_upgrade, but after the package is upgraded, in
  #     case the package has restarted the service.
  pe_install::stop_service { 'Stop pe-postgresql before pg_upgrade':
    service          => 'pe-postgresql',
    before_reference => [Exec['pg_upgrade: migrate to new postgres']],
    after_reference  => [Package['postgresql-server']],
    disable_service  => $disable_service,
  }

  #  4. exec a script that executes pg_upgrade
  #     * before  => Class['pe_postgresql::server::config'],
  #     * require => Class['pe_postgresql::server::initdb'],

  exec { 'pg_upgrade: migrate to new postgres':
    environment => ["LD_LIBRARY_PATH=${new_lib_dir}:${old_lib_dir}:\$LD_LIBRARY_PATH"],
    command     => "${new_bin_dir}/pg_upgrade -b ${old_bin_dir} -B ${new_bin_dir} -d ${old_data_dir} -D ${new_data_dir}",
    user        => 'pe-postgres',
    cwd         => $postgres_data_root,
    timeout     => $pg_upgrade_timeout,
    before      => Class['pe_postgresql::server::config'],
    require     => $package_dependencies + Class['pe_postgresql::server::initdb'],
  }

  #  5. Post upgrade, we need to remove pglogical triggers if they are in an
  #    orphaned state. See (PE-23115)
  #
  if $require_pglogical and !$replica_migration {
    class { 'puppet_enterprise::pg::pglogical::drop_orphaned_triggers':
      require => Class['pe_postgresql::server::service'],
      before  => Exec['vacuum the newly migrated database'],
    }
    contain 'puppet_enterprise::pg::pglogical::drop_orphaned_triggers'
  }

  #  6. post upgrade scripts may be generated by the pg_upgrade process?
  #     * also we do need to regenerate optimizer statistics (see point 14 in pg_upgrade docs: https://www.postgresql.org/docs/9.6/static/pgupgrade.html)
  exec { 'vacuum the newly migrated database':
    command => "${new_bin_dir}/vacuumdb --analyze --all",
    user    => 'pe-postgres',
    cwd     => $postgres_data_root,
    timeout => $pg_upgrade_timeout,
    require => Class['pe_postgresql::server::service'],
  }
}
