define puppet_enterprise::trapperkeeper::orchestrator(
  $master_url,
  $puppetdb_url,
  $classifier_url,
  $rbac_url,
  $activity_url,
  $console_url,
  $console_services_url,
  $inventory_service_url,
  $allow_unauthenticated_status,
  $bolt_server_max_connections,
  $ace_server_max_connections,
  Integer $bolt_server_port                          = $puppet_enterprise::bolt_server_port,
  Integer $bolt_server_conn_timeout                  = 10000,
  Integer $ace_server_port                           = $puppet_enterprise::ace_server_port,
  Integer $ace_server_conn_timeout                   = 10000,
  $pcp_broker_url                                    = undef,
  Array[String] $pcp_brokers                         = [],
  $client_certname                                   = $puppet_enterprise::puppet_master_host,
  $container                                         = $title,
  $database_host                                     = 'localhost',
  $database_port                                     = $puppet_enterprise::database_port,
  $database_name                                     = $puppet_enterprise::orchestrator_database_name,
  $database_user                                     = $puppet_enterprise::orchestrator_service_regular_db_user,
  $database_migration_user                           = $puppet_enterprise::orchestrator_service_migration_db_user,
  $database_password                                 = undef,
  $database_properties                               = '',
  $user                                              = "pe-${title}",
  $group                                             = "pe-${title}",
  $max_message_size                                  = $puppet_enterprise::pcp_max_message_size,
  Optional[Integer] $allowed_pcp_status_timeouts     = undef,
  Optional[Integer] $global_concurrent_compiles      = undef,
  Optional[Integer] $task_concurrency                = undef,
  Optional[Integer] $job_prune_threshold             = undef,
  Optional[Integer] $pcp_timeout                     = undef,
  Optional[Boolean] $app_management                  = undef,
  Optional[Integer] $jruby_borrow_timeout            = undef,
  Optional[Integer] $jruby_max_active_instances      = undef,
  Boolean $jruby_multithreaded                       = false,
  Integer $jruby_max_requests_per_instance           = 0,
  Enum['jit', 'off'] $jruby_compile_mode             = 'jit',
  Optional[Integer] $default_task_node_timeout       = undef,
  Optional[Integer] $default_deploy_node_timeout     = undef,
  Optional[Integer] $default_plan_timeout            = undef,
  Integer $max_connections_per_route_authenticated   = 20, # maximum number of per-route (hostname+port) connections when the client cert is provided
  Integer $max_connections_total_authenticated       = 20, # maximum number of total connections (for all routes) when the client cert is provided
  Integer $max_connections_per_route_unauthenticated = 20, # maximum number of per-route (hostname+post) connections when the client cert is not provided (using user auth)
  Integer $max_connections_total_unauthenticated     = 20, # maximum number of total connections (for all routes), when the client cert is not provided (using user auth)
  Integer $socket_timeout                            = 120000, # timeout in milliseconds
  Boolean $plan_runner_active                        = false,
) {

  if ($max_connections_total_authenticated < $max_connections_per_route_authenticated) {
    warning('puppet_enterprise::trapperkeeper::orchestrator::max_connections_total_authenticated is less than puppet_enterprise::trapperkeeper::orchestrator::max_connections_per_route_authenticated will limit the per-route connections available to less than what was specified.')
  }

  if ($max_connections_total_unauthenticated < $max_connections_per_route_unauthenticated) {
    warning('puppet_enterprise::trapperkeeper::orchestrator::max_connections_total_unauthenticated is less than puppet_enterprise::trapperkeeper::orchestrator::max_connections_per_route_unauthenticated which will limit the per-route connections available to less than what was specified.')
  }

  $confdir = "/etc/puppetlabs/${container}/conf.d"
  $orchestrator_conf_path = "${confdir}/orchestrator.conf"

  file { $orchestrator_conf_path:
    ensure => present,
    owner  => $user,
    group  => $group,
    mode   => '0640',
  }

  file { "${confdir}/secrets":
    ensure => directory,
    owner  => $user,
    group  => $group,
    mode   => '0640',
  }

  Pe_hocon_setting {
    ensure => present,
    notify => Service["pe-${container}"],
  }

  pe_hocon_setting { "${container}.orchestrator.encryption-store":
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.encryption-store',
    value   => "${confdir}/secrets/orchestrator-encryption-keys.json",
  }

  pe_hocon_setting { "${container}.orchestrator.master-url":
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.master-url',
    value   => $master_url,
  }

  pe_hocon_setting { "${container}.orchestrator.puppetdb-url":
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.puppetdb-url',
    value   => $puppetdb_url,
  }

  pe_hocon_setting { "${container}.orchestrator.classifier-service":
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.classifier-service',
    value   => $classifier_url,
  }

  pe_hocon_setting { "${container}.orchestrator.console-services-url":
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.console-services-url',
    value   => $console_services_url,
  }

  pe_hocon_setting { "${container}.orchestrator.inventory-service-url":
    path => $orchestrator_conf_path,
    setting => 'orchestrator.inventory-service-url',
    value => $inventory_service_url,
  }

  pe_hocon_setting { "${container}.orchestrator.bolt-server-url":
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.bolt-server-url',
    value => "https://${client_certname}:${bolt_server_port}"
  }

  pe_hocon_setting { "${container}.orchestrator.bolt-server-conn-timeout":
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.bolt-server-conn-timeout',
    value => $bolt_server_conn_timeout,
  }

  pe_hocon_setting { "${container}.orchestrator.bolt-server-max-connections":
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.bolt-server-max-connections',
    value => $bolt_server_max_connections,
  }

  pe_hocon_setting { "${container}.orchestrator.ace-server-url":
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.ace-server-url',
    value => "https://${client_certname}:${ace_server_port}"
  }

  pe_hocon_setting { "${container}.orchestrator.ace-server-conn-timeout":
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.ace-server-conn-timeout',
    value => $ace_server_conn_timeout,
  }

  pe_hocon_setting { "${container}.orchestrator.ace-server-max-connections":
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.ace-server-max-connections',
    value => $ace_server_max_connections,
  }

  pe_hocon_setting { "${container}.rbac-consumer.api-url":
    path    => $orchestrator_conf_path,
    setting => 'rbac-consumer.api-url',
    value   => $rbac_url,
  }

  pe_hocon_setting { "${container}.activity-consumer.api-url":
    ensure  => present,
    path    => $orchestrator_conf_path,
    setting => 'activity-consumer.api-url',
    value   => $activity_url,
  }

  pe_hocon_setting { "${container}.orchestrator.pcp-broker-url":
    ensure  => absent,
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.pcp-broker-url',
  }

  pe_hocon_setting { "${container}.orchestrator.console-url":
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.console-url',
    value   => $console_url,
  }

  pe_hocon_setting { "${container}.orchestrator.ssl-ca-cert":
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.ssl-ca-cert',
    value   => $puppet_enterprise::params::localcacert,
  }

  pe_hocon_setting { "${container}.orchestrator.ssl-cert":
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.ssl-cert',
    value   => "/etc/puppetlabs/${container}/ssl/${client_certname}.cert.pem",
  }

  pe_hocon_setting { "${container}.orchestrator.ssl-key":
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.ssl-key',
    value   => "/etc/puppetlabs/${container}/ssl/${client_certname}.private_key.pem",
  }

  $pcp_timeout_ensure = $pcp_timeout ? {
    undef   => 'absent',
    default => 'present',
  }


  $global_concurrent_compiles_ensure = $global_concurrent_compiles ? {
    undef   => 'absent',
    default => 'present',
  }

  $task_concurrency_ensure = $task_concurrency ? {
    undef   => 'absent',
    default => 'present',
  }


  $job_prune_threshold_ensure = $job_prune_threshold ? {
    undef   => 'absent',
    default => 'present',
  }

  $app_management_ensure = $app_management ? {
    undef   => 'absent',
    default => 'present',
  }

  $default_task_node_timeout_ensure = $default_task_node_timeout ? {
    undef   => 'absent',
    default => 'present',
  }

  $default_deploy_node_timeout_ensure = $default_deploy_node_timeout ? {
    undef   => 'absent',
    default => 'present',
  }

  $default_plan_timeout_ensure = $default_plan_timeout ? {
    undef   => 'absent',
    default => 'present',
  }

  pe_hocon_setting { "${container}.orchestrator.pcp-timeout":
    ensure  => $pcp_timeout_ensure,
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.pcp-timeout',
    value   => $pcp_timeout,
  }

  pe_hocon_setting { "${container}.orchestrator.global-concurrent-compiles":
    ensure  => $global_concurrent_compiles_ensure,
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.global-concurrent-compiles',
    value   => $global_concurrent_compiles,
  }

  pe_hocon_setting { "${container}.orchestrator.task-concurrency":
    ensure  => $task_concurrency_ensure,
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.task-concurrency',
    value   => $task_concurrency,
  }

  pe_hocon_setting { "${container}.orchestrator.job-prune-days-threshold":
    ensure  => $job_prune_threshold_ensure,
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.job-prune-days-threshold',
    value   => $job_prune_threshold,
  }

  pe_hocon_setting { "${container}.orchestrator.app-management":
    ensure  => $app_management_ensure,
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.app-management',
    value   => $app_management,
  }

  pe_hocon_setting { "${container}.orchestrator.database.subname":
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.database.subname',
    value   => "//${database_host}:${database_port}/${database_name}${database_properties}",
  }

  pe_hocon_setting { "${container}.orchestrator.database.user":
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.database.user',
    value   => $database_user,
  }
  pe_hocon_setting { "${container}.orchestrator.database.migration-user":
    ensure  => present,
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.database.migration-user',
    value   => $database_migration_user,
  }

  if !pe_empty($database_password) {
    pe_hocon_setting { "${container}.orchestrator.database.password":
      path    => $orchestrator_conf_path,
      setting => 'orchestrator.database.password',
      value   => $database_password,
    }
    pe_hocon_setting { "${container}.orchestrator.database.migration-password":
      path    => $orchestrator_conf_path,
      setting => 'orchestrator.database.migration-password',
      value   => $database_password,
    }
  }

  pe_hocon_setting { "${container}.orchestrator.jruby.compile-mode":
    ensure  => present,
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.jruby.compile-mode',
    value   => $jruby_compile_mode,
  }

  pe_hocon_setting { "${container}.orchestrator.jruby.multithreaded":
    ensure  => present,
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.jruby.multithreaded',
    value   => $jruby_multithreaded,
  }

  # 'max-borrows-per-instance' is the name of this setting in jruby-utils, while
  # puppetserver calls this 'max-requests-per-instance'. The parameter uses
  # puppetserver's name for consistency.
  pe_hocon_setting { "${container}.orchestrator.jruby.max-requests-per-instance":
    ensure  => present,
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.jruby.max-requests-per-instance',
    value   => $jruby_max_requests_per_instance,
    notify  => Exec["pe-${container} service full restart"],
  }

  if $jruby_borrow_timeout {
    $jruby_borrow_timeout_ensure = present
  } else {
    $jruby_borrow_timeout_ensure = absent
  }

  pe_hocon_setting { "${container}.orchestrator.jruby.borrow-timeout":
    ensure  => $jruby_borrow_timeout_ensure,
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.jruby.borrow-timeout',
    value   => $jruby_borrow_timeout,
  }

  pe_hocon_setting { "${container}.orchestrator.jruby.max-active-instances":
    ensure  => present,
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.jruby.max-active-instances',
    value   => $jruby_max_active_instances,
  }

  if $allowed_pcp_status_timeouts {
    $allowed_pcp_status_timeouts_ensure = present
  } else {
    $allowed_pcp_status_timeouts_ensure = absent
  }
  pe_hocon_setting { "${container}.orchestrator.allowed-pcp-status-timeouts":
    ensure  => $allowed_pcp_status_timeouts_ensure,
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.allowed-pcp-status-timeouts',
    type    => 'number',
    value   => $allowed_pcp_status_timeouts,
  }

  pe_hocon_setting { "${container}.orchestrator.max-message-size":
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.max-message-size',
    type    => 'number',
    value   => $max_message_size,
  }

  pe_hocon_setting { "${container}.orchestrator.default-task-node-timeout":
    ensure  => $default_task_node_timeout_ensure,
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.default-task-node-timeout',
    type    => 'number',
    value   => $default_task_node_timeout,
  }

  pe_hocon_setting { "${container}.orchestrator.default-deploy-node-timeout":
    ensure  => $default_deploy_node_timeout_ensure,
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.default-deploy-node-timeout',
    type    => 'number',
    value   => $default_deploy_node_timeout,
  }

  pe_hocon_setting { "${container}.orchestrator.default-plan-timeout":
    ensure  => $default_plan_timeout_ensure,
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.default-plan-timeout',
    type    => 'number',
    value   => $default_plan_timeout,
  }

  pe_hocon_setting {"${container}.orchestrator.max-connections-per-route-auth":
    ensure  => present,
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.max-connections-per-route-auth',
    value   => $max_connections_per_route_authenticated,
  }

  pe_hocon_setting {"${container}.orchestrator.max-connections-total-auth":
    ensure  => present,
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.max-connections-total-auth',
    value   => $max_connections_total_authenticated,
  }

  pe_hocon_setting {"${container}.orchestrator.max-connections-per-route-unauth":
    ensure  => present,
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.max-connections-per-route-unauth',
    value   => $max_connections_per_route_unauthenticated,
  }

  pe_hocon_setting {"${container}.orchestrator.max-connections-total-unauth":
    ensure  => present,
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.max-connections-total-unauth',
    value   => $max_connections_total_unauthenticated,
  }

  pe_hocon_setting {"${container}.orchestrator.socket-timeout":
    ensure  => present,
    path    => $orchestrator_conf_path,
    setting => 'orchestrator.socket-timeout',
    value   => $socket_timeout,
  }

  Pe_puppet_authorization::Rule {
    path    => "${confdir}/auth.conf",
    notify  => Service["pe-${container}"],
  }

  $runner_auth_ensure = $plan_runner_active ? {
    true    => 'present',
    default => 'absent'
  }

  pe_puppet_authorization::rule { 'v2 plans api':
    ensure             => $runner_auth_ensure,
    match_request_type => 'path',
    match_request_path => '/orchestrator/v1/v2/private/plans',
    allow              => [$client_certname],
    sort_order         => 500,
  }

  pe_puppet_authorization::rule { 'dispatch: allow pcp-brokers':
    match_request_type => 'path',
    match_request_path => '/server',
    allow              => [$client_certname] + sort($pcp_brokers),
    sort_order         => 500,
  }

  $allow_status_service = $allow_unauthenticated_status ? {
                           false => '*',
                           default => undef,
                         }

  pe_puppet_authorization::rule { 'orchestrator simple status endpoint':
    match_request_type    => 'path',
    match_request_method  => 'get',
    match_request_path    => '/status/v1/simple',
    allow_unauthenticated => true,
    sort_order            => 500,
  }

  pe_puppet_authorization::rule { 'orchestrator status service':
    match_request_type    => 'path',
    match_request_method  => 'get',
    match_request_path    => '/status/v1/services',
    allow_unauthenticated => $allow_unauthenticated_status,
    allow                 => $allow_status_service,
    sort_order            => 500,
  }

  # Allow users with a valid puppet cert to access the metrics api
  pe_puppet_authorization::rule { 'orchestrator metrics api':
    match_request_type    => 'path',
    match_request_method  => ['get', 'post'],
    match_request_path    => '/metrics',
    allow                 => '*',
    sort_order            => 500,
  }

  puppet_enterprise::trapperkeeper::analytics_client_settings { $container:
    container => $container,
    owner     => $user,
  }

  Puppet_enterprise::Trapperkeeper::Bootstrap_cfg {
    container => $container,
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:orchestrator orchestrator-service" :
    namespace => 'puppetlabs.orchestrator.service',
    service   => 'orchestrator-service',
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:orchestrator orchestrator-dispatch-service" :
    namespace => 'puppetlabs.orchestrator.dispatch',
    service   => 'orchestrator-dispatch-service',
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:orchestrator jetty10-service" :
    namespace => 'puppetlabs.trapperkeeper.services.webserver.jetty10-service',
    service   => 'jetty10-service',
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:orchestrator remote-rbac-consumer-service" :
    namespace => 'puppetlabs.rbac-client.services.rbac',
    service   => 'remote-rbac-consumer-service',
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:orchestrator remote-activity-reporter" :
    container => $container,
    namespace => 'puppetlabs.rbac-client.services.activity',
    service   => 'remote-activity-reporter',
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container}:orchestrator jruby-pool-manager-service" :
    namespace => 'puppetlabs.services.jruby-pool-manager.jruby-pool-manager-service',
    service   => 'jruby-pool-manager-service'
  }
}
