require 'puppet/indirector/face'
require 'puppet/application/apply'
require 'puppet/util/command_line'
require 'puppet/agent'
require 'puppet/error'
require 'puppet/util/pe_conf'
require 'puppet_x/util/stringformatter'
require 'puppet_x/puppetlabs/meep/caching'
require 'puppet_x/puppetlabs/meep/configure/validator'
require 'puppet_x/puppetlabs/meep/configure/password'
require 'puppet_x/puppetlabs/meep/configure/postgres'
require 'puppet_x/puppetlabs/meep/configure/psql'
require 'puppet_x/puppetlabs/meep/config/modify'
require 'puppet_x/puppetlabs/meep/hiera_adapter'
require 'puppet_x/puppetlabs/meep/infra/defaults'
require 'puppet_x/puppetlabs/meep/infra/layout'
require 'puppet_x/puppetlabs/meep/infra/lookup'
require 'puppet_x/puppetlabs/meep/puppet_context'
require 'puppet_x/puppetlabs/meep/util'
require 'puppet_x/util/password'
require 'puppet_x/util/analytics'
require 'puppet_x/util/ha'
require 'puppet_x/util/bolt'
require 'puppet_x/util/classification'
require 'puppet_x/util/classifier'
require 'puppet_x/util/service_status'

# The initial version of the face is based of Reid Vandewiele's nimbus
# module's latest version, which is 0.7.0.
#
# https://github.com/puppetlabs/tse-module-nimbus/releases/tag/0.7.0
Puppet::Face.define(:infrastructure, '1.0.0') do

  extend Puppet::Agent::Locker
  extend PuppetX::Puppetlabs::Meep::PuppetContext
  extend PuppetX::Puppetlabs::Meep::Caching
  extend PuppetX::Puppetlabs::Meep::Configure::Password
  extend PuppetX::Puppetlabs::Meep::Configure::Postgres
  extend PuppetX::Puppetlabs::Meep::Infra::Defaults
  extend PuppetX::Puppetlabs::Meep::Infra::Layout
  extend PuppetX::Puppetlabs::Meep::Infra::Lookup
  extend PuppetX::Puppetlabs::Meep::Util
  extend PuppetX::Util::HA

  action :configure do

    summary _('Configure the local system using given Puppet all-in-one configuration file(s).')
    description PuppetX::Util::String::Formatter.join_and_indent([
      _("Configure the local system using given Puppet all-in-one configuration file(s)."),
      "\n",
      _("Will exit (17) if another puppet run is already in progress.")])
    when_invoked do |options|
      install_method = options[:install_method]
      @is_upgrade = options.include?(:upgrade_from) && !options[:upgrade_from].nil?
      @is_install = !(install_method.nil? || install_method.empty?) && !@is_upgrade
      action = @is_upgrade ? "upgrade" : "install"
      begun_to_apply = false

      begin
        lock_unless_agent_pid_matches(options[:agent_pid]) do
          begin
            attempt_to_recover_configuration(options)
            update_pe_conf_to_disable_analytics
          ensure
            puppet_infrastructure_context do
              send_analytics_event("pe_installer","pe_version",options[:install])
              validate_current_installation(options)
              if @is_upgrade
                save_upgrade_in_process(options)
                send_analytics_event("pe_installer","upgrade_start","upgrade_start")
                send_analytics_event("pe_installer","upgrade_from",options[:upgrade_from])
              end
              send_analytics_event("pe_installer","install_start",install_method) if @is_install

              update_pe_conf_with_database_encoding_if_migrating(options)

              argv = default_arguments(options)
              argv << '--detailed-exitcodes' if options[:detailed_exitcodes]
              command_line = Puppet::Util::CommandLine.new('puppet', argv)
              apply = Puppet::Application::Apply.new(command_line)
              apply.parse_options
              Puppet::Util::Log.close(:console)
              begun_to_apply = true
              apply.run_command
            end
          end
        end
      # Don't use code 37, as it was previously used for errors unsyncing the replica
      rescue PuppetX::Puppetlabs::Meep::Configure::Validator::InstallationInvalid
        send_analytics_event("pe_installer","#{action}_finish","failed")
        send_analytics_event("pe_installer","#{action}_fail_type","validator.installationinvalid")
        exit(27)
      rescue Puppet::LockError
        send_analytics_event("pe_installer","#{action}_finish","failed")
        send_analytics_event("pe_installer","#{action}_fail_type","puppet_lock_error")
        Puppet.notice _("Puppet run already in progress") + "; " + _("aborting  (%{lockfile_path} exists)") % { lockfile_path: lockfile_path }
        exit(17)
      rescue SystemExit => e
        if [0,2].include?(e.status)
          send_analytics_event("pe_installer","#{action}_finish","succeeded")
          inform_on_successful_postgres_migration(options)
          remove_upgrade_in_process
          notify_reset_url if options.include?(:install) && !options[:upgrade_from] && password_omitted? && console_node?
        else
          send_analytics_event("pe_installer","#{action}_finish","failed")
          send_analytics_event("pe_installer","#{action}_fail_type","exit_code_not_0_or_2")
        end
        raise e
      rescue StandardError => e
        send_analytics_event("pe_installer","#{action}_finish","failed")
        if !begun_to_apply
          Puppet.err _("Unexpected exception: %{message}%{backtrace}") % {
            :message => e.message,
            :backtrace => e.respond_to?(:backtrace) ? "\n#{e.backtrace.join("\n")}" : '',
          }
          send_analytics_event("pe_installer","#{action}_fail_type","unexpected_error_before_apply")
          exit(47) # to trigger possible rollback
        else
          send_analytics_event("pe_installer","#{action}_fail_type","unexpected_error_during_apply")
          raise e
        end
      end
    end

    option('--detailed-exitcodes') do
      summary _('Provide extra information about the run via exit codes')
      description PuppetX::Util::String::Formatter.join_and_indent([
        _("Provide extra information about the run via exit codes.") + " " + _("If enabled, 'puppet infrastructure configure' will use the following exit codes:"),
        "\n",
        _("0: The run succeeded with no changes or failures; the system was already in the desired state."),
        "\n",
        _("1: The run failed."),
        "\n",
        _("2: The run succeeded, and some resources were changed."),
        "\n",
        _("4: The run succeeded, and some resources failed."),
        "\n",
        _("6: The run succeeded, and included both changes and failures.")])
    end

    option('--install <version>') do
      summary _('Install version submitted by the puppet-enterprise-installer shim')
      description PuppetX::Util::String::Formatter.indent(_("The installer shim includes this flag to distinguish this as a bootstrap run of puppet-infrastructure intended to install or upgrade the specified version."))
    end

    option('--install-method <method>') do
      summary _('Method of install chosen by the user, submitted by the installer shim')
      description <<-EOT
        The installer shim will supply this flag to tell the configure action what method
        of install the user chose (e.g. express, text, web). This is used for sending
        install analytics, unless puppet_enterprise::send_analytics_data = false in
        pe.conf.
      EOT
    end

    option('--upgrade-from <version>') do
      summary _('The currently installed version of PE that we are upgrading from')
      description <<-EOT
        The installer shim will supply this flag if it detects a previous install
        of PE that is different from the version we are currently installing. This
        affects the installation process; notably, cached catalogs are cleared, and
        all services are stopped before packages are installed.

        The fact that we are upgrading is preserved in the file
        #{PuppetX::Puppetlabs::Meep::Infra::Constants::UPGRADE_IN_PROCESS_FILE},
        which is automatically cleared after a successful configure run. If the
        configure is not successful, the value in this file is pulled in as the default
        for --upgrade-from in subsequent executions of configure.

        If that behavior is not desired, the file can be deleted.
      EOT

      default_to do
        configure_action = @face
        infrastructure_interface = configure_action.face
        upgrade_from = infrastructure_interface.upgrade_in_process
        unless upgrade_from.nil?
          message = _("Setting --upgrade-from=%{upgrade_from} based on presence of %{upgrade_in_process_file}.") % { upgrade_from: upgrade_from, upgrade_in_process_file: PuppetX::Puppetlabs::Meep::Infra::Constants::UPGRADE_IN_PROCESS_FILE }
          Puppet.notice(message)
        end
        upgrade_from
      end
    end

    option('--[no-]postgres-migration') do
      summary _('Override the automatic inclusion/exclusion of the postgres migration class')
      description <<-EOT
        Whether to include puppet code to migrate from an older version of
        Postgresql to the current Puppet Enterprise version is determined
        automatically based on the currently installed pe-postgresql-server
        package.

        If the configure run is interrupted or fails after the new pe-postgresql-server
        is installed, but before pg_upgrade is called to perform the data
        migration, then you can include --postgres-migration to force the migration.
      EOT
      default_to do
        configure_action = @face
        infrastructure_interface = configure_action.face

        postgres_server_version = infrastructure_interface.postgres_server_version
        requested_postgres_version = infrastructure_interface.requested_postgres_version
        cmp = infrastructure_interface.version_cmp(postgres_server_version, requested_postgres_version)
        postgres_migration = (cmp == -1)

        if postgres_migration
          message = _("Including pe_install::upgrade::postgres migration class based on current pe-postgresql-server %{postgres_server_version} being older than the version being installed (%{requested_postgres_version}).") % { postgres_server_version: postgres_server_version, requested_postgres_version: requested_postgres_version }
          Puppet.notice(message)
        end
        postgres_migration
      end
    end

    option('--force') do
      summary _('Force the installation even if validation fails.')
      description <<-EOT
        Normally if a step validating the state of the current installation fails,
        the install is halted. If you are certain the validation does not apply,
        you may force the install to proceed by supplying this flag.

        This is not recommended.
      EOT
      default_to { false }
    end

    option('--agent-pid PID') do
      summary 'Process ID of the puppet agent lock that executed MEEP'
      description <<-EOT
        This is used internally if a process with a puppet agent lock is
        launching MEEP to configure PE infrastructure on a node.
      EOT
    end

    option('--[no-]recover') do
      summary _('Attempt to recover PE configuration')
      default_to { nil }
    end

    option('--pe-environmentpath PATH') do
      summary _('The environmentpath to be used when recovering configuration.')
      description <<-EOT
        The puppet-infrastructure face loads from an isolated enterprise
        environmentpath under /opt/puppetlabs/server. But it must recover hiera
        configuration from PE's environmentpath which defaults to
        /etc/puppetlabs/code/environments. If this value has been changed in
        puppet.conf it can be set here to ensure proper configuration recovery.
      EOT
    end

    option('--pe-environment ENVIRONMENT') do
      summary _('The environment to be used when recovering configuration.')
      description <<-EOT
        The puppet-infrastructure face load from an isolated enterprise
        environment under /opt/puppetlabs/server. But it recovers hiera configuration
        from PE's environment as specified in the PE Node Groups. Commonly this
        is the 'production' environment, but if PE infrastructure has been configured
        in a different environment it may be specified here.
      EOT
    end
  end

  def send_analytics_event(category,action,label)
    return if ENV['DISABLE_ANALYTICS']
    PuppetX::Util::Analytics.send_analytics_event(category,action,label) if @is_install || @is_upgrade
  end

  def update_pe_conf_to_disable_analytics
    if ENV['DISABLE_ANALYTICS']
      Puppet.notice(_("Detected DISABLE_ANALYTICS environment variable. Setting puppet_enterprise::send_analytics_data = false in pe.conf."))
      enterprise_conf_dir = infra_default(:ENTERPRISE_CONF_DIR)
      pe_conf = PuppetX::Puppetlabs::Meep::Modify.new(enterprise_conf_dir)
      pe_conf.set_in_pe_conf("puppet_enterprise::send_analytics_data",false)
      PuppetX::Util::Analytics.check_analytics_setting
    end
  end

  def is_ha?
    if File.exist?(PuppetX::Util::ServiceStatus.config_path)
      replicas = PuppetX::Util::ServiceStatus.get_replicas
    else
      replicas = puppet_lookup('puppet_enterprise::ha_enabled_replicas')
    end
    !(replicas.nil? || replicas.empty?)
  end

  def validate_current_installation(options)
    state = {
      :force => options[:force],
      :upgrade_from => options[:upgrade_from],
    }
    rules = []

    if options[:postgres_migration]
      state.merge!({
        :postgres_migration => options[:postgres_migration],
        :pe_postgresql_info => pe_postgresql_info,
        :previous_postgres_version => previous_postgres_version,
        :new_postgres_version => requested_postgres_version,
      })
      rules << PuppetX::Puppetlabs::Meep::Configure::PostgresMigrationSpace
    end

    if !options[:upgrade_from].nil? && !options[:upgrade_from].empty?
      state.merge!({
          :jruby_9k_enabled => puppet_lookup('puppet_enterprise::master::puppetserver::jruby_9k_enabled')
      })
      rules << PuppetX::Puppetlabs::Meep::Configure::ParameterValidation

      if separate_database_node? && master? && local_postgresql_client_is_behind? &&
          version_cmp(options[:upgrade_from], '2019.0') >= 0 
        state.merge!({
          :pe_postgresql_info => pe_postgresql_info,
          :separate_database_node => separate_database_node?,
          :database_host => database_host_parameter,
          :external_database_version => external_database_version,
          :external_database_behind => version_cmp(external_database_version, requested_postgres_version) == -1,
        })
        rules << PuppetX::Puppetlabs::Meep::Configure::ExternalPostgres
      end

    end

    validator = PuppetX::Puppetlabs::Meep::Configure::Validator.new(state)
    validator.register(rules)

    unless validator.valid?
      validator.log_and_fail!
    end
    validator.valid?
  end

  def upgrade_in_process_file
    infra_default(:UPGRADE_IN_PROCESS_FILE)
  end

  def upgrade_in_process
    File.read(upgrade_in_process_file).chomp if File.exist?(upgrade_in_process_file)
  end

  def save_upgrade_in_process(options)
    upgrade_from = options[:upgrade_from]
    return if upgrade_from.nil? || upgrade_from.empty?
    File.write(upgrade_in_process_file, upgrade_from)
  end

  def remove_upgrade_in_process
    FileUtils.rm(upgrade_in_process_file) if File.exist?(upgrade_in_process_file)
  end

  # In order for MEEP to be executed by an agent (or a process with an agent lock),
  # we need to check and see if a passed --agent-pid matches the lock id before locking.
  # If it matches, we can skip locking since the process with the agent lock should
  # only be executing MEEP from a properly isolated stage after it has enforced
  # the rest of the Stage['main'] catalog.
  def lock_unless_agent_pid_matches(agent_pid)
    if agent_pid && agent_pid.to_i == lockfile.lock_pid
      yield
    else
      lock do
        yield
      end
    end
  end

  def puppet_infrastructure_context(user_arguments = [], &block)
    Puppet::Node.indirection.reset_terminus_class

    settings_to_override = {
      :splay                 => false,
      :node_terminus         => 'plain',
      :data_binding_terminus => 'hiera',
      :default_file_terminus => 'file_server',
      :hiera_config          => enterprise_hiera_yaml,
      :user                  => 'pe-puppet',
      :group                 => 'pe-puppet',
    }

    # Puppet.settings.use(:main) was already called in Puppet::Application::Infrastructure#setup()
    # Clear the Puppet::Settings @service_user_available so that future calls to
    # Puppet.settings.use() bubbling up from the Indirector during the catalog run
    # (lib/puppet/indirector/ssl_file.rb, lib/puppet/indirector/report/processor.rb)
    # can act appropriately if pe-puppet has been created during the execution
    # of the catalog.
    Puppet.settings.send(:remove_instance_variable, :@service_user_available) if Puppet.settings.send(:instance_variable_defined?, :@service_user_available)
    puppet_environment_context(default_environmentpath, default_environment, '', settings_to_override, &block)
  end

  def attempt_recovery?(recover = :auto)
    case recover
    when true then true
    when false then false
    # :auto (default) case -- only attempt recovery if we are upgrading the
    # primary node, as that is the only node that has the codedir for hiera
    # overrides in addition to being able to query puppetdb/classifier...
    # TODO: This should be replaced with a classification lookup in Flanders
    else master? && installed?
    end
  end

  def attempt_to_recover_configuration(options, analytics = true)
    if attempt_recovery?(options[:recover])
      recover_options = options.select { |k,v| [:pe_environmentpath, :pe_environment].include?(k) }
      conf = Puppet::Util::Pe_conf.recover_and_save_pe_configuration(recover_options)
      pp_pe_conf = JSON.pretty_generate(conf.pe_conf)
      Puppet.debug(_("Generated the following pe.conf: %{conf}") % { conf: pp_pe_conf })

      if conf.nodes_conf.size > 0
        pp_nodes_conf = JSON.pretty_generate(conf.nodes_conf)
        Puppet.debug(_("Generated the following node specific overrides: %{conf}") % { conf: pp_nodes_conf })
      end
      send_analytics_event("pe_installer","pe_conf_recovery","succeeded") if analytics
      # Because this may have changed Hiera data, we need to clear the cache
      # that the HieraAdapter uses.
      clear_cache
      return true
    else
      return false
    end
  rescue Exception => e
    Puppet.notice(_("Unable to recover PE configuration: %{error}") % { error: e })
    if e.respond_to?(:backtrace)
      Puppet.debug("#{e}:#{e.backtrace.join("\n")}")
    end
    send_analytics_event("pe_installer","pe_conf_recovery","failed") if analytics
    return false
  end

  def classes_to_include(options)
    classes = []

    installing = options.include?(:install) && !options[:install].nil?
    upgrading  = options.include?(:upgrade_from) && !options[:upgrade_from].nil?
    repairing  = options.include?(:install_method) && !options[:install_method].nil? && options[:install_method] == 'repair' && !upgrading

    # We'll try to see if we have any compiler nodes, and if the rule already exists. 
    # If PuppetDB is down during repair, we assume no compilers exist so the rule is not enforced.
    # If classifier is down during repair, we assume the rule exists if pe_compilers exist.
    if installing && !upgrading && !repairing
      pe_compilers_exist = false
      rule_exists = false
    else
      begin
        pe_compilers_exist = !get_nodes_with_role('pe_compiler').empty?
      rescue StandardError, LoadError
        pe_compilers_exist = false
      end
      begin
        nc = PuppetX::Util::Classification.get_node_classifier
        all_groups = PuppetX::Util::Classifier.get_groups(nc)
        pe_master = PuppetX::Util::Classifier.find_group(all_groups, 'PE Master')
        rule_exists = PuppetX::Util::Classification.rule_exists?(pe_master['rule'], ['=', ['trusted', 'extensions', 'pp_auth_role'], 'pe_compiler'])
      rescue StandardError, LoadError
        rule_exists = pe_compilers_exist
      end
    end

    
    # Fresh installs, repairs with PE compilers, or plain configure runs with PE compilers
    enforce_pe_master_rule =  (installing && !upgrading && !repairing) || 
                              (repairing && pe_compilers_exist && !rule_exists) ||
                              (!installing && pe_compilers_exist && !rule_exists)

    pe_install_params = []
    pe_install_params << "bootstrap_to => '#{options[:install]}'," if installing
    pe_install_params << "upgrade_from => '#{options[:upgrade_from]}'," if upgrading
    pe_install_params << "repairing => true," if repairing
    pe_install_params << "enforce_pe_master_rule => #{enforce_pe_master_rule},"

    pe_install_class = <<-EOS
class { pe_install:
  #{pe_install_params.join("\n  ")}
}
    EOS

    if installing || upgrading
      classes.push(
        <<-EOS
class { puppet_enterprise::packages:
  installing => true,
}
        EOS
      )
    end
    classes.push(pe_install_class)

    if options[:postgres_migration]
      classes.push(
        <<-EOS
class { pe_install::upgrade::postgres:
  old_postgres_version => '#{previous_postgres_version}',
  new_postgres_version => '#{requested_postgres_version}',
}
        EOS
      )
    end
    classes.join("\n")
  end

  def default_arguments(options)
    [
      'apply',
      '--execute', classes_to_include(options),
      '--logdest', 'timestamped_console',
    ]
  end

  def default_environmentpath
    '/opt/puppetlabs/server/data/environments'
  end

  def default_environment
    'enterprise'
  end
end
