class puppet_enterprise::params {
  $database_listen_addresses      = '*'

  ######################
  # Postgresql Variables

  # The version of postgresql to be installed in the absence of any development
  # or testing overrides
  $default_postgres_version = '14'

  # Settings (postgresql.conf)
  # this number was taken from the pe-puppetdb module
  $reserved_memory_for_os_in_mb = 512
  $wal_buffers                  = '8MB'
  $log_min_duration_statement   = '5000'

  # Variables that may vary by package version

  # *NOTE* this parameter is a development feature flag that is not suitable
  # for production. Do not use it for anything other than testing bidirectional
  # replication, compatibility of PE with later Postgresql versions or
  # Postgresql migration in development builds. It is currently only implemented as a
  # lookup(), and so must be set in pe.conf or another production hiera layer.
  # (This was done to prevent the parameter from showing up in the classifier)
  $postgres_version_override = lookup('puppet_enterprise::postgres_version_override', { 'default_value' => undef })

  # Whether to use pglogical or Postgres 10+ native logical replication.
  # Expects either 'pglogical' or 'native', defaults to 'pglogical'.
  # (This is another development feature flag. Native logical replication support
  # is not yet tested or supported in production releases.)
  $postgres_logical_replication = lookup('puppet_enterprise::postgres_logical_replication', { 'default_value' => 'pglogical' })

  $postgres_version = pe_pick($postgres_version_override, $default_postgres_version)

  # This is a bit confusing, because '9.6' and '9.4' are both major versions, but
  # we need something to easily allow us to distinguish between 9 and 10+, so we're
  # splitting so as to be able to compare just the first number.
  $postgres_version_array = split($postgres_version, '\.')
  $postgres_first_version_number = Integer($postgres_version_array[0])

  # The newer pe-postgresql packages can be installed side by side, so the packages
  # have the version in the package name.
  $postgres_package_prefix = "pe-postgresql${join($postgres_version_array,'')}"

  # Determines which shared libraries to preload when starting up the postgresql
  # server. For versions before Postgresql 10, we include pglogical. For 10+
  # have the option of relying on the logical replication provided natively.
  $postgres_shared_preload_libraries = ($postgres_logical_replication == 'pglogical') ? {
    true  => ['pglogical'],
    false => [],
  }

  $postgresql_default_database = 'postgres'

  # The actual OS package names for the various postgresql packages.
  # (May vary by postgres major version)
  $postgresql_client_package_name = $postgres_package_prefix
  $postgresql_server_package_name = "${postgres_package_prefix}-server"
  $postgresql_contrib_package_name = "${postgres_package_prefix}-contrib"
  $postgresql_pglogical_package_name = "${postgres_package_prefix}-pglogical"
  $postgresql_pgrepack_package_name = "${postgres_package_prefix}-pgrepack"

  # Where the license key will be written to on all managed nodes.
  $dest_license_key_path = '/etc/puppetlabs/license.key'

  # Puppet path
  case $facts['os']['family'] {
    'windows': {
      $ssl_dir = "${facts['common_appdata']}/PuppetLabs/puppet/etc/ssl"
      $confdir = "${facts['common_appdata']}/PuppetLabs/puppet/etc"
    }
    default: {
      $ssl_dir = '/etc/puppetlabs/puppet/ssl'
      $confdir = '/etc/puppetlabs/puppet'
    }
  }
  $localcacert = "${ssl_dir}/certs/ca.pem"
  $hostcrl     = "${ssl_dir}/crl.pem"

  # Networking configuration
  $ip_version = lookup('puppet_enterprise::ip_version', {
    'value_type' => Variant[ Integer[4,4], Integer[6,6] ],
    'default_value' => 4 })
  $ipv6_only = lookup('puppet_enterprise::ipv6_only', { 'value_type' => Boolean, 'default_value' => false })

  if $ipv6_only and $ip_version != 6 {
    fail('The $ipv6_only flag can only be set when $puppet_enterprise::ip_version is also set to 6.')
  }

  # We bind any HTTP connection to localhost/loopback to hopefully prevent eavesdropping.
  $plaintext_address_ipv4 = '127.0.0.1'
  $plaintext_address_ipv6 = '::1'

  # We bind HTTPS connections to an unspecified address so that a request on any interface/address
  # on the machine should respond. If a user wanted to bind to a specific interface/address, they
  # should provide the actual address via the various class $ssl_address overrides.
  $ssl_address_ipv4 = '0.0.0.0'
  $ssl_address_ipv6 = '::'

  # Standardized parameters for other classes to use when they need a raw IP address so that
  # they do not need to worry about whether the system is IPv4 or IPv6
  $plaintext_address = lookup('puppet_enterprise::plaintext_address', {
    'default_value' => $ip_version ? {
      6 => "[${plaintext_address_ipv6}]",
      4 => 'localhost'
  }})
  $ssl_address = lookup('puppet_enterprise::ssl_address', {
    'value_type' => Puppet_enterprise::Ip,
    'default_value' => $ip_version ? {
      6 => $ssl_address_ipv6,
      4 => $ssl_address_ipv4
    }})

  # Using an IPv6 address in a URL without square brackets results in an invalid URL such as:
  # https://:::8000
  # These variables wrap the raw IPv6 addresses in square brackets as defined in
  # RFC2732 (Format for Literal IPv6 Addresses in URL's)
  # Due to this, these parameters are provided for classes to use when they need an address
  # in a URL string.
  $plaintext_address_url_safe = $plaintext_address
  $ssl_address_url_safe = $ip_version ? {
    6 => "[${ssl_address}]",
    4 => $ssl_address
  }

  $puppetdb_confdir               = '/etc/puppetlabs/puppetdb/conf.d'
  $puppetdb_database_name         = 'pe-puppetdb'
  $puppetdb_database_user         = 'pe-puppetdb'
  $puppetdb_database_read_user    = 'pe-puppetdb-read'
  $puppetdb_migrator_user         = 'pe-puppetdb-migrator'
  $puppetdb_database_password     = ''
  $puppetdb_node_ttl              = '7d'
  $puppetdb_report_ttl            = '14d'
  $puppetdb_listen_port           = '8080'
  $puppetdb_ssl_listen_port       = '8081'
  $puppetdb_ssl_dir               = '/etc/puppetlabs/puppetdb/ssl'
  $puppetdb_auto_configure_sync   = false
  if $trusted['extensions']['pp_auth_role'] == 'pe_compiler' {
    $puppetdb_gc_interval = 0
    $puppetdb_migrate = false
  } else {
    $puppetdb_gc_interval = 60
    $puppetdb_migrate = true
  }

  $console_ssl_listen_port              = 443
  $console_services_listen_port         = '4430'
  $console_services_ssl_listen_port     = '4431'
  $console_services_api_listen_port     = '4432'
  $console_services_api_ssl_listen_port = '4433'
  $console_services_query_cache_ttl     = 30000

  $activity_url_prefix           = '/activity-api'
  $activity_database_name        = 'pe-activity'
  $activity_database_password    = ''

  $classifier_url_prefix             = '/classifier-api'
  # how often should the classes cache be updated in seconds
  $classifier_synchronization_period = 600
  $classifier_prune_threshold        = 7
  $classifier_database_name          = 'pe-classifier'
  $classifier_database_password      = ''
  $classifier_allow_config_data      = lookup('puppet_enterprise_classifier_data_backend_present', {default_value => false})
  #how often should the fact-path cache be updated in seconds
  $classifier_fact_path_sync_period  = 900

  $rbac_url_prefix               = '/rbac-api'
  $rbac_database_name            = 'pe-rbac'
  $rbac_database_password        = ''

  # How often console session is polled to see if the user is still logged in
  $console_session_timeout_polling_frequency_seconds = 60
  # Show warning n seconds before the user is forcibly logged out
  $console_session_timeout_warning_seconds = 120

  $jdbc_ssl_properties   = "?ssl=true&sslfactory=org.postgresql.ssl.jdbc4.LibPQFactory&sslmode=verify-full&sslrootcert=${localcacert}"

  # pcp-broker defaults
  if ! pe_is_integer($facts['processors']['count']) or (Integer($facts['processors']['count'])) <= 1 {
    $pcp_broker_accept_consumers = 2
    $pcp_broker_delivery_consumers = 2
  } else {
    $pcp_broker_accept_consumers   = Integer($facts['processors']['count'])
    $pcp_broker_delivery_consumers = Integer($facts['processors']['count'])
  }

  # pxp-agent paths
  case $facts['os']['family'] {
    'windows':{
      $pxp_agent_base = "${facts['common_appdata']}/PuppetLabs/pxp-agent"
      $pxp_agent_etc = "${pxp_agent_base}/etc"
    }
    default:{
      $pxp_agent_etc = '/etc/puppetlabs/pxp-agent'
    }
  }

  # Pxp isn't available before version 4.3.0. To prevent errors in puppet agent
  # runs on systems with a previous version, opt out of managing pxp related
  # resources.
  $pxp_compatible = versioncmp($facts['puppetversion'], '4.3.0') >= 0

  # eventlog was added in pxp-agent 1.15.9 => puppet-agent 7.1.0
  if $facts['os']['family'] == 'windows' and versioncmp($facts['aio_agent_version'], '7.1.0') >= 0 {
    $pxp_logfile = 'eventlog'
  } else {
    $pxp_logfile = undef
  }

  # Guard puppet settings and pxp-agent settings for the versions they become
  # available. This is used prevent errors in puppet agent runs on systems with
  # a previous version.

  $spool_ttl_compatible = versioncmp($facts['puppetversion'], '4.4.0') >= 0
  $agent_failover_compatible = versioncmp($facts['puppetversion'], '4.6.0') >= 0
  $ping_interval_compatible = versioncmp($facts['puppetversion'], '4.9.0') >= 0
  $pcp_v2_compatible = versioncmp($facts['puppetversion'], '4.9.0') >= 0
  $_agent_version = pe_pick($facts['aio_agent_version'], '0.0.1')
  $pxp_task_compatible = versioncmp($_agent_version, '5.2.0') >= 0
  $task_cache_ttl_compatible = versioncmp($_agent_version, '5.4.0') >= 0
  $task_download_timeout_compatible = versioncmp($_agent_version, '5.5.2') >= 0
  $pxp_proxy_compatible = versioncmp($_agent_version, '5.5.4') >= 0
  # TODO: Remove the check for prerelease versions once 5.5.2 is released.
  $pxp_max_message_size_compatible = versioncmp($_agent_version, '6.18.0') >= 0 or $_agent_version !~ /^\d+\.\d+\.\d+$/
  # As a breadcrumb: the 5.5.0 version of puppet-agent was the first package
  # that included a pxp-agent version that could ignore new configuration
  # settings without failing. We check for at least this version so we can
  # avoid adding this setting if the agent version is too low to accept
  # configuration options that it doesn't understand.
  $pxp_crl_compatible = versioncmp($_agent_version, '5.5.0') >= 0

  $defaults_dir = $facts['os']['family'] ? {
    'Debian'      => '/etc/default',
    /RedHat|Suse/ => '/etc/sysconfig',
    default       => undef,
  }

  ######################
  # Java Variables

  # The major version of java to be installed in the absence of any development
  # or testing overrides.
  $default_java_version = '17'

  # pe-java versioned packages available in the tarball
  $available_java_versions = ['17']

  # This feature flag is for internal puppetlabs testing of newer Java packages.
  # To use it set puppet_enterprise::java_version_override in a hiera layer (typically pe.conf).
  $java_version_override = lookup('puppet_enterprise::java_version_override', { 'default_value' => undef })

  # The major version of the pe-java package to install.
  $java_version = pe_pick($java_version_override, $default_java_version)

  # The pe-java package names now vary by major version so that we can specify, for example,
  # pe-java11 and pe-java17 for testing.
  $java_package_name = "pe-java${java_version}"

  # The java security file that configures Bouncy Castle providers (must be version specific)
  $jdk_security_policy = $facts['fips_enabled'] ? {
    true    => "/opt/puppetlabs/share/jdk${java_version}-fips-security",
    default => "/opt/puppetlabs/share/jdk${java_version}-security",
  }

  # End Java Variables
  ######################

  $cpu = Integer($facts['processors']['count'])

  $memorysize_in_bytes = Integer($facts['memory']['system']['total_bytes'])
  # Saving a mult operation: (1024*1024) => 1048576
  $memorysize_in_mb = $memorysize_in_bytes / 1048576

  $puppetdb_heap_size = puppet_enterprise::calculate_puppetdb_heap_size()
  $puppetdb_java_args = {
    'Xmx' => "${puppetdb_heap_size}m",
    'Xms' => "${puppetdb_heap_size}m",
  }

  $puppetserver_jruby_max_active_instances = puppet_enterprise::calculate_puppetserver_jruby_max_active_instances()

  $puppetserver_heap_size = puppet_enterprise::calculate_puppetserver_heap_size(
    $puppetserver_jruby_max_active_instances )

  $puppetserver_code_cache_int = puppet_enterprise::calculate_puppetserver_reserved_code_cache(
    $puppetserver_jruby_max_active_instances,
    true )

  $puppetserver_code_cache = $puppetserver_code_cache_int ? {
    undef   => undef,
    default => "${$puppetserver_code_cache_int}m"
  }

  $puppetserver_java_args = puppet_enterprise::calculate_puppetserver_java_args(
    $puppetserver_jruby_max_active_instances
  )

  $console_services_heap_size = 256
  $console_services_java_args = {
    'Xmx' => "${console_services_heap_size}m",
    'Xms' => "${console_services_heap_size}m",
  }

  $builtin_bolt_content_dir = [
    '/opt/puppetlabs/server/data/builtin_bolt_content',
    '/opt/puppetlabs/server/apps/bolt-server/lib/modules/default_builtin_content'
  ]

  $orchestrator_heap_size = 704
  $orchestrator_java_args = { 'Xmx' => "${orchestrator_heap_size}m",
                              'Xms' => "${orchestrator_heap_size}m",
                            }

  $host_action_collector_heap_size = 256
  $host_action_collector_java_args = {
    'Xmx' => "${host_action_collector_heap_size}m",
    'Xms' => "${host_action_collector_heap_size}m",
  }

  # Estimate off-heap usage by the JVM of 10% for PDB and console, 50% for puppetserver and orchestrator
  $memory_used_by_pe_services_in_mb = Integer($puppetdb_heap_size * 1.1) +
                                      Integer($puppetserver_heap_size * 1.5) +
                                      Integer($console_services_heap_size * 1.1) + 
                                      $puppetserver_code_cache_int + 
                                      Integer($orchestrator_heap_size * 1.5) +
                                      $host_action_collector_heap_size +
                                      512 # Extremely rough estimate for ace server, bolt server, nginx, pxp-agent, puppet service
  $avail_mem_in_mb = $memorysize_in_mb - $reserved_memory_for_os_in_mb - $memory_used_by_pe_services_in_mb
  $avail_mem_in_gb = $avail_mem_in_mb / 1024
  $work_mem_calculation = pe_clamp( 4, Integer((($avail_mem_in_gb / 8.0) + 0.5)), 16)

  $work_mem = "${work_mem_calculation}MB"

  # set a min if too low
  $ecs_value = pe_max((($avail_mem_in_mb * 3) / 5), 128)
  $effective_cache_size = "${$ecs_value}MB"

  # Maximum value: 4096, Minimum value: 256
  $sb_value = pe_max(pe_min((($avail_mem_in_mb * 1) / 4), 4096), 256)
  $shared_buffers = "${$sb_value}MB"

  # Puppet user by OS
  case $facts['os']['family'] {
    'AIX':{
      $puppet_user    = 'puppet'
      $puppet_group   = 'puppet'
    }
    default:{
      $puppet_user    = 'pe-puppet'
      $puppet_group   = 'pe-puppet'
    }
  }

  # Root user by OS
  $root_user = $facts['os']['family'] ? {
    'windows' => 'S-1-5-32-544', # Adminstrators
    default => 'root',
  }

  # Root group by OS
  $root_group = $facts['os']['family'] ? {
    'AIX' => 'system',
    'windows' => 'S-1-5-32-544', # Adminstrators
    'Darwin' => 'wheel',
    default => 'root',
  }

  # Root mode by OS
  $root_mode = $facts['os']['family'] ? {
    'windows' => '0664',         # Both user and group need write permission
    default => '0644',
  }

  # We would like to run silent with regards to the deprecation warning around
  # the package type's allow_virtual parameter. However, the allow_virtual
  # parameter did not exist prior to Puppet 3.6.1. Therefore we have a few
  # places where we need to set package resource defaults dependent on what
  # version of Puppet the client is running (so long as we support clients
  # older than 3.6.1). Performing the calculation here as it is used in more
  # than one location elsewhere in the module.
  $supports_allow_virtual = versioncmp($facts['puppetversion'],'3.6.1') >= 0
  $allow_virtual_default = $supports_allow_virtual ? {
    true  => true,
    false => undef,
  }

  # Because the default for OracleLinux is up2date...
  if $facts['os']['name'] == 'OracleLinux' {
    $package_options = {
      provider      => 'yum',
      allow_virtual => $allow_virtual_default,
    }
  } elsif $facts['os']['name']== 'SLES' {
    $package_options = {
      provider      => 'zypper',
      allow_virtual => $allow_virtual_default,
    }
  } else {
    $package_options = {
      allow_virtual => $allow_virtual_default,
    }
  }

  # Client tool configuration
  $client_tools_confdir = '/etc/puppetlabs/client-tools'


  # PE Configuration Mount Point
  $enterprise_conf_path = '/etc/puppetlabs/enterprise'
  $enterprise_conf_mountpoint = 'enterprise_conf'

  # PE Inventory
  $package_inventory_enabled_file = $facts['os']['family'] ? {
    'windows' => "${facts['common_appdata']}/PuppetLabs/puppet/cache/state/package_inventory_enabled",
    default   => '/opt/puppetlabs/puppet/cache/state/package_inventory_enabled',
  }
}
