# frozen_string_literal: true

require 'bolt/error'
require 'concurrent/promise'
require 'json'

require 'plan_runner/service/encryption'
require 'plan_runner/service/http_client'
require 'plan_runner/service/runner'

module PlanRunner
  module Service
    class ProcessManager
      # rubocop:disable Lint/SuppressedException
      # We need to ignore a several exceptions that may be thrown
      # when reading IO or searching for processes.

      attr_accessor :process_table,
                    :logger,
                    :config

      def initialize(config)
        @logger = Logging.logger[self]
        @config = config

        @process_table = {}
        @encryption_key = @config[:encryption_key]
      end

      #
      # These top methods can be considered this class' public api.
      #

      # clear any finished processes from the process table,
      # block if we'd go over our process budget until we have space,
      # fork, and then update the process table
      def run(data, &block)
        clear_table_of_completed_processes

        while @process_table.length >= @config[:max_plans]
          @logger.info("Reached maximum allowed concurrent plans: " \
                       "#{@process_table.length}/#{@config[:max_plans]}, " \
                       "waiting until one finishes.")

          _pid, process_status = wait_for_any_process
          clear_table_of_completed_process(process_status)
        end

        pipe_reader, pipe_writer = IO.pipe

        pid = managed_fork(block, data, pipe_writer)

        pipe_writer.close

        if @process_table[pid]
          @logger.error("Existing entry in process table with pid: #{pid}, overwriting!")
        end

        @process_table[pid] = {
          output: pipe_reader,
          plan_details: {
            name: data.dig('input', 'plan_name'),
            id: data['plan_id']
          }
        }
        pid
      end

      StubProcess = Struct.new(:pid, :exitstatus)

      def clear_process_table
        @logger.info("Clearing all processes")
        @process_table.each_pair do |pid, info|
          _pid, status = fetch_given_process_result(pid)
          if status
            @logger.debug("Process #{pid} already completed")
          else
            terminate_pid(pid, info)
            _pid, status = fetch_given_process_result(pid)
          end
        rescue Errno::ECHILD => _e
          @logger.warn("Process #{pid} in process table could not be found by the kernel")
        ensure
          if status.nil?
            # Create a stubbed process so that we can still clear the process from the table
            # and more importantly read from the output reader.
            status = StubProcess.new(pid, 'unknown')
          end
          clear_table_of_completed_process(status)
        end
      end

      def clear_table_of_completed_process(process)
        info = @process_table[process.pid]
        details = info[:plan_details]
        pipe_output = read_from_pipe(info[:output])

        log_message = "
                      Process #{process.pid} executing '#{details[:name]}'" \
                      "(#{details[:id]}), exited with #{process.exitstatus}."

        unless pipe_output.empty?
          log_message += " Output: #{pipe_output}"
        end

        log_message = log_message.lines.map(&:strip).join("\n")
        # Only log at the WARN level if there was something in the pipe_output
        # and the process failed. Otherwise assume that failures have already
        # been logged elsewhere so as not to not fill logs with warnings for
        # things like plans that failed during execution.
        if process.exitstatus != 0 && !pipe_output.empty?
          @logger.warn(log_message)
        else
          @logger.debug(log_message)
        end

        info[:output].close
        @process_table.delete(process.pid)
      end

      def read_from_pipe(pipe_reader)
        output = String.new
        # read anything flushed when the proceess ended
        begin
          loop { output << pipe_reader.read_nonblock(4096) }
        rescue IO::WaitReadable, Errno::EAGAIN, EOFError
        end
        output
      end

      def clear_table_of_completed_processes
        while (info = fetch_any_process_result)
          (pid, process_status) = info
          clear_table_of_completed_process(process_status)
        end
      rescue Errno::ECHILD => _e
        # Create a stubbed process so that we can still clear the process from the table
        # and more importantly read from the output reader.
        process_status = StubProcess.new(pid, 'unknown')
        begin
          clear_table_of_completed_process(process_status)
        rescue StandardError => _e
          # If we fail again just let it go
        end
      end

      def terminate_plan_id(plan_id)
        pid_entry = @process_table.find { |_, value| value.dig(:plan_details, :id) == plan_id }
        terminate_pid(pid_entry[0], pid_entry[1])
      end

      def terminate_pid(pid, info)
        @logger.debug("Sending process #{pid} executing plan #{info[:plan_details][:id]} the TERM signal")
        signal("TERM", pid)
        sleep 0.001

        _pid, status = fetch_given_process_result(pid)
        if status.nil?
          signal("KILL", pid)
          @logger.warn(
            "Process #{pid} executing plan #{info[:plan_details][:id]} " \
            "did not quit gracefully, forcefully stopped"
          )
        end
      end

      def managed_fork(block, data, pipe_writer)
        @logger.info("Forking new process to run plan " \
                     "'#{data.dig('input', 'plan_name')}' (#{data.dig('plan_id')})")

        internal_fork do
          # Reset file descriptors, IO pipes etc.
          set_signal_handlers
          reset_file_handles(pipe_writer)
          config = reinitialize_configuration(@config)

          # Actually run the thing now
          block.call(config, data)

          close_io($stdout, $stderr)
        rescue StandardError => e
          # Make an attempt to use the logger to log any failures, but if that fails puts to the
          # pipe so that the parent process can log it.
          #
          # We only attempt to use the logger if the type is Bolt::Error, which could only be thrown
          # after the full re-init of the fork has finished. Otherwise we risk trying to write to the
          # logger before it was reinitialized
          if e.is_a?(Bolt::Error)
            @logger.warn("Plan failed with #{e.message}")
          else
            pipe_writer.puts e.full_message
          end
          exit(1)
        end
      end

      def reset_file_handles(pipe_writer)
        begin
          $stdin.close
          $stdout.reopen(pipe_writer)
          $stderr.reopen(pipe_writer)
        rescue Errno::EBADF => _e
          puts("Could not reopen stdout #{$stdout} or stderr #{stderr} in child process")
        end

        file_descriptors = Dir['/proc/self/fd/*']
                           .map    { |f| File.basename(f).to_i }
                           .filter { |f| f >= 3 }

        close_io(*file_descriptors)
      end

      def reinitialize_configuration(config)
        # Re-open the file descriptor for the log file so that we don't
        # clobber the parent process' descriptor per
        # https://github.com/TwP/logging/blob/master/examples/fork.rb
        Logging.reopen
        new_http_client = HttpClient.new(config[:orch_url], config)
        config.merge({ http_client: new_http_client })
      end

      def fetch_given_process_result(pid)
        Process.wait2(pid, Process::WNOHANG)
      end

      def fetch_any_process_result
        Process.wait2(-1, Process::WNOHANG)
      end

      def wait_for_any_process
        Process.wait2(-1)
      end

      def signal(sig, pid)
        Process.kill(sig, pid)
      end

      def close_io(*ios)
        ios.each do |fd|
          fd.close
        rescue StandardError
        end
      end

      def internal_fork(&block)
        fork(&block)
      end

      def set_signal_handlers
        Signal.trap("TERM") do
          exit
        end
      end
      # rubocop:enable Lint/SuppressedException
    end
  end
end
