# frozen_string_literal: true

require 'json'
require 'logging'
require 'plan_runner/execution/environment_plan'
require 'plan_runner/service/http_client'
require 'plan_runner/service/process_manager'
require 'concurrent/promise'

module PlanRunner
  module Service
    class Runner
      # writers for test mocking
      attr_writer :stop, :http_client, :process_manager

      def initialize(config)
        @logger = Logging.logger[self]
        @config = config

        @logger.debug("Initializing Http Client")
        @http_client = HttpClient.new(@config[:orch_url], @config)

        @stop = false
        @process_manager = ProcessManager.new(config)
        @rsa_key = @config[:encryption_key]
        @http_failures = 0
      end

      def stop!
        @stop = true
      end

      def run!
        http_promise = Concurrent::Promise.execute { operations_post }
        until @stop
          begin
            if http_promise.complete?
              process_response(*http_promise.value)
              http_promise = Concurrent::Promise.execute { operations_post }
            else
              # We don't need to loop that fast, and slowing down the loop
              # means we don't spam the syscall that clear_table uses below forever
              sleep 0.5
            end
            @process_manager.clear_table_of_completed_processes
          rescue StandardError => e
            @logger.warn("Unexpected exception during new operations API call: #{e.full_message}")
            @http_failures += 1
            handle_api_failure(failure_count: @http_failures)
            http_promise = Concurrent::Promise.execute { operations_post }
          end
        end
        @logger.info("Shutting down the main process")
        # If we got a new plan and the loop hasn't processed it yet, try to stop it
        if http_promise.complete?
          response, code = http_promise.value
          if code == 200
            @logger.info("Stopping new plan #{response['plan_id']}")
            # Give it a couple seconds to make an attempt at the http call
            send_stop_plan_result(response['plan_id']).value(5)
          end
        end
        @process_manager.clear_process_table
      end

      def operations_post
        url = '/v2/private/plans/plan_runner/operations'
        @logger.debug("POST to API to look for new operations")
        @http_client.post_with_orch_cert(url, { public_key: @rsa_key.public_key })
      end

      def process_response(response, http_code)
        case http_code
        when 200
          @http_failures = 0
          case response['operation']
          when 'run_plan'
            pid = @process_manager.run(response) do |config, data|
              run_plan(config, data)
            end
            Concurrent::ScheduledTask.execute(response['timeout']) do
              _pid, status = @process_manager.fetch_given_process_result(pid)
              if status.nil?
                timeout_plan(response['plan_id'])
              end
            rescue StandardError => e
              @logger.error(e.full_message)
            end
          when 'stop_plan'
            stop_plan(response['plan_id'])
          else
            @logger.error("Unknown operation response from Orchestrator")
          end
        when 204
          @http_failures = 0
          @logger.debug("No operations in queue")
        else
          @logger.warn("Failed new operations API call with http code #{http_code}: #{response}")
          @http_failures += 1
          handle_api_failure(failure_count: @http_failures)
        end
      end

      def handle_api_failure(failure_count:)
        exp = [failure_count, 7].min
        wait_time = 2**exp
        @logger.info("Failed querying #{failure_count} times, waiting #{wait_time} seconds to retry")
        sleep wait_time
      end

      def run_plan(config, plan_spec)
        input = plan_spec['input']
        id = plan_spec['plan_id'].to_s
        aes_key = @rsa_key.decrypt(plan_spec['plan_key'])

        plan = Execution::EnvironmentPlan.new(
          config,
          id,
          input['plan_name'],
          input['environment'],
          input['parameters'],
          input['sensitive_parameters'],
          aes_key
        )
        plan.run
        plan.submit_result

        raise plan.error if plan.failed?
      end

      def stop_plan(plan_id)
        @logger.debug("Terminating plan #{plan_id}")
        @process_manager.terminate_plan_id(plan_id)
        send_stop_plan_result(plan_id)
      rescue StandardError => e
        @logger.debug("Exception thrown while stopping plan #{plan_id}: #{e.message}")
      end

      def send_stop_plan_result(plan_id)
        url = '/v2/private/plans/' + plan_id.to_s + '/result'
        # PUT a 'stopped' plan result back up to Orchestrator, but do it in a promise
        # so the main thread can loop again and grab the next operation right away.
        Concurrent::Promise.execute do
          @logger.debug("PUT to API to finish plan #{plan_id} with 'stopped' result")
          response, http_code = @http_client.put_with_orch_cert(
            url,
            {
              'result' => {
                'msg' => 'plan stopped',
                'kind' => 'puppetlabs.tasks/job-stopped-error',
                'issue_code' => 'JOB_STOPPED',
                'details' => { 'plan_id' => plan_id }
              },
              'status' => 'failure'
            }
          )
          if http_code != 200
            @logger.error("Failed to send stopped plan result with http code #{http_code}: #{response}")
          end
        end
      end

      # Pretty much the same as stop plan, but the result indicates the plan timed out
      # and the PUT to Orchestrator doesn't need to be in a promise (timeouts are already
      # inside a ScheduledTask which is the same as a promise).
      def timeout_plan(plan_id)
        url = '/v2/private/plans/' + plan_id.to_s + '/result'
        @logger.info("Plan #{plan_id} timed out, terminating")
        @process_manager.terminate_plan_id(plan_id)
        @logger.debug("PUT to API to finish plan #{plan_id} with 'stopped' result")
        response, http_code = @http_client.put_with_orch_cert(
          url,
          {
            'result' => {
              'msg' => 'plan timed out',
              'kind' => 'puppetlabs.tasks/job-timeout-error',
              'issue_code' => 'JOB_TIMEOUT',
              'details' => { 'plan_id' => plan_id }
            },
            'status' => 'failure'
          }
        )
        if http_code != 200
          msg = "Failed API response with code #{http_code}: #{response}"
          raise Bolt::Error.new(msg, 'bolt.plan-executor/errored-response')
        end
      rescue StandardError => e
        @logger.debug("Exception thrown while stopping plan #{plan_id}: #{e.message}")
      end
    end
  end
end
