# This Bolt plan is run over the orchestrator. It will take a list of
# agent nodes and regenerate the certificate on those nodes.
#
# Be careful when using a node_type other than agent, as other considerations
# may need to be made when regenerating a certificate on an infrastructure node,
# such as restarting services.
#
# $agent may be multiple nodes (comma-separated), as long as they are all the same type with the
# same parameters.
#
# Alternatively, $agent_pdb_query can be used to specify a group of nodes to run this plan against
# based on the result of a PQL query. This query should return 'certname', so something like
#    'facts[certname] { name = "domain" and value ~ "agent.node.com" }'
# If both $agent_pdb_query and $agent are specified, the result of
# the query and the agents specified in $agent will be merged together.
#
# Only agents installed under the root user are supported at this time.
#
# By default, this plan cleans and signs certs on the CA via API endpoint. If this is problematic
# for some reason, set $use_puppetserver_cli = true to fall back to calling out to the
# puppetserver ca cli instead. This is much slower.
#
# When running this plan via orchestrator (default behavior for puppet infra run regenerate_agent_certificate),
# we rely on the websocket staying open between cleaning the old cert and downloading
# the new one. If you have reason to believe that websockets may unexpectedly close in your
# environment, it may be safer to run this command with the --use-ssh or --use-winrm flags.
#
# Steps the plan will take:
# 1. If dns_alt_names is defined as a parameter, verify that puppetserver conf files in
#    /etc/puppetlabs/puppetserver/conf.d have certificate-authority.allow-subject-alt-names
#    set to true. This should be managed to true by default. If dns_alt_names is not passed
#    in as a parameter, make sure it is not set in puppet.conf as well or fail otherwise.
# 2. Disable the agent on the primary and agent nodes with puppet agent --disable.
# 3. Stop the puppet service on the agent nodes, if it is running.
# 4. If the dns_alt_names plan parameter is not set, check if dns_alt_names is set in puppet.conf
#    on the agent node. If so, do not regenerate the cert on that node until either that setting is
#    removed, or the plan is rerun with the dns_alt_names parameter.
# 5. Backup existing certs in /etc/puppetlabs/puppet/ssl or C:/ProgramData/PuppetLabs/puppet/etc/ssl
#    on agent nodes. They are backed up to a directory in the same location with a _bak_<timestamp> suffix.
# 6. If extension_requests or custom_attributes is defined, these are set in in the
#    csr_attributes.yaml file, located at /etc/puppetlabs/puppet or C:/ProgramData/PuppetLabs/puppet/etc.
# 7. Remove the cached catalog from the agent nodes.
# 8. If dns_alt_names is defined, run puppet config set --section main
#    dns_alt_names <dns_alt_names value> on the agent nodes.
# 9. Clean the cert for the agent nodes from the CA.
# 10. Run 'puppet ssl clean' on the agent nodes. --localca is specified if $clean_crl == true.
# 11. Run 'puppet ssl submit_request' on the agent nodes to generate a CSR and submit it to the primary.
# 12. Wait for the CSR to show up on the CA, or the signed cert if autosigning is enabled. If
#     dns_alt_names was not specified, verify the CSR does not contain any alt names. Sign
#     the cert if needed. If dns_alt_names was specified, check that the cert contains those
#     alt names.
# 13. Run 'puppet ssl download_cert' on the agent nodes to download signed certificate from the primary.
# 14. Run 'puppet ssl verify' to ensure the new cert is working correctly.
# 15. If manage_pxp_service is set to true (default), restart the pxp-agent
#     service on the agent node so the service picks up the new cert.
# 16. Restore the puppet service to its state before the plan started.
# 17. Re-enable the agent on the primary with puppet agent --enable.
plan enterprise_tasks::agent_cert_regen(
  TargetSpec $primary                 = 'localhost',
  String $node_type                   = 'agent',
  Boolean $manage_pxp_service         = true,
  Boolean $clean_crl                  = false,
  Boolean $force                      = false,
  Boolean $use_puppetserver_cli       = false,
  Optional[TargetSpec] $agent         = undef,
  Optional[String] $agent_pdb_query   = undef,
  Optional[Hash] $extension_requests  = undef,
  Optional[Hash] $custom_attributes   = undef,
  Optional[String] $dns_alt_names     = undef,
) {
  # Developer notes:
  # Things get a little tricky in the area between when we clean the cert from the
  # server and the node, and when the pxp-agent service is restarted, where it will pick up
  # the new cert. Once the cert is gone, we can no longer download new tasks, as this
  # utilizes curl underneath and needs to load the certs each time. We are relying
  # on the existing websocket staying open through this process to be able to continue
  # to issue run_commands to the node. If pxp-agent ever tries establishing a new
  # websocket in the middle of this process (customer environments can be finicky
  # with websockets), things will probably fail. Unfortunately, we don't really
  # have a good way around this when you're using the orchestrator to run this plan.
  # Using SSH/WinRM may be more reliable, but is not an option for all customers.
  #
  # We use tasks as much as possible in this plan in order to more easily parallelize
  # the actions. Since the puppet binary can live in different places, especially on
  # Windows where we make it relatively easy to install wherever you want, we need
  # to determine the path to the binary when we want to run puppet commands. We do this
  # in tasks via the puppet_helper library. 
  #
  # If we were to do puppet ssl commands via a task, and ensure we do 'puppet ssl clean'
  # before 'puppetserver ca clean', we could conceivably continue to use this ssl task, since it
  # was downloaded to the node prior to cleaning the cert, and will remain in the
  # task cache. However, if the cached task becomes invalidated (TTL is 14 days,
  # but there is still an edge case where this could happen) or if a user has set
  # TTL to 0, pxp-agent will attempt to download the task again and fail. Therefore,
  # we must use run_command instead, using the path to the puppet binary discovered
  # for that node via the enterprise_tasks::get_puppet_bin_path task earlier in the plan.
  # It is harder to parallelize this since the path is potentially unique for each node.
  # We try to mitigate this by grouping nodes with the same path together. In practice,
  # it's likely all nodes will share the same installation path, meaning we only run
  # through the loop with the run_command once.
  #
  # When we encounter errors in the plan when running via orchestrator, those errors
  # are printed automatically in red error text. When running via Bolt, we have to use
  # log::error to print out the actual errors, as they are not printed at the default
  # logging level for the console. The errors may look a bit different when orchestrator
  # handles it vs. Bolt, but the problem should be sufficiently exposed either way. The
  # call to log::error does nothing when running via orchestrator, so we shouldn't get errors
  # printed twice.
  #
  # If adding or changing steps to this plan ensure that you:
  #  - Modify the $agent_ variables appropriately to pass through passing agents to the next step.
  #  - Add a block at the end to print errors
  #  - Modify the very end of the plan where we find critical errors and any error and
  #    print a message/fail_plan accordingly

  #TODO:
  # See if we can make errors at the end red
  # Skip rest of plan if no more nodes remaining

  # The task utilizing the API endpoints directly is much faster, but this option
  # exists in case a user ends up having problems with it.
  if $use_puppetserver_cli {
    $puppetserver_ca_clean_task = 'enterprise_tasks::puppetserver_ca_clean'
    $puppetserver_ca_sign_task = 'enterprise_tasks::puppetserver_ca_sign'
  } else {
    $puppetserver_ca_clean_task = 'enterprise_tasks::puppetserver_ca_clean_api'
    $puppetserver_ca_sign_task = 'enterprise_tasks::puppetserver_ca_sign_api'
  }

  if !$agent and !$agent_pdb_query {
    fail_plan('You must specify either $agent or $agent_pdb_query.')
  }
  if $agent_pdb_query {
    $agent_pdb_query_result = puppetdb_query($agent_pdb_query)
    $agent_certnames = $agent_pdb_query_result.map |$r| { $r['certname'] }
    if $agent {
      $agents = (get_targets($agent_certnames) + get_targets($agent)).unique
    } else {
      $agents = get_targets($agent_certnames)
    }
  } else {
    $agents = get_targets($agent)
  }

  enterprise_tasks::message('agent_cert_regen', "Testing connection to ${agents.length} node(s)")
  $test_connection_result = run_task(enterprise_tasks::test_connect, $agents, '_catch_errors' => true)
  if !($test_connection_result.error_set.empty) {
    log::error($test_connection_result.error_set)
  }
  $agents_with_connection = $test_connection_result.ok_set.targets

  enterprise_tasks::message('agent_cert_regen', 'Verifying node roles')
  enterprise_tasks::verify_node($primary, 'primary', $force)
  enterprise_tasks::verify_node($agents_with_connection, $node_type, $force)

  if $dns_alt_names {
    run_plan(enterprise_tasks::is_subject_alt_names_allowed, primary => $primary, force => $force)
  }
  $allow_subject_alt_names = $dns_alt_names ? {
    undef   => false,
    default => true
  }

  # Set puppet-agent feature on all agent targets
  $agents_with_connection.each |$a| {
    enterprise_tasks::set_feature($a, 'puppet-agent', true)
  }

  # Get the path to the puppet binary on each node
  enterprise_tasks::message('agent_cert_regen', "Finding the puppet binary path on ${agents_with_connection.length} node(s)")
  $get_puppet_bin_path_result = run_task(enterprise_tasks::get_puppet_bin_path, $agents_with_connection,
    '_catch_errors' => true,
  )
  if !($get_puppet_bin_path_result.error_set.empty) {
    log::error($get_puppet_bin_path_result.error_set)
  }
  $agents_for_disable = $get_puppet_bin_path_result.ok_set.targets
  $get_puppet_bin_path_result.each |$r| {
    $r.target.set_var('puppet_bin', $r.value['puppet_bin'])
  }
  $puppet_bin_paths = $get_puppet_bin_path_result.map |$r| { $r.value['puppet_bin'] }.unique

  $disable_enable_result = enterprise_tasks::with_agent_disabled([$primary] + $agents_for_disable, true) |$disabled_targets| {
    # Because the primary is in the list, we need to filter rather than just using the list as-is
    $agents_for_regen = $agents_for_disable.filter |$a| { $a in $disabled_targets }

    # Save state of service on all nodes to restore at the end
    enterprise_tasks::message('agent_cert_regen', "Recording status of puppet service on ${agents_for_regen.length} node(s)")
    $puppet_service_result = run_task(service, $agents_for_regen,
      action => 'status',
      name => 'puppet',
      '_catch_errors' => true,
    )
    if !($puppet_service_result.error_set.empty) {
      log::error($puppet_service_result.error_set)
    }
    $puppet_running_agents = ResultSet.new($puppet_service_result.ok_set.filter |$result| { $result.value['status'] == 'running' }).targets

    # Record group of nodes we are stopping the agent on so we ensure we start it again,
    # no matter what.
    enterprise_tasks::message('agent_cert_regen', "Stopping puppet service on ${puppet_running_agents.length} node(s)")
    $pe_services_stop_result = run_task(enterprise_tasks::pe_services, $puppet_running_agents,
      state => 'stopped',
      '_catch_errors' => true,
    )
    if !($pe_services_stop_result.error_set.empty) {
      log::error($pe_services_stop_result.error_set)
    }
    $puppet_stopped_agents = $pe_services_stop_result.ok_set.targets
    $agents_for_dns_alt_names_check = $puppet_service_result.ok_set.targets - $pe_services_stop_result.error_set.targets

    # Fail early if puppet.conf has a dns_alt_names setting and agent_cert_regen is called without
    # dns_alt_names parameter. Otherwise CSR will be generated with dns_alt_names and signing will
    # fail later.
    if !$dns_alt_names {
      enterprise_tasks::message('agent_cert_regen', "Checking for existing dns_alt_names on ${agents_for_dns_alt_names_check.length} node(s)")
      $dns_alt_names_result = run_task(enterprise_tasks::get_puppet_config_setting, $agents_for_dns_alt_names_check,
        setting => 'dns_alt_names',
        '_catch_errors' => true,
      )
      if !($dns_alt_names_result.error_set.empty) {
        log::error($dns_alt_names_result.error_set)
      }
      $nodes_with_dns_alt_names_results = $dns_alt_names_result.ok_set.filter_set |$result| { !$result.value['value'].empty }
      $nodes_with_dns_alt_names_set = $nodes_with_dns_alt_names_results.targets
      if !($nodes_with_dns_alt_names_set.empty) {
        enterprise_tasks::message('agent_cert_regen', 'On the following nodes, dns_alt_names is set in puppet.conf, but the plan was run without a value for the dns_alt_names parameter. Either remove this setting from puppet.conf on these nodes, or run this plan with the dns_alt_names parameter matching the value on these nodes.')
        $dns_alt_names_to_print = $nodes_with_dns_alt_names_results.map |$r| { "  ${r.target.host} has dns_alt_names set to ${r.value['value']}" }
        enterprise_tasks::message('agent_cert_regen', "  ${dns_alt_names_to_print.join("\n                    ")}")
      }
      $agents_for_backup = $agents_for_dns_alt_names_check - $dns_alt_names_result.error_set.targets - $nodes_with_dns_alt_names_set
    } else {
      $dns_alt_names_result = ResultSet.new([])
      $nodes_with_dns_alt_names_set = []
      $agents_for_backup = $agents_for_dns_alt_names_check
    }

    # Backup certs in case we fail in the process somewhere after removing them and they need to be restored.
    enterprise_tasks::message('agent_cert_regen', "Backing up certs on ${agents_for_backup.length} node(s)")
    $backup_result = run_task(enterprise_tasks::backup_certs, $agents_for_backup, '_catch_errors' => true)
    if !($backup_result.error_set.empty) {
      log::error($backup_result.error_set)
    }
    $agents_for_csr_attributes = $backup_result.ok_set.targets

    # Set extension requests or custom attributes on the nodes, if either parameter was passed into the plan.
    if $extension_requests or $custom_attributes {
      enterprise_tasks::message('agent_cert_regen', "Setting extension requests and/or custom attributes in csr_attributes.yaml on ${agents_for_csr_attributes.length} node(s)")
      $set_csr_attributes_result = run_task(enterprise_tasks::set_csr_attributes, $agents_for_csr_attributes,
        extension_requests => $extension_requests,
        custom_attributes => $custom_attributes,
        '_catch_errors' => true,
      )
      if !($set_csr_attributes_result.error_set.empty) {
        log::error($set_csr_attributes_result.error_set)
      }
      $agents_for_remove_cached_catalog = $set_csr_attributes_result.ok_set.targets
    } else {
      $set_csr_attributes_result = ResultSet.new([])
      $agents_for_remove_cached_catalog = $agents_for_csr_attributes
    }

    # The cached catalog may include references to old certs (especially if we're running this after a CA rebuild)
    enterprise_tasks::message('agent_cert_regen', "Removing cached catalog on ${agents_for_remove_cached_catalog.length} node(s)")
    $remove_cached_catalog_result = run_task(enterprise_tasks::remove_cached_catalog, $agents_for_remove_cached_catalog,
      '_catch_errors' => true,
    )
    if !($remove_cached_catalog_result.error_set.empty) {
      log::error($remove_cached_catalog_result.error_set)
    }
    $agents_for_set_dns_alt_names = $remove_cached_catalog_result.ok_set.targets

    # Setting dns_alt_names in puppet.conf will cause a CSR to be generated with those alt names
    if $dns_alt_names {
      enterprise_tasks::message('agent_cert_regen', "Setting dns_alt_names in puppet.conf to ${dns_alt_names} on ${agents_for_set_dns_alt_names.length} node(s)")
      $set_dns_alt_names_result = run_task(enterprise_tasks::set_puppet_config_setting, $agents_for_set_dns_alt_names,
        setting => 'dns_alt_names',
        value => $dns_alt_names,
        '_catch_errors' => true,
      )
      if !($set_dns_alt_names_result.error_set.empty) {
        log::error($set_dns_alt_names_result.error_set)
      }
      $agents_for_ca_clean = $agents_for_set_dns_alt_names - $set_dns_alt_names_result.error_set.targets
    } else {
      $set_dns_alt_names_result = ResultSet.new([])
      $agents_for_ca_clean = $agents_for_set_dns_alt_names
    }

    enterprise_tasks::message('agent_cert_regen', "Cleaning certs for ${agents_for_ca_clean.length} node(s) on primary")
    $puppetserver_ca_clean_result = run_task($puppetserver_ca_clean_task, $primary,
      certname => $agents_for_ca_clean.map |$a| { $a.host },
      '_catch_errors' => true,
    )
    if !($puppetserver_ca_clean_result.error_set.empty) {
      log::error($puppetserver_ca_clean_result.error_set)
    }
    if !($puppetserver_ca_clean_result.ok) {
      # If this is a handled error, the result should have the 'errored_certnames' key. If it's an unexpected exception,
      # it won't, so assume cleaning all nodes failed.
      $puppetserver_ca_clean_errored_nodes = $puppetserver_ca_clean_result.first.error.details['errored_certnames'] ? {
        undef => $agents_for_ca_clean,
        default => $puppetserver_ca_clean_result.first.error.details['errored_certnames']
      }
    } else {
      $puppetserver_ca_clean_errored_nodes = []
    }
    $agents_for_ssl_clean = $agents_for_ca_clean - get_targets($puppetserver_ca_clean_errored_nodes)

    # Run 'puppet ssl clean' on each node. We must use run_command here, since we may not be able to download
    # new tasks to the node once certs are cleaned on either the server or the node.
    enterprise_tasks::message('agent_cert_regen', "Cleaning SSL directory on ${agents_for_ssl_clean.length} node(s)")
    $localca = $clean_crl ? {
      true => '--localca',
      default => '',
    }
    $ssl_clean_results_array = $puppet_bin_paths.reduce([]) |$memo, $path| {
      $nodes = $agents_for_ssl_clean.filter |$n| { $n.vars['puppet_bin'] == $path }
      $memo + run_command("${path} ssl clean ${localca}", $nodes, '_catch_errors' => true).results
    }
    $ssl_clean_result = ResultSet.new($ssl_clean_results_array)
    if !($ssl_clean_result.error_set.empty) {
      log::error($ssl_clean_result.error_set)
    }
    $agents_for_submit_request = $ssl_clean_result.ok_set.targets

    # Submit the CSR on each node
    enterprise_tasks::message('agent_cert_regen', "Generating CSR and submitting on ${agents_for_submit_request.length} node(s)")
    $submit_request_results_array = $puppet_bin_paths.reduce([]) |$memo, $path| {
      $nodes = $agents_for_submit_request.filter |$n| { $n.vars['puppet_bin'] == $path }
      $memo + run_command("${path} ssl submit_request", $nodes, '_catch_errors' => true).results
    }
    $submit_request_result = ResultSet.new($submit_request_results_array)
    if !($submit_request_result.error_set.empty) {
      log::error($submit_request_result.error_set)
    }
    $agents_for_ca_sign = $submit_request_result.ok_set.targets

    # This signs each individual cert one-by-one, since we don't have a bulk signing option yet.
    enterprise_tasks::message('agent_cert_regen', "Signing certs for ${agents_for_ca_sign.length} node(s)")
    $puppetserver_ca_sign_result = run_task($puppetserver_ca_sign_task, $primary,
      certname => $agents_for_ca_sign.map |$a| { $a.host },
      allow_subject_alt_names => $allow_subject_alt_names,
      dns_alt_names => $dns_alt_names,
      '_catch_errors' => true,
    )
    if !($puppetserver_ca_sign_result.error_set.empty) {
      log::error($puppetserver_ca_sign_result.error_set)
    }
    if !($puppetserver_ca_sign_result.ok) {
      # If this is a handled error, the result should have the 'errored_certnames' key. If it's an unexpected exception,
      # it won't, so assume signing all nodes failed.
      $puppetserver_ca_sign_errored_nodes = $puppetserver_ca_sign_result.first.error.details['errored_certnames'] ? {
        undef => $agents_for_ca_sign,
        default => $puppetserver_ca_sign_result.first.error.details['errored_certnames']
      }
    } else {
      $puppetserver_ca_sign_errored_nodes = []
    }
    $agents_for_download_cert = $agents_for_ca_sign - get_targets($puppetserver_ca_sign_errored_nodes)

    # Download new cert to each node
    enterprise_tasks::message('agent_cert_regen', "Downloading certs on ${agents_for_download_cert.length} node(s)")
    $download_cert_results_array = $puppet_bin_paths.reduce([]) |$memo, $path| {
      $nodes = $agents_for_download_cert.filter |$n| { $n.vars['puppet_bin'] == $path }
      $memo + run_command("${path} ssl download_cert", $nodes, '_catch_errors' => true).results
    }
    $download_cert_result = ResultSet.new($download_cert_results_array)
    if !($download_cert_result.error_set.empty) {
      log::error($download_cert_result.error_set)
    }
    $agents_for_verify = $download_cert_result.ok_set.targets

    # Verify the cert is valid
    enterprise_tasks::message('agent_cert_regen', "Verifying certs on ${agents_for_verify.length} node(s)")
    $verify_cert_results_array = $puppet_bin_paths.reduce([]) |$memo, $path| {
      $nodes = $agents_for_verify.filter |$n| { $n.vars['puppet_bin'] == $path }
      $memo + run_command("${path} ssl verify", $nodes, '_catch_errors' => true).results
    }
    $verify_cert_result = ResultSet.new($verify_cert_results_array)
    if !($verify_cert_result.error_set.empty) {
      log::error($verify_cert_result.error_set)
    }
    $agents_for_pxp_restart = $verify_cert_result.ok_set.targets

    # The pxp-agent service must be restarted in order for it to load the new cert.
    if $manage_pxp_service {
      enterprise_tasks::message('agent_cert_regen', "Restarting pxp-agent on ${agents_for_pxp_restart.length} node(s)")
      $restart_pxp_agent_result = run_task(service, $agents_for_pxp_restart,
        action => 'restart',
        name => 'pxp-agent',
        _catch_errors => true,
      )
      if !($restart_pxp_agent_result.error_set.empty) {
        log::error($restart_pxp_agent_result.error_set)
      }
      wait_until_available($agents_for_pxp_restart,
        wait_time => 30,
        '_catch_errors' => true
      )
    } else {
      $restart_pxp_agent_result = ResultSet.new
    }

    # Restarting the puppet service will allow the service to load the new cert
    enterprise_tasks::message('agent_cert_regen', "Restarting puppet service on ${puppet_stopped_agents.length} node(s) that had it running before the plan was run")
    $restart_puppet_service_result = run_task(service, $puppet_stopped_agents,
      action => 'start',
      name => 'puppet',
      _catch_errors => true,
    )
    if !($restart_puppet_service_result.error_set.empty) {
      log::error($restart_puppet_service_result.error_set)
    }

    # Print all the errors
    $critical_error = !($ssl_clean_result.error_set.empty) or
    !($submit_request_result.error_set.empty) or
    !($puppetserver_ca_clean_result.ok) or
    !($puppetserver_ca_sign_result.ok) or
    !($download_cert_result.error_set.empty) or
    !($verify_cert_result.error_set.empty) or
    !($restart_pxp_agent_result.error_set.empty)

    $any_error = $critical_error or
    !($test_connection_result.error_set.empty) or
    !($get_puppet_bin_path_result.error_set.empty) or
    !($puppet_service_result.error_set.empty) or
    !($pe_services_stop_result.error_set.empty) or
    !($dns_alt_names_result.error_set.empty) or
    !($nodes_with_dns_alt_names_set.empty) or
    !($backup_result.error_set.empty) or
    !($set_csr_attributes_result.error_set.empty) or
    !($remove_cached_catalog_result.error_set.empty) or
    !($set_dns_alt_names_result.error_set.empty) or
    !($restart_puppet_service_result.error_set.empty)

    if $any_error {
      enterprise_tasks::message('agent_cert_regen', '***** Errors during plan execution *****')
    }
    if !($test_connection_result.error_set.empty) {
      enterprise_tasks::message('agent_cert_regen', 'The following nodes encountered an error testing the connection to the nodes. Ensure pxp-agent is running and connected, or the node can be reached via SSH/WinRM if using the --use-ssh or --use-winrm flags.')
      enterprise_tasks::message('agent_cert_regen', "  ${test_connection_result.error_set.targets.join("\n                    ")}")
    }
    if !($get_puppet_bin_path_result.error_set.empty) {
      enterprise_tasks::message('agent_cert_regen', 'The following nodes encountered an error attempting to find the path to the puppet binary. No changes were made to these nodes.')
      enterprise_tasks::message('agent_cert_regen', "  ${get_puppet_bin_path_result.error_set.targets.join("\n                    ")}")
    }
    if !($puppet_service_result.error_set.empty) {
      enterprise_tasks::message('agent_cert_regen', 'The following nodes encountered an error reading the state of the puppet service. No changes were made to these nodes.')
      enterprise_tasks::message('agent_cert_regen', "  ${puppet_service_result.error_set.targets.join("\n                    ")}")
    }
    if !($pe_services_stop_result.error_set.empty) {
      enterprise_tasks::message('agent_cert_regen', 'The following nodes encountered an error stopping the puppet service. Check the state of the service on these nodes. No other changes were made to these nodes.')
      enterprise_tasks::message('agent_cert_regen', "  ${pe_services_stop_result.error_set.targets.join("\n                    ")}")
    }
    if !($dns_alt_names_result.error_set.empty) {
      enterprise_tasks::message('agent_cert_regen', 'The following nodes encountered an error checking if the dns_alt_names puppet setting was already set. No changes were made to certificates on these nodes.')
      enterprise_tasks::message('agent_cert_regen', "  ${dns_alt_names_result.error_set.targets.join("\n                    ")}")
    }
    if !($nodes_with_dns_alt_names_set.empty) {
      # Repeating above since we want to expose all nodes that had errors at the end
      enterprise_tasks::message('agent_cert_regen', 'On the following nodes, dns_alt_names is set in puppet.conf, but the plan was run without a value for the dns_alt_names parameter. Either remove this setting from puppet.conf on these nodes, or run this plan with the dns_alt_names parameter matching the value on these nodes.')
      enterprise_tasks::message('agent_cert_regen', "  ${dns_alt_names_to_print.join("\n                    ")}")
    }
    if !($backup_result.error_set.empty) {
      enterprise_tasks::message('agent_cert_regen', 'The following nodes encountered an error backing up existing certificate artifacts. No changes to the certificate or keys on these nodes have been made.')
      enterprise_tasks::message('agent_cert_regen', "  ${backup_result.error_set.targets.join("\n                    ")}")
    }
    if !($set_csr_attributes_result.error_set.empty) {
      enterprise_tasks::message('agent_cert_regen', 'The following nodes encountered an error setting values in csr_attributes.yaml. No changes to the certificate or keys on these nodes have been made.')
      enterprise_tasks::message('agent_cert_regen', "  ${set_csr_attributes_result.error_set.targets.join("\n                    ")}")
    }
    if !($remove_cached_catalog_result.error_set.empty) {
      enterprise_tasks::message('agent_cert_regen', 'The following nodes encountered an error removing the cached catalog. See the error logged during the plan run more details.')
      enterprise_tasks::message('agent_cert_regen', "  ${remove_cached_catalog_result.error_set.targets.join("\n                    ")}")
    }
    if !($set_dns_alt_names_result.error_set.empty) {
      enterprise_tasks::message('agent_cert_regen', 'The following nodes encountered an error when setting the dns_alt_names puppet setting. See the error logged during the plan run for more details.')
      enterprise_tasks::message('agent_cert_regen', "  ${set_dns_alt_names_result.error_set.targets.join("\n                    ")}")
    }
    if !($puppetserver_ca_clean_result.ok) {
      enterprise_tasks::message('agent_cert_regen', 'The existing certificate for the following nodes could not be cleaned from the primary. See the error logged during the plan run for more details.')
      enterprise_tasks::message('agent_cert_regen', "  ${puppetserver_ca_clean_errored_nodes.join("\n                    ")}")
    }
    if !($ssl_clean_result.error_set.empty) {
      enterprise_tasks::message('agent_cert_regen', 'The following nodes encountered an error when cleaning the existing certificate artifacts from the node. See the error logged during the plan run more details.')
      enterprise_tasks::message('agent_cert_regen', "  ${ssl_clean_result.error_set.targets.join("\n                    ")}")
    }
    if !($submit_request_result.error_set.empty) {
      enterprise_tasks::message('agent_cert_regen', 'The following nodes encountered an error when generating and submitting a certificate signing request. See the error logged during the plan run more details.')
      enterprise_tasks::message('agent_cert_regen', "  ${submit_request_result.error_set.targets.join("\n                    ")}")
    }
    if !($puppetserver_ca_sign_result.ok) {
      enterprise_tasks::message('agent_cert_regen', 'Could not sign the certificate signing request for the following nodes. See the error logged during the plan run for more details.')
      enterprise_tasks::message('agent_cert_regen', "  ${puppetserver_ca_sign_errored_nodes.join("\n                    ")}")
    }
    if !($download_cert_result.error_set.empty) {
      enterprise_tasks::message('agent_cert_regen', 'The following nodes encountered an error when attempting to download the signed certificate. See the error logged during the plan run for more details.')
      enterprise_tasks::message('agent_cert_regen', "  ${download_cert_result.error_set.targets.join("\n                    ")}")
    }
    if !($verify_cert_result.error_set.empty) {
      enterprise_tasks::message('agent_cert_regen', 'The following nodes encountered an error when attempting to verify the new certificate. See the error logged during the plan run for more details.')
      enterprise_tasks::message('agent_cert_regen', "  ${verify_cert_result.error_set.targets.join("\n                    ")}")
    }
    if !($restart_pxp_agent_result.error_set.empty) {
      enterprise_tasks::message('agent_cert_regen', 'The following nodes encountered an error when attempting to restart the pxp-agent service. See the error logged during the plan run for more details, and restart the service manually on these nodes.')
      enterprise_tasks::message('agent_cert_regen', "  ${restart_pxp_agent_result.error_set.targets.join("\n                    ")}")
    }
    if !($restart_puppet_service_result.error_set.empty) {
      enterprise_tasks::message('agent_cert_regen', 'The following nodes encountered an error when attempting to restart the puppet service. See the error logged during the plan run for more details, and restart the service manually on these nodes.')
      enterprise_tasks::message('agent_cert_regen', "  ${restart_puppet_service_result.error_set.targets.join("\n                    ")}")
    }
    if $critical_error {
      enterprise_tasks::message('agent_cert_regen', 'Because an error was encountered on one or more nodes during the process of regenerating the certificate, the primary may no longer be able to communicate with the node(s). This may result in additional errors shown at the end of this plan run.')
    }
    $any_error
  }

  $disable_result = $disable_enable_result['disable_result']
  $enable_result = $disable_enable_result['enable_result']
  $any_error_during_block = $disable_enable_result['block_result']
  if !($disable_result.error_set.empty) {
    enterprise_tasks::message('agent_cert_regen', 'The following nodes encountered an error when attempting to disable the agent in preparation of regenerating the certificate. No changes have been made to the certificate on these nodes. Ensure the agent is enabled by running "puppet agent --enable" on these nodes.')
    enterprise_tasks::message('agent_cert_regen', "  ${disable_result.error_set.targets.join("\n                    ")}")
  }
  if !($enable_result.error_set.empty) {
    enterprise_tasks::message('agent_cert_regen', 'The following nodes encountered an error when attempting to re-enable the agent. This may be due to a previous error when regenerating the certificate. The puppet agent is likely still disabled on these nodes. Run "puppet agent --enable" to re-enable the agent.')
    enterprise_tasks::message('agent_cert_regen', "  ${enable_result.error_set.targets.join("\n                    ")}")
  }
  if (
    !($disable_result.error_set.empty) or
    !($enable_result.error_set.empty) or
    $any_error_during_block
  ) {
    fail_plan('One or more errors occurred during plan execution, but some nodes may have successfully regenerated certificates. Check log for details.')
  }
}
